<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\CodeLine;
use Gek\PhpLang\Contracts\IToIndentedString;


class CodeLineCollection extends TypedListWrapper implements IToIndentedString, \Serializable
{

    #region ctor

    /**
     * IntList constructor.
     * @param CodeLine ...$items
     */
    public function __construct(CodeLine ...$items)
    {
        parent::__construct(new ArrayList(), CodeLine::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * @param CodeLine $item
     */
    public function add(CodeLine $item): void
    {
        $this->innerList->add($item);
    }

    /**
     * @param CodeLine ...$items
     */
    public function addRange(CodeLine ...$items): void
    {
        $this->innerList->addRange($items);
    }


    /**
     * @param CodeLine $item
     * @return bool
     */
    public function contains(CodeLine $item): bool
    {
        return $this->innerList->contains($item);
    }

    /**
     * @param CodeLine $item
     * @return bool
     */
    public function remove(CodeLine $item): bool
    {
        return $this->innerList->remove($item);
    }

    /**
     * @param CodeLine $item
     * @return int
     */
    public function indexOf(CodeLine $item): int
    {
        return $this->innerList->indexOf($item);
    }

    /**
     * @param int $index
     * @param CodeLine $item
     */
    public function insert(int $index, CodeLine $item): void
    {
        $this->innerList->insert($index, $item);
    }

    /**
     * @param int $index
     * @return CodeLine
     */
    public function getAt(int $index): CodeLine
    {
        return $this->innerList[$index];
    }

    /**
     * @param int $index
     * @param CodeLine $item
     */
    public function setAt(int $index, CodeLine $item): void
    {
        $this->innerList[$index] = $item;
    }

    /**
     * @param int $index
     * @param CodeLine ...$items
     */
    public function insertRange(int $index, CodeLine ...$items): void
    {
        $this->innerList->insertRange($index, $items);
    }


    /**
     * @param CodeLine|string $code
     * @param int $indentLevel
     */
    public function addCodeLine($code, int $indentLevel = 0)
    {
        if ($code instanceof CodeLine) {
            $code->setIndentLevel($indentLevel);
            $this->add($code);
        } else {
            $this->add(new CodeLine(trim($code), $indentLevel));
        }
    }

    /**
     * @param string $codes
     */
    public function addCodes(string $codes)
    {
        $paramArr = CodeLine::parseCodes($codes);
        $this->addRange(...$paramArr);
    }


    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion methods

    #region IToIndentedString

    /**
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $lines = $this->select(function (CodeLine $item)use($indentLevel,$indentChars){
            return $item->toIndentedString($indentLevel,$indentChars);
        })->toArray();
        return implode(PHP_EOL, $lines) . PHP_EOL;
    }

    #endregion IToIndentedString

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
       $data = $this->toArray();
       return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data =  unserialize($serialized);
        parent::__construct(new ArrayList(), CodeLine::class);
        $this->addRange(...$data);
    }

    #endregion Serializable

}
