<?php


namespace Gek\PhpLang\Tests\DocComments;


use Gek\PhpLang\DocComments\DocComment;
use Gek\PhpLang\DocComments\DocCommentTag;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class DocCommentTagTest extends TestCase
{

    /**
     * @param $name
     * @return \ReflectionMethod
     * @throws \ReflectionException
     */
    protected static function getMethod($name) {
        $class = new ReflectionClass(DocCommentTag::class);
        $method = $class->getMethod($name);
        $method->setAccessible(true);
        return $method;
    }

    public function testFixKey(){
        $tag = new DocCommentTag('tag');
        $fixKeyFn = self::getMethod('fixKey');
        $res = $fixKeyFn->invokeArgs($tag,['@var']);
        $this->assertEquals('var',$res, 'keydeki "@" iaşreti temizlenmeli.');

        $res = $fixKeyFn->invokeArgs($tag,["var \n \n"]);
        $this->assertEquals('var',$res, 'keydeki "\n" iaşreti temizlenmeli.');

        $res = $fixKeyFn->invokeArgs($tag,["var \t \t"]);
        $this->assertEquals('var',$res, 'keydeki "\t" iaşreti temizlenmeli.');

        $res = $fixKeyFn->invokeArgs($tag,["var \r \r"]);
        $this->assertEquals('var',$res, 'keydeki "\r" iaşreti temizlenmeli.');

        $res = $fixKeyFn->invokeArgs($tag,["property get"]);
        $this->assertEquals('property-get',$res, 'keydeki boşluklar "-" ile değiştirilmeli.');

        $res = $fixKeyFn->invokeArgs($tag,["property  \t  \n  \r  
         get"]);
        $this->assertEquals('property-get',$res, 'keyde arada bulunan çooklu boşluklar yek boşluğa indirilmeli.');

        $res = $fixKeyFn->invokeArgs($tag,["property_get"]);
        $this->assertEquals('property-get',$res, 'keydeki "_" işareti "-" ile değiştirilmeli.');

        $res = $fixKeyFn->invokeArgs($tag,["pRoPErTy-gET"]);
        $this->assertEquals('property-get',$res, 'keydeki bütün harfler küçük olmalı.');


    }

    public function testConstructor()
    {
        $tag = new DocCommentTag('test');

        $this->assertEquals(
            'test',
            $tag->getKey()
        );

        $this->assertEquals(
            '',
            $tag->getValue()
        );
        $tag = new DocCommentTag('key', 'val');

        $this->assertEquals(
            'key',
            $tag->getKey()
        );

        $this->assertEquals(
            'val',
            $tag->getValue()
        );

        $this->expectException(\TypeError::class);
        $tag = new DocCommentTag(array('key'));
    }

    public function testGetKey(){
        $tag = new DocCommentTag('deneme');
        $this->assertEquals(
            'deneme',
            $tag->getKey()
        );
        $this->assertEquals(
            'deneme',
            $tag->key
        );
        $tag = new DocCommentTag('@deneme');
        $this->assertEquals(
            'deneme',
            $tag->getKey()
        );

        $this->assertEquals(
            'deneme',
            $tag->key
        );

        $tag = new DocCommentTag('  @dEnEMe ');
        $this->assertEquals(
            'deneme',
            $tag->getKey()
        );
        $this->assertEquals(
            'deneme',
            $tag->key
        );
    }

    public function testSetKey(){
        $tag = new DocCommentTag('var');
        $tag->setKey('prop');
        $this->assertEquals('prop', $tag->getKey());

        $tag->key ='propa';
        $this->assertEquals('propa', $tag->getKey());

        $tag->setKey("    \t 
        @prop \n ");
        $this->assertEquals('prop', $tag->getKey());

        $tag->key = "    \t 
        @propa \n ";
        $this->assertEquals('propa', $tag->getKey());

        $tag->setKey("    \t 
        @pRoP  
        tAg \n ");
        $this->assertEquals('prop-tag', $tag->getKey());

        $tag->key = "    \t 
        @pRoPA  
        tAg \n ";
        $this->assertEquals('propa-tag', $tag->getKey());

        $this->expectException(\TypeError::class);
        $tag->key = array('key');
    }

    public function testGetValue(){
        $tag = new DocCommentTag('var', 'string');
        $this->assertEquals(
            'string',
            $tag->getValue()
        );
        $tag = new DocCommentTag('var', 'int');
        $this->assertEquals(
            'int',
            $tag->value
        );
    }

    public function testSetValue(){
        $tag = new DocCommentTag('var', 'string');
        $tag->setValue('int');
        $this->assertEquals(
            'int',
            $tag->getValue()
        );
        $tag->value = 'float';
        $this->assertEquals(
            'float',
            $tag->getValue()
        );
    }

    public function testToString(){
        $tag = new DocCommentTag('var', 'int');
        $this->assertEquals(
            '@var int',
            $tag->__toString()
        );
        $this->assertEquals(
            '@var int',
            $tag . ''
        );
        $this->assertEquals(
            '@var int',
            strval($tag)
        );

        $tag = new DocCommentTag('enum');
        $this->assertEquals(
            '@enum',
            $tag->__toString()
        );
        $tag = new DocCommentTag('enum',"  ");
        $this->assertEquals(
            '@enum',
            $tag->__toString()
        );



    }

}
