<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\TraitItem;
use Gek\PhpLang\UseItem;

class TraitItemCollection extends TypedListWrapper implements IUseCreator, IToIndentedString, \Serializable
{

    #region ctor

    /**
     * IntList constructor.
     * @param TraitItem ...$items
     */
    public function __construct(TraitItem ...$items)
    {
        parent::__construct(new ArrayList(), TraitItem::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * @param TraitItem $item
     */
    public function add(TraitItem $item):void{
        $this->innerList->add($item);
    }

    /**
     * @param TraitItem ...$items
     */
    public function addRange(TraitItem ...$items):void{
        $this->innerList->addRange($items);
    }



    /**
     * @param TraitItem $item
     * @return bool
     */
    public function contains(TraitItem $item):bool{
        return $this->innerList->contains($item);
    }

    /**
     * @param TraitItem  $item
     * @return bool
     */
    public function remove(TraitItem $item):bool{
        return $this->innerList->remove($item);
    }

    /**
     * @param TraitItem $item
     * @return int
     */
    public function indexOf(TraitItem $item): int{
        return $this->innerList->indexOf($item);
    }

    /**
     * @param int $index
     * @param TraitItem $item
     */
    public function insert(int $index, TraitItem $item): void{
        $this->innerList->insert($index,$item);
    }

    /**
     * @param int $index
     * @return TraitItem
     */
    public function getAt(int $index):TraitItem {
        return $this->innerList[$index];
    }

    /**
     * @param int $index
     * @param TraitItem $item
     */
    public function setAt(int $index, TraitItem $item):void{
        $this->innerList[$index] = $item;
    }

    /**
     * @param int $index
     * @param TraitItem ...$items
     */
    public function insertRange(int $index, TraitItem ...$items):void{
        $this->innerList->insertRange($index,$items);
    }


    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion methods

    #region IUseCreator

    /**
     * @return array|UseItem[]
     */
    public function getUseArray(): array
    {
        return $this->where(function (TraitItem $item){
            return $item->getType()->isUseItem();
        })->aggregate(function (&$arr, TraitItem $item){
            $arr[] = $item->getType()->getUseItem();
        },array());
    }

    #endregion IUseCreator


    #region IToIndentedString

    /**
     * @param int $indentLevel girinti seviyesi
     * @param string $indentChars girinti karakterleri
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        return $this->aggregate(function (&$res,TraitItem $traitItem) use($indentLevel,$indentChars){
            $res .= $traitItem->toIndentedString($indentLevel,$indentChars) . PHP_EOL;
        },'');
    }

    #endregion IToIndentedString


    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->toArray();
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data =  unserialize($serialized);
        parent::__construct(new ArrayList(), TraitItem::class);
        $this->addRange(...$data);
    }

    #endregion Serializable

}
