<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\ClassMember;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\UseItem;

class ClassMemberCollection extends TypedListWrapper implements IUseCreator, \Serializable
{

    #region ctor

    /**
     * ClassMemberCollection constructor.
     * @param ClassMember ...$items
     */
    public function __construct(ClassMember ...$items)
    {
        parent::__construct(new ArrayList(), ClassMember::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * @param ClassMember $item
     */
    public function add(ClassMember $item): void
    {
        $this->innerList->add($item);
    }

    /**
     * @param ClassMember ...$items
     */
    public function addRange(ClassMember ...$items): void
    {
        $this->innerList->addRange($items);
    }


    /**
     * @param ClassMember $item
     * @return bool
     */
    public function contains(ClassMember $item): bool
    {
        return $this->innerList->contains($item);
    }

    /**
     * @param ClassMember $item
     * @return bool
     */
    public function remove(ClassMember $item): bool
    {
        return $this->innerList->remove($item);
    }

    /**
     * @param ClassMember $item
     * @return int
     */
    public function indexOf(ClassMember $item): int
    {
        return $this->innerList->indexOf($item);
    }

    /**
     * @param int $index
     * @param ClassMember $item
     */
    public function insert(int $index, ClassMember $item): void
    {
        $this->innerList->insert($index, $item);
    }

    /**
     * @param int $index
     * @return ClassMember
     */
    public function getAt(int $index): ClassMember
    {
        return $this->innerList[$index];
    }

    /**
     * @param int $index
     * @param ClassMember $item
     */
    public function setAt(int $index, ClassMember $item): void
    {
        $this->innerList[$index] = $item;
    }

    /**
     * @param int $index
     * @param ClassMember ...$items
     */
    public function insertRange(int $index, ClassMember ...$items): void
    {
        $this->innerList->insertRange($index, $items);
    }

    #endregion methods

    #region IUseCreator

    /**
     * @return array|UseItem[]
     */
    public function getUseArray(): array
    {
        return $this->where(function (ClassMember $item) {
            return ($item instanceof IUseCreator);
        })->aggregate(function (&$arr, IUseCreator $item) {
            $arr = array_merge($arr, $item->getUseArray());
        }, array());
    }

    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->toArray();
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data =  unserialize($serialized);
        parent::__construct(new ArrayList(), ClassMember::class);
        $this->addRange(...$data);
    }

    #endregion Serializable
}
