<?php


namespace Gek\PhpLang;


use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Traits\ValueAware;
use Gek\Infrastructure\Str;

class ClassConst extends ClassMember implements IToIndentedString, \Serializable
{
    use ValueAware;



    #region ctor

    /**
     * ClassConst constructor.
     * @param string $name
     * @param $value
     * @param int $valueRenderType
     * @throws \ReflectionException
     */
    public function __construct(string $name, $value, int $valueRenderType = LiteralTypes::NONE)
    {
        parent::__construct($name, PhpVisibility::PUBLIC());
        $this->setValue($value)
            ->setValueRenderType($valueRenderType);

    }

    #endregion ctor


    #region Methods

    /**
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion Methods

    #region IToIndentedString

    /**
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0,string $indentChars = '    '): string
    {
        $strRes = str_repeat($indentChars,$indentLevel);
        $strRes .= $this->visibility->getValue() . " ";

        $strRes .= 'const ' . $this->getName() . " ";
        if (isset($this->defaultValue)) {
            $strRes .= "= ";
            $defVal = $this->value;
            $defVal = strval($defVal);

            switch ($this->valueRenderType) {
                case LiteralTypes::SINGLE_QUOTES:
                    $defVal = Str::format("'{0}'", $defVal);
                    break;
                case LiteralTypes::DOUBLE_QUOTES:
                    $defVal = Str::format('"{0}"', $defVal);
            }
            $strRes .= $defVal;
        }
        $strRes .= ";";
        return $strRes;
    }

    #endregion IToIndentedString


    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->getDataForSerialize();
        $data['vl'] = $this->value;
        $data['rt'] = $this->valueRenderType;
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->setDataForSerialize($data);
        $this->value = $data['vl'];
        $this->valueRenderType = $data['rt'];
    }

    #endregion Serializable
}