<?php


namespace Gek\PhpLang;



use Gek\Infrastructure\Str;

class PhpType implements \Serializable
{
    #region field

    protected string $name;

    /**
     * @var TypeUsage
     */
    protected TypeUsage $usage;

    protected bool $nullable = false;

    protected bool $isClassType = false;

    protected ?UseItem $useItem = null;

    #endregion field

    #region ctor

    public function __construct(string $type, bool $nullable = false, bool $isClassType = false)
    {
        $this->name = $type;
        $this->nullable = $nullable;
        $this->isClassType = $isClassType;
        $this->setUsage(TypeUsage::FIELD_USE());
    }

    #endregion ctor

    #region properties



    /**
     * @return string
     */
    public function getName():string {
        return $this->name;
    }

    /**
     * @param string $name
     * @return self
     */
    public function setName(string $name):self {
        $this->name = $name;
        return $this;
    }

    /**
     * @return TypeUsage
     */
    public function getUsage():TypeUsage{
        return $this->usage;
    }

    /**
     * @param TypeUsage|int $usage
     * @return self
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function setUsage($usage):self {
        if(is_int($usage)){
            $usage = new TypeUsage($usage);
        }
        $this->usage = $usage;
        return $this;
    }

    /**
     * @param bool $nullable
     * @return self
     */
    public function setNullable(bool $nullable = true):self {
        $this->nullable = $nullable;
        return $this;
    }

    /**
     * @param bool $isClassType
     * @return self
     */
    public function setClassType(bool $isClassType = true):self {
        $this->isClassType = $isClassType;
        return $this;
    }

    /**
     * @return UseItem|null
     */
    public function getUseItem():?UseItem{
        return $this->useItem;
    }

    /**
     * @param UseItem|null $useItem
     * @return self
     */
    public function setUseItem(?UseItem $useItem):self {
        $this->useItem = $useItem;
        return $this;
    }

    #endregion properties

    #region method

    /**
     * @return bool
     */
    public function isNullable():bool {
        return $this->nullable;
    }

    /**
     * @return bool
     */
    public function isClassType():bool {
        return $this->isClassType;
    }

    public function isUseItem():bool {
        return $this->useItem !== null;
    }

    public static function parseClassType($name):PhpType{
        $name = trim($name);
        $phpType = new  PhpType($name,false,true);
        if(Str::contains($name,'\\')){
            $phpType->setUseItem(new UseItem($name));
        }
        return $phpType;
    }


    #endregion method

    #region utils


    #endregion utils

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            'nm' => $this->name,
            'ug' => $this->getUsage()->toInt(),
            'nl' => $this->nullable,
            'ic' => $this->isClassType
        ];
        if($this->isUseItem()){
            $data['ui'] = $this->useItem;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->name = $data['nm'];
        $this->usage = new TypeUsage($data['ug']);
        $this->nullable =  $data['nl'];
        $this->isClassType = $data['ic'];
        if(isset($data['ui'])){
           $this->useItem =  $data['ui'];
        }else{
            $this->useItem = null;
        }

    }

    #endregion Serializable
}
