<?php


namespace Gek\PhpLang;


use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\Infrastructure\Str;

class CodeLine implements IToIndentedString, \Serializable
{

    #region fields

    protected string $code = '';

    protected int $indentLevel = 0;

    #endregion fields

    #region ctor

    public function __construct(string $code = '',int $indentLevel = 0)
    {
        $this->setCode($code)
            ->setIndentLevel($indentLevel);
    }

    #endregion ctor

    #region properties

    /**
     * @return string
     */
    public function  getCode():string {
        return $this->code;
    }

    /**
     * @param string $code
     * @return CodeLine
     */
    public function setCode(string $code):self {
        $this->code = $code;
        return $this;
    }

    /**
     * @return int
     */
    public function getIndentLevel():int{
        return $this->indentLevel;
    }

    public function setIndentLevel(int $indentLevel):self{
        if($indentLevel < 0){
            $indentLevel = 0;
        }
        $this->indentLevel = $indentLevel;
        return $this;
    }

    #endregion properties

    #region method

    /**
     * @param int $add
     * @return int
     */
    public function incrementIndentLevel(int $add = 1):int {
        $this->setIndentLevel($this->indentLevel + $add);
        return $this->getIndentLevel();
    }

    /**
     * @param int $sub
     * @return int
     */
    public function decrementIndentLevel(int $sub = 1):int {
        $this->setIndentLevel($this->indentLevel - $sub);
        return $this->getIndentLevel();
    }

    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion method

    #region utils

    /**
     * @param string $code
     * @return CodeLine
     */
    public static function parseLine(string $code):CodeLine{
        $indent = '    ';
        $indentCount = strlen($indent);
        $indentLevel = 0;
        while (Str::startsWith($code,$indent)){

            $indentLevel++;
            $code = substr($code,$indentCount);
        }
        $codeLine = new CodeLine(trim($code),$indentLevel);
        return $codeLine;
    }

    /**
     * @param string $codes
     * @return array
     */
    public static function parseCodes(string $codes):array {
        $resArr = array();

        $linesArr = explode(PHP_EOL,$codes);
        foreach ($linesArr as $strLine){
            $resArr[] = static::parseLine($strLine);
        }
        return $resArr;
    }

    #endregion utils

    #region IToIndentedString

    /**
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0,string $indentChars = '    '): string
    {
        return str_repeat($indentChars,$this->indentLevel + $indentLevel) . $this->code;
    }

    #endregion IToIndentedString

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = ['c' => $this->code];
        if($this->indentLevel > 0){
            $data['il'] = $this->indentLevel;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->code = $data['c'];
        $this->indentLevel = isset($data['il']) ? $data['il'] : 0;

    }
}
