<?php



namespace Gek\PhpLang;

use Gek\PhpLang\PhpVisibility;
use Gek\PhpLang\UseItem;
use Gek\PhpLang\ClassMember;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\Traits\StaticAware;
use Gek\PhpLang\Traits\AbstractAware;
use Gek\PhpLang\Traits\FinalAware;
use Gek\PhpLang\PhpTypeDeclared;
use Gek\PhpLang\Collections\MethodParamCollection;
use Gek\PhpLang\Collections\CodeLineCollection;
use Gek\PhpLang\MethodParam;
use Gek\PhpLang\CodeLine;

/**
 * Class ClassMethod
 * @package Gek\PhpLang
 */
class ClassMethod extends ClassMember implements IToIndentedString, IUseCreator, \Serializable
{

    use StaticAware;
    use AbstractAware;
    use FinalAware;

    #region fields

    /**
     *
     * @var bool
     */
    protected bool $interfaceMethod = false;

    /**
     *
     * @var PhpTypeDeclared|null
     */
    protected ?PhpTypeDeclared $returnType = null;

    /**
     *
     * @var MethodParamCollection
     */
    protected MethodParamCollection $params;

    /**
     *
     * @var CodeLineCollection
     */
    protected CodeLineCollection $codeLines;

    #endregion fields

    #region ctor

    /**
     *
     * @param string $name
     * @param PhpVisibility|null $visibility
     */
    public function __construct(string $name, ?PhpVisibility $visibility = null){
        parent::__construct($name, $visibility);
        $this->params = new MethodParamCollection();
        $this->codeLines = new CodeLineCollection();
    }

    #endregion ctor

    #region Properties

    /**
     *
     * @param bool $interfaceMethod
     * @return self
     */
    public function setInterfaceMethod(bool $interfaceMethod = true):self {
        $this->interfaceMethod = $interfaceMethod;
        return $this;
    }

    /**
     *
     * @return string|null
     */
    public function getReturnType():?string {
        return $this->returnType !== null ? $this->returnType->getRaw() : null;
    }

    /**
     *
     * @param PhpTypeDeclared|array|string|null $returnType
     * @return self
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function setReturnType($returnType):self {
        if (is_array($returnType)) {
            $temp = array();
            foreach ($returnType as $tp) {
                if ($tp === null) {
                    $tp = 'null';
                }
                if (empty($tp)) {
                    continue;
                }
                $temp[] = $tp;
            }
            $returnType = implode('|', $temp);
        }
        if (is_string($returnType)) {
            $returnType = new PhpTypeDeclared($returnType);
        }
        $this->returnType = $returnType;
        return $this;
    }

    /**
     *
     * @return MethodParamCollection
     */
    public function getParams():MethodParamCollection {
        return $this->params;
    }

    /**
     *
     * @param MethodParamCollection $params
     * @return self
     */
    public function setParams(MethodParamCollection $params):self {
        $this->params = $params;
        return $this;
    }

    /**
     *
     * @return CodeLineCollection
     */
    public function getCodeLines():CodeLineCollection {
        return $this->codeLines;
    }

    /**
     *
     * @param CodeLineCollection $codeLines
     * @return self
     */
    public function setCodeLines(CodeLineCollection $codeLines):self {
        $this->codeLines = $codeLines;
        return $this;
    }

    /**
     *
     * @return string
     */
    public function getBody():string {
        return strval($this->codeLines);
    }

    /**
     *
     * @param string $codes
     * @return self
     */
    public function setBody(string $codes):self {
        $this->codeLines->clear();
        $this->codeLines->addCodes($codes);
        return $this;
    }

    #endregion Properties

    #region Methods

    /**
     *
     * @return bool
     */
    public function isInterfaceMethod():bool {
        return $this->interfaceMethod;
    }

    /**
     *
     * @return bool
     */
    public function hasParams():bool {
        $this->params->any();
    }

    /**
     *
     * @param MethodParam|string $param
     * @param string|null $type
     * @param string|null $val
     * @param int|null $renderType
     * @return self
     * @throws \ReflectionException
     * @throws \Gek\Infrastructure\Exceptions\GekException
     */
    public function addParam($param, ?string $type = null, ?string $val = null, ?int $renderType = null):self {
        $this->params->addParam($param, $type, $val, $renderType);
        return $this;
    }

    /**
     *
     * @param string $params
     * @return self
     * @throws \ReflectionException
     * @throws \Gek\Infrastructure\Exceptions\GekException
     */
    public function addParams(string $params):self {
        $this->params->addParams($params);
        return $this;
    }

    /**
     *
     * @param CodeLine|string $code
     * @param int $indentLevel
     * @return self
     */
    public function addCodeLine($code, int $indentLevel = 0):self {
        $this->codeLines->addCodeLine($code, $indentLevel);
        return $this;
    }

    /**
     *
     * @param string $code
     * @return self
     */
    public function addCodeLines(string $code):self {
        $this->codeLines->addCodes($code);
        return $this;
    }

    /**
     *
     * @return bool
     */
    public function hasBody():bool {
        return $this->codeLines->any();
    }

    /**
     *
     * @return bool
     */
    public function hasReturnType():bool {
        return $this->getReturnType() !== null;
    }

    /**
     *
     * @return self
     */
    public function autoDocComment():self {
        $this->getDocComment()->getTags()->clear();
        if ($this->params->any()) {
            foreach ($this->params as $prm) {
                $this->addCommentTag('param', $prm->renderForComment());
            }
        }
        if ($this->hasReturnType()) {
            $this->addCommentTag('return', $this->returnType->renderForComment());
        }
        return $this;
    }

    public function __toString(){
        return $this->toIndentedString();
    }

    #endregion Methods

    #region IToIndentedString

    /**
     *
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '):string {
        $indent = str_repeat($indentChars, $indentLevel);

        $strRes = '';
        if(!$this->getDocComment()->isEmpty()){
            $strRes .= $this->getDocComment()->toIndentedString($indentLevel,$indentChars);
        }

        $strRes .= $indent . $this->visibility->getValue() . " ";
        if ($this->isAbstract()) {
            $strRes .= 'abstract ';
        } elseif ($this->isFinal()) {
            $strRes .= 'final ';
        }
        if ($this->isStatic()) {
            $strRes .= 'static ';
        }
        $strRes .= 'function '.  $this->getName();
        $strRes .= '(' . $this->params . ")";
        if ($this->hasReturnType()) {
            $retType = $this->returnType->renderForReturn();
            if(!empty($retType)){
                $strRes .= ':' . $retType . " ";
            }
        }

        if ($this->isInterfaceMethod() || $this->isAbstract()) {
            $strRes .= ';';
        } else {
            $strRes .= '{' . PHP_EOL;

            if ($this->hasBody()) {
                /** @var CodeLineCollection $lines */
                $lines = $this->codeLines->select(function (CodeLine $item) {
                    $item->incrementIndentLevel();
                    return $item;
                })->toTypedClass(CodeLineCollection::class);
                $strRes .= $lines->toIndentedString($indentLevel,$indentChars);
            }
            $strRes .= $indent . '}' . PHP_EOL;

        }

        return $strRes;
    }

    #endregion IToIndentedString

    #region IUseCreator

    /**
     *
     * @return array|UseItem[]
     */
    public function getUseArray():array {
        $useArr  = $this->params->getUseArray();
        if($this->hasReturnType()){
            $useArr = array_merge($useArr, $this->returnType->getUseArray());
        }
        return $useArr;
    }

    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->getDataForSerialize();
        if($this->isStatic()){
            $data['stc'] = $this->static;
        }
        if($this->isFinal()){
            $data['fnl'] = $this->final;
        }
        if($this->isAbstract()){
            $data['abs'] = $this->abstract;
        }
        if($this->isInterfaceMethod()){
            $data['imtd'] = $this->interfaceMethod;
        }
        if($this->returnType !== null){
            $data['rettyp'] = $this->returnType;
        }

        if($this->params->any()){
            $data['prm'] = $this->params;
        }
        if($this->codeLines->any()){
            $data['cln'] = $this->codeLines;
        }

        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->setDataForSerialize($data);
        if(isset($data['stc'])){
            $this->static = $data['stc'];
        }
        if(isset($data['fnl'])){
            $this->final = $data['fnl'];
        }
        if(isset($data['abs'])){
            $this->abstract = $data['abs'];
        }
        if(isset($data['imtd'])){
            $this->interfaceMethod = $data['imtd'];
        }
        $this->returnType = isset($data['rettyp']) ? $data['rettyp'] : null;
        $this->params = isset($data['prm']) ? $data['prm'] : new MethodParamCollection();
        $this->codeLines = isset($data['cln']) ? $data['cln'] : new CodeLineCollection();
    }

    #endregion Serializable
}
