<?php


namespace Gek\PhpLang;


use Gek\Collections\ArrayList;
use Gek\Collections\Enumerable;
use Gek\Collections\Typed\StringDictionary;
use Gek\PhpLang\Collections\ClassMemberCollection;
use Gek\PhpLang\Collections\PhpTypeCollection;
use Gek\PhpLang\Collections\TraitItemCollection;
use Gek\PhpLang\Collections\UseItemCollection;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\DocComments\DocComment;
use Gek\PhpLang\Traits\AbstractAware;
use Gek\PhpLang\Traits\DocCommentAware;
use Gek\PhpLang\Traits\FinalAware;
use Gek\PhpLang\Traits\NameAware;

class PhpClass implements IToIndentedString, \Serializable
{
    use DocCommentAware;
    use AbstractAware;
    use FinalAware;
    use NameAware;

    #region fields

    private ClassTypes $classType;

    private ?string $namespace = null;

    private UseItemCollection $uses;

    /**
     * @var PhpType|null
     */
    private ?PhpType $extends = null;

    private PhpTypeCollection $implements;

    private ClassMemberCollection $members;

    private TraitItemCollection $traits;

    #endregion fields

    #region ctor

    public function __construct(string $name, ?ClassTypes $classType = null)
    {
        $this->setName($name);
        $this->classType = ($classType !== null) ? $classType : ClassTypes::CLASS_();
        $this->uses = new UseItemCollection();
        $this->implements = new PhpTypeCollection();
        $this->members = new ClassMemberCollection();
        $this->traits = new TraitItemCollection();
    }

    #endregion ctor

    #region property

    /**
     * @return ClassTypes
     */
    public function getClassType(): ClassTypes
    {
        return $this->classType;
    }

    /**
     * @param ClassTypes $classType
     * @return self
     */
    public function setClassType(ClassTypes $classType): self
    {
        $this->classType = $classType;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getNamespace(): ?string
    {
        return $this->namespace;
    }

    /**
     * @param string|null $namespace
     * @return self
     */
    public function setNamespace(?string $namespace): self
    {
        $this->namespace = $namespace;
        return $this;
    }

    /**
     * @return PhpType
     */
    public function getExtends(): PhpType
    {
        return $this->extends;
    }

    /**
     * @param PhpType|string|null $extends
     * @return PhpClass
     */
    public function setExtends($extends): self
    {
        if (is_string($extends)) {
            $extends = PhpType::parseClassType($extends);
        }
        $this->extends = $extends;
        return $this;
    }

    /**
     * @return PhpTypeCollection
     */
    public function getImplements(): PhpTypeCollection
    {
        return $this->implements;
    }

    /**
     * @param PhpTypeCollection $implements
     * @return PhpClass
     */
    public function setImplements(PhpTypeCollection $implements): self
    {
        $this->implements = $implements;
        return $this;
    }

    /**
     * @return ClassMemberCollection
     */
    public function getMembers(): ClassMemberCollection
    {
        return $this->members;
    }

    /**
     * @param ClassMemberCollection $members
     * @return PhpClass
     */
    public function setMembers(ClassMemberCollection $members): self
    {
        $this->members = $members;
        return $this;
    }

    /**
     * @return TraitItemCollection
     */
    public function getTraits(): TraitItemCollection
    {
        return $this->traits;
    }

    /**
     * @param TraitItemCollection $traits
     * @return $this
     */
    public function setTraits(TraitItemCollection $traits): self
    {
        $this->traits = $traits;
        return $this;
    }

    /**
     * @return UseItemCollection
     */
    public function getUses(): UseItemCollection
    {
        return $this->uses;
    }

    /**
     * @param UseItemCollection $uses
     * @return $this
     */
    public function setUses(UseItemCollection $uses): self
    {
        $this->uses = $uses;
        return $this;
    }

    #endregion property

    #region Methods

    /**
     * @param $use
     * @param string|null $aliasName
     * @return $this
     */
    public function addUse($use, ?string $aliasName = null): self
    {
        if (is_string($use)) {
            $use = new UseItem($use);
        }
        $use->setAliasName($aliasName);
        $this->uses->add($use);
        return $this;
    }

    /**
     * @return string
     */
    public function getFullName(): string
    {
        if ($this->getNamespace() != null) {
            return $this->getNamespace() . "\\" . $this->getName();
        }
        return "\\" . $this->getName();
    }

    /**
     * @return bool
     */
    public function hasExtends(): bool
    {
        return $this->extends !== null;
    }

    /**
     * @return bool
     */
    public function hasImplements(): bool
    {
        return $this->implements->any();
    }

    /**
     * @param PhpType|string $implement
     * @return self
     */
    public function addImplement($implement): self
    {
        if (is_string($implement)) {
            $implement = phpType::parseClassType($implement);
        }
        $this->implements->add($implement);
        return $this;
    }

    /**
     * @param ClassMember $member
     * @return $this
     */
    public function addMember(ClassMember $member): self
    {
        $this->members->add($member);
        return $this;
    }

    /**
     * @param string $name
     * @param PhpTypeDeclared|array|string|null $type
     * @return ClassField
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function addField(string $name, $type = null): ClassField
    {
        $field = new ClassField($name, $type);
        $this->addMember($field);
        return $field;
    }

    /**
     * @param string $name
     * @return ClassMethod
     */
    public function addMethod(string $name): ClassMethod
    {
        $method = new ClassMethod($name);
        $this->members->add($method);
        return $method;
    }

    /**
     * @param string|true|null $region
     * @return ClassMethod
     */
    public function addConstructor($region = null): ClassMethod
    {
        if ($region === true) {
            $region = 'ctor';
        }

        return $this->addMethod('__construct')
            ->setRegion($region);
    }

    /**
     * @param TraitItem|string $trait
     * @param Enumerable|StringDictionary|array<string,string>|null $aliases
     * @return $this
     */
    public function addTrait($trait, $aliases = null): self
    {
        if (is_string($trait)) {
            $trait = new TraitItem($trait, $aliases);
        }
        $this->traits->add($trait);
        return $this;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }


    #endregion Methods

    #region IToIndentedString

    /**
     * @param int $indentLevel girinti seviyesi
     * @param string $indentChars girinti karakterleri
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $this->syncUses();
        $indent = str_repeat($indentChars, $indentLevel);
        $res = $indent;
        if ($this->getNamespace() !== null) {
            $res .= 'namespace ' . $this->getNamespace() . ';' . PHP_EOL;
            $res .= PHP_EOL;
        }

        if ($this->uses->any()) {
            $res .= $this->uses->toIndentedString($indentLevel, $indentChars);
            $res .= PHP_EOL;
        }

        if (!$this->getDocComment()->isEmpty()) {
            $res .= $this->getDocComment()->toIndentedString($indentLevel, $indentChars);
        }

        $res .= $indent;

        if ($this->isFinal()) {
            $res .= 'final ';
        } elseif ($this->isAbstract()) {
            $res .= 'abstract';
        }
        $res .= $this->getClassType()->getValue() . ' ';
        $res .= $this->getName() . ' ';

        if ($this->hasExtends()) {
            $extendsType = $this->extends->getName();
            if ($this->extends->isUseItem()) {
                $extendsType = $this->extends->getUseItem()->getName();
                if ($this->extends->getUseItem()->isAliasName()) {
                    $extendsType = $this->extends->getUseItem()->getAliasName();
                }
            }
            $res .= 'extends ' . $extendsType . ' ';
        }
        if ($this->implements->any()) {
            $res .= 'implements ';
            $implements = $this->implements->select(function (PhpType $item) {
                $typeName = $item->getName();
                if ($item->isUseItem()) {
                    $typeName = $item->getUseItem()->getName();
                    if ($item->getUseItem()->isAliasName()) {
                        $typeName = $item->getUseItem()->getAliasName();
                    }
                }
                return $typeName;
            })->toArray();
            $res .= implode(', ', $implements) . ' ';
        }
        $res .= PHP_EOL . $indent . '{' . PHP_EOL . PHP_EOL;
        $indentLevel += 1;

        if ($this->traits->any()) {
            $res .= $this->traits->toIndentedString($indentLevel, $indentChars);
            $res .= PHP_EOL;
        }

        if ($this->members->any()) {
            $sections = $this->members->select(function (ClassMember $member) {
                return $member->getRegion();
            })->distinct()->toArray();

            foreach ($sections as $section) {
                if ($section !== null) {
                    $res .= str_repeat($indentChars, $indentLevel) . '#region ' . $section . PHP_EOL . PHP_EOL;
                }

                $res .= $this->members->where(function (ClassMember $member) use ($section) {
                    return $member->getRegion() === $section;
                })->aggregate(function (&$mbr, IToIndentedString $member) use ($indentLevel, $indentChars) {
                    $mbr .= $member->toIndentedString($indentLevel, $indentChars) . PHP_EOL;
                }, '');

                if ($section !== null) {
                    $res .= str_repeat($indentChars, $indentLevel) . '#endregion ' . $section . PHP_EOL . PHP_EOL;
                }
            }
        }

        $res .= $indent . '}' . PHP_EOL;
        return $res;

    }

    #endregion IToIndentedString

    #region utils

    private function syncUses()
    {
        $useItemArr = new ArrayList();
        if ($this->hasExtends()) {
            if ($this->getExtends()->isUseItem()) {
                $useItemArr->add($this->getExtends()->getUseItem());
            }
        }
        if ($this->hasImplements()) {
            $useItemArr->addRange($this->getImplements()->getUseArray());
        }
        if ($this->traits->any()) {
            $useItemArr->addRange($this->traits->getUseArray());
        }
        if ($this->getMembers()->any()) {
            $membersUses = $this->getMembers()
                ->where(function ($item) {
                    return ($item instanceof IUseCreator);
                })->aggregate(function (&$res, IUseCreator $itm) {
                    $res = array_merge($res, $itm->getUseArray());
                }, array());
            $useItemArr->addRange($membersUses);
        }
        $this->uses->syncUseItems($useItemArr->toArray());
    }

    #endregion utils

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            'nm' => $this->name,
        ];
        if ($this->isAbstract()) {
            $data['abs'] = $this->abstract;
        }
        if ($this->isFinal()) {
            $data['fnl'] = $this->final;
        }

        if (!$this->getDocComment()->isEmpty()) {
            $data['dc'] = $this->docComment;
        }

        if ($this->classType->getValue() !== ClassTypes::CLASS_) {
            $data['clst'] = $this->classType->getValue();
        }

        if ($this->namespace !== null) {
            $data['ns'] = $this->namespace;
        }
        if ($this->uses->any()) {
            $data['uss'] = $this->uses;
        }
        if ($this->extends !== null) {
            $data['ext'] = $this->extends;
        }

        if ($this->implements->any()) {
            $data['imp'] = $this->implements;
        }

        if ($this->members->any()) {
            $data['mbrs'] = $this->members;
        }

        if ($this->traits->any()) {
            $data['trts'] = $this->traits;
        }

        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->name = $data['nm'];
        if (isset($data['abs'])) {
            $this->abstract = $data['abs'];
        }
        if (isset($data['fnl'])) {
            $this->final = $data['fnl'];
        }
        $this->docComment = isset($data['dc']) ? $data['dc'] : new DocComment();
        $this->classType = isset($data['clst']) ? new ClassTypes($data['clst']) : ClassTypes::CLASS_();
        $this->namespace = isset($data['ns']) ? $data['ns'] : null;
        $this->uses = isset($data['uss']) ? $data['uss'] : new UseItemCollection();
        $this->extends = isset($data['ext']) ? $data['ext'] : null;
        $this->implements = isset($data['imp']) ? $data['imp'] : new PhpTypeCollection();
        $this->members = isset($data['mbrs']) ? $data['mbrs'] : new ClassMemberCollection();
        $this->traits = isset($data['trts']) ? $data['trts'] : new TraitItemCollection();
    }

    #endregion Serializable

}
