<?php


namespace Gek\PhpLang;


use Gek\Collections\Typed\StringList;
use Gek\PhpLang\Contracts\IToIndentedString;

class UseItem implements IToIndentedString
{

    #region fields

    private StringList $segments;

    private string $name;

    private ?string $aliasName = null;

    #endregion fields

    #region ctor

    public function __construct(string $fullName, ?string $aliasName = null)
    {
        $this->segments =  new StringList();
        $this->init($fullName,$aliasName);
    }

    #endregion ctor

    #region properties

    public function getSegments():StringList{
        return $this->segments;
    }

    /**
     * @param StringList $segments
     * @return self
     */
    public function setSegments(StringList $segments):self {
        $this->segments = $segments;
        return $this;
    }

    /**
     * @return string
     */
    public function getName():string {
        return $this->name;
    }

    /**
     * @param string $name
     * @return self
     */
    public function setName(string $name):self {
        $this->name = $name;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getAliasName():?string {
        return $this->aliasName;
    }

    /**
     * @param string|null $aliasName
     * @return self
     */
    public function setAliasName(?string $aliasName):self {
        $this->aliasName = $aliasName;
        return $this;
    }

    #endregion properties

    #region method

    /**
     * @return bool
     */
    public function isAliasName():bool {
        return !empty($this->aliasName);
    }

    /**
     * @return string
     */
    public function getFullName():string {
        $strRes = '';
        if($this->segments->any()){
            $strRes .= implode("\\",$this->segments->toArray());
        }
        $strRes .= "\\" .  $this->name;
        return $strRes;
    }

    public function __toString()
    {
        $strRes = 'use ';
        if($this->segments->any()){
            $strRes .= implode("\\",$this->segments->toArray());
        }
        $strRes .= "\\" .  $this->name;
        if($this->aliasName !== null){
            $strRes .= " as " . $this->aliasName;
        }
        $strRes .= ";";

        return $strRes;
    }

    #endregion method

    #region utils

    /**
     * @param string $fullName
     * @param string|null $aliasName
     */
    private function init(string $fullName, ?string $aliasName = null){
        $segments = explode("\\",$fullName);
        $name = array_pop($segments);
        $this->segments->clear();
        foreach ($segments as $seg){
            if(!empty($seg)){
                $this->segments->add($seg);
            }
        }
        $this->name = $name;
        $this->aliasName = $aliasName;
    }

    #endregion utils


    #region IToIndentedString

    /**
     * @param int $indentLevel girinti seviyesi
     * @param string $indentChars girinti karakterleri
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $indent = str_repeat($indentChars,$indentLevel);
        $strRes = $indent . 'use ';
        if($this->segments->any()){
            $strRes .= implode("\\",$this->segments->toArray());
        }
        $strRes .= "\\" .  $this->name;
        if($this->aliasName !== null){
            $strRes .= " as " . $this->aliasName;
        }
        $strRes .= ";";

        return $strRes;
    }

    #endregion IToIndentedString

}
