<?php


namespace Gek\PhpLang;


use Gek\Collections\Enumerable;
use Gek\Collections\Typed\StringDictionary;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Traits\NameAware;
use Gek\Infrastructure\Str;

class TraitItem implements IToIndentedString
{

    #region fields

    protected StringDictionary $aliases;

    protected PhpType $type;

    #endregion fields

    #region ctor

    /**
     * TraitItem constructor.
     * @param string $name
     * @param Enumerable|StringDictionary|array<string,string>|null $aliases
     */
    public function __construct(string $name, $aliases = null)
    {
        $this->type = PhpType::parseClassType($name);
        if(!($aliases instanceof StringDictionary)){
            $aliases = new StringDictionary($aliases);
        }
        $this->setAliases($aliases);
    }

    #endregion ctor

    #region properties

    /**
     * @return PhpType
     */
    public function getType():PhpType{
        return $this->type;
    }

    /**
     * @param PhpType|string $type
     * @return $this
     */
    public function setType($type):self {
        if(is_string($type)){
            $type = PhpType::parseClassType($type);
        }
        $this->type = $type;
        return $this;
    }

    public function getTypeName():string {
        return $this->type->getName();
    }

    /**
     * @return StringDictionary
     */
    public function getAliases():StringDictionary{
        return $this->aliases;
    }

    /**
     * @param StringDictionary $aliases
     * @return self
     */
    public function setAliases(StringDictionary $aliases):self {
        $this->aliases = $aliases;
        return $this;
    }

    #endregion properties

    #region Method

    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion Method


    #region IToIndentedString

    /**
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0,string $indentChars = '    '): string
    {
        $indent = str_repeat($indentChars,$indentLevel);
        $typeName = $this->type->getName();
        if($this->type->isUseItem()){
            $typeName = $this->type->getUseItem()->getName();
            if($this->type->getUseItem()->isAliasName()){
                $typeName = $this->type->getUseItem()->getAliasName();
            }
        }
        $strRes = $indent. 'use ' . $typeName;
        if($this->getAliases()->any()){
            $strRes .= ' {' . PHP_EOL;
            $indent = str_repeat($indentChars,$indentLevel + 1);
            foreach ($this->getAliases() as $ky => $vl){
                $strRes.= Str::format('{0}{1} as {2};', $indent, $ky, $vl) . PHP_EOL;
            }
            $indent = str_repeat($indentChars,$indentLevel);
            $strRes .= $indent . '}';
        }else{
            $strRes .= ';';
        }
        return $strRes;
    }

    #endregion IToIndentedString
}
