<?php


namespace Gek\PhpLang;

use Gek\Collections\Enumerable;
use Gek\Infrastructure\Str;

/**
 * Class PhpTypeHelper
 * @package Gek\PhpLang
 * @internal
 */
class PhpTypeHelper
{
    private const FIELD_USE = 1;
    private const RETURN_USE = 2;
    private const PARAM_USE = 4;
    private const COMMENT_USE = 8;
    private const All_USE = self::FIELD_USE | self::RETURN_USE | self::PARAM_USE | self::COMMENT_USE;

    private const PHP_TYPES_USAGES = [
        // SCALAR_TYPES
        'bool' => self::All_USE,
        'boolean' => self::COMMENT_USE,
        'int' => self::All_USE,
        'integer' => self::COMMENT_USE,
        'float' => self::All_USE,
        'double' => self::COMMENT_USE,
        'string' => self::All_USE,

        //COMPOUND_TYPES
        'array' => self::All_USE,
        'object' => self::All_USE,
        'callable' => self::All_USE ^ self::FIELD_USE,
        'iterable' => self::All_USE,

        //SPECIAL_TYPES
        'resource' => self::COMMENT_USE,
        'null' => self::COMMENT_USE,

        //PSEUDO_TYPES
        'mixed' => self::COMMENT_USE,
        'number' => self::COMMENT_USE,
        'callback' => self::COMMENT_USE,
        'void' => self::RETURN_USE | self::COMMENT_USE,

        //LATE STATİC BINDINGS
        'self' => self::All_USE,
        'parent' => self::All_USE,
        'static' => self::COMMENT_USE,
    ];

    private const TYPE_MAP = [
        'boolean' => 'bool',
        'integer' => 'int',
        'double' => 'float',
        'callback' => 'callable',
    ];

    /**
     * @param string $type
     * @param bool $createUse
     * @param array|UseItem[] $useItems
     * @return string
     */
    public static function renderForComment(string $type, bool $createUse = false, array &$useItems = array()): string
    {
        return self::renderForPos(self::COMMENT_USE,$type,$createUse,$useItems);
    }


    /**
     * @param string $type
     * @param bool $createUse
     * @param array|UseItem[] $useItems
     * @return string
     */
    public static function renderForReturn(string $type, bool $createUse = false, array &$useItems = array()): string
    {
        return self::renderForPos(self::RETURN_USE,$type,$createUse,$useItems);
    }

    /**
     * @param string $type
     * @param bool $createUse
     * @param array|UseItem[] $useItems
     * @return string
     */
    public static function renderForField(string $type, bool $createUse = false, array &$useItems = array()): string
    {
        return self::renderForPos(self::FIELD_USE,$type,$createUse,$useItems);
    }

    /**
     * @param string $type
     * @param bool $createUse
     * @param array|UseItem[] $useItems
     * @return string
     */
    public static function renderForParam(string $type, bool $createUse = false, array &$useItems = array()): string
    {
        return self::renderForPos(self::PARAM_USE,$type,$createUse,$useItems);
    }

    protected static function renderForPos(int $pos,string $type, bool $createUse = false, array &$useItems = array()):string {
        $isNullable = self::isNullable($type);
        $cleanTypes = self::getCleanTypes($type);


        if ($pos !== self::COMMENT_USE) {
            if(count($cleanTypes) > 1){
                return '';
            }
            $clnType = $cleanTypes[0];

            if (isset(self::PHP_TYPES_USAGES[$clnType])) {
                if (self::PHP_TYPES_USAGES[$clnType] & $pos === $pos) {
                    return $isNullable ? '?' . $clnType : $clnType;
                } elseif (isset(self::TYPE_MAP[$clnType])) {
                    $clnType = self::TYPE_MAP[$clnType];
                    if (self::PHP_TYPES_USAGES[$clnType] & $pos === $pos) {
                        return $isNullable ? '?' . $clnType : $clnType;
                    }
                }
                return '';
            }
            $classType = static::renderClassTypeForCode($clnType, $isNullable, $createUse, $useItm);
            if ($createUse) {
                $useItems[] = $useItm;
            }
            return $classType;
        }

        foreach ($cleanTypes as &$clnType) {
            if (isset(self::PHP_TYPES_USAGES[$clnType])) {
                continue;
            }
            $clnType = static::renderClassTypeForCode($clnType, false, $createUse, $useItm);
            if ($createUse) {
                $useItems[] = $useItm;
            }
        }

        if ($isNullable) {
            $cleanTypes[] = 'null';
        }
        return implode('|', $cleanTypes);
    }

    /**
     * @param string $classType
     * @param bool $nullable
     * @param bool $createUse
     * @param UseItem|null $useItem
     * @return string
     */
    protected static function renderClassTypeForCode(string $classType, bool $nullable = false, bool $createUse = false, ?UseItem &$useItem = null): string
    {
        $classType = trim($classType);
        if ($createUse) {
            $useItem = new UseItem($classType);
            $classType = $useItem->getName();
        }
        if ($nullable) {
            $classType = '?' . $classType;
        }
        return $classType;
    }

    /**
     * @param  $type
     * @return bool
     */
    protected static function isNullable(string $type): bool
    {
        $type = trim($type);
        if (Str::contains($type, '|')) {
            return Enumerable::fromArray(explode('|', $type))
                ->any(function (string $item) {
                    $item = strtolower(trim($item));
                    return $item === 'null' || Str::startsWith($item, '?');
                });
        }
        return Str::startsWith($type, '?') || $type === 'null';
    }

    /**
     * @param string $type
     * @return array
     */
    protected static function getCleanTypes(string $type): array
    {
        $type = trim($type);
        $cleanTypes = array();
        if (Str::contains($type, '|')) {
            $cleanTypes = Enumerable::fromArray(explode('|', $type))
                ->where(function (string $item) {
                    return !empty(trim($item)) && strtolower(trim($item)) !== 'null';
                })->select(function (string $item) {
                    return ltrim($item, '?');
                })->toArray();

        } elseif (!empty($type) && strtolower($type) !== 'null') {
            $cleanTypes[] = ltrim($type, '?');
        }

        foreach ($cleanTypes as &$cln) {
            $test = strtolower($cln);
            if (isset(self::PHP_TYPES_USAGES[$test])) {
                $cln = $test;
            }
        }
        return $cleanTypes;
    }


}
