<?php


namespace Gek\PhpLang;


use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\Traits\NameAware;
use Gek\PhpLang\Traits\ValueAware;
use Gek\Infrastructure\Str;

class MethodParam implements IUseCreator
{
    use NameAware;
    use ValueAware;

    #region fields

    /**
     * @var PhpTypeDeclared|null
     */
    protected ?PhpTypeDeclared $type = null;

    /**
     * @var bool
     */
    protected bool $dotted = false;

    #endregion fields

    #region ctor

    /**
     * MethodParam constructor.
     * @param string $name
     * @param string|null $type
     * @param bool $dotted
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function __construct(string $name, $type = null, bool $dotted = false)
    {
        $this->setName($name)
            ->setType($type)
            ->setDotted($dotted);
    }

    #endregion ctor

    #region Properties

    /**
     * @return string|null
     */
    public function getType(): ?string
    {
        return $this->type !== null ?
            $this->type->getRaw() :
            null;
    }

    /**
     * @param PhpTypeDeclared|array|string|null $type
     * @return self
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     */
    public function setType($type): self
    {
        if(is_array($type)){
            $temp = array();
            foreach ($type as $tp){
                if($tp === null){
                    $tp = 'null';
                }
                if(empty($tp)){
                    continue;
                }
                $temp[] = $tp;
            }
            $type =  implode('|',$temp);
        }
        if(is_string($type)){
            $type = new PhpTypeDeclared($type);
        }
        $this->type = $type;
        return $this;
    }

    /**
     * @param bool $dotted
     * @return $this
     */
    public function setDotted(bool $dotted = true):self {
        $this->dotted = $dotted;
        return $this;
    }

    #endregion Properties

    #region Methods

    /**
     * @return bool
     */
    public function hasType(): bool
    {
        return $this->type !== null;
    }

    /**
     * @return bool
     */
    public function isDotted():bool {
        return $this->dotted;
    }


    public function renderForComment():string {
        $strRes = '';
        $type = $this->hasType() ? $this->type->renderForComment() : '';
        if(!empty($type)){
            $strRes .= $type. ' ';
        }
        if($this->dotted){
            $strRes .= '...';
        }

        $strRes .= '$' . $this->getName();
        return $strRes;
    }

    public function __toString()
    {
        $strRes = '';
        $type = $this->hasType() ? $this->type->renderForParam() : '';
        if(!empty($type)){
            $strRes .= $type. ' ';
        }
        if($this->isDotted()){
            $strRes .= '...';
        }
        $strRes .= '$' . $this->getName();
        if($this->isSetValue()){
            $strRes .= ' = ';
            if($this->getValueRenderType() === LiteralTypes::SINGLE_QUOTES){
                $strRes .= Str::format("'{0}'",$this->getValue());
            }elseif ($this->getValueRenderType() === LiteralTypes::DOUBLE_QUOTES){
                $strRes .= Str::format('"{0}"',$this->getValue());
            }else{
                $strRes .=  $this->getValue();
            }
        }
        return $strRes;
    }

    #endregion Methods

    #region utils

    /**
     * @param string $params
     * @return array|MethodParam[]
     * @throws \Gek\Infrastructure\Exceptions\GekException
     * @throws \ReflectionException
     * @throws \Exception
     */
    public static function parseParams(string $params): array
    {
        $resArray = array();
        $strParamArray = explode(',', $params);
        foreach ($strParamArray as $strParam) {
            $pType = static::parseType($strParam);
            $pName = static::parseName($strParam, $isDotted);
            $pValue = static::parseValue($strParam);

            $mp = new MethodParam($pName,$pType);
            $mp->setDotted($isDotted);
            if($pValue !== null){

                $renderType = LiteralTypes::NONE;
                if(Str::startsWith($pValue,'"')){
                    $pValue = trim($pValue,'"');
                    $renderType = LiteralTypes::DOUBLE_QUOTES;
                }elseif (Str::startsWith($pValue,'\'')){
                    $pValue = trim($pValue,"'");
                    $renderType = LiteralTypes::SINGLE_QUOTES;
                }
                $mp->setValue($pValue,$renderType);
            }
            $resArray[] = $mp;
        }
        return $resArray;
    }

    /**
     * @param string $strParam
     * @return string|null
     * @throws \Exception
     */
    private static function parseType(string &$strParam): ?string
    {
        $strParam = trim($strParam);
        if (Str::startsWith($strParam, '$')) {
            return null;
        }
        $spacePos = strpos($strParam, ' ');

        if ($spacePos === false) {
            throw new \Exception('format hatalı.(parseType)');
        }
        $type = substr($strParam, 0, $spacePos);
        $strParam = substr($strParam, $spacePos);
        return $type;
    }

    /**
     * @param string $strParam
     * @return string|null
     * @throws \Exception
     */
    private static function parseName(string &$strParam, &$isDotted): ?string
    {
        $strParam = trim($strParam);
        $isDotted = false;
        if(Str::startsWith($strParam,'...')){
            $isDotted = true;
            $strParam = trim(substr($strParam,2));
        }
        if (!Str::startsWith($strParam, '$')) {
            throw new \Exception('format hatalı.(parseName)');
        }
        $assignmentPos = strpos($strParam, '=');
        if ($assignmentPos === false) {
            $pName = ltrim($strParam, '$');
            $strParam = '';
        } else {
            $pName = substr($strParam, 0, $assignmentPos);
            $pName = trim($pName);
            $pName = ltrim($pName, '$');
            $strParam = substr($strParam, $assignmentPos);
        }
        return $pName;

    }

    private static function parseValue(string &$strParam): ?string
    {
        $strParam = trim($strParam," \t\n\r\0\x0B=");

        if ($strParam === '') {
            return null;
        }
        $pValue = trim($strParam);
        $strParam = '';
        return $pValue;

    }

    #endregion utils

    #region IUseCreator

    /**
     * @return array|UseItem[]
     */
    public function getUseArray(): array
    {
        if($this->hasType()){
            return $this->type->getUseArray();
        }
        return array();
    }

    #endregion IUseCreator
}
