<?php


namespace Gek\PhpLang\DocComments;



use Gek\Collections\Enumerable;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\Collections\DocCommentTagCollection;

class DocComment implements \Serializable, IToIndentedString
{

    #region fields

    private ?string $summary = null;

    private ?string $description = null;

    private DocCommentTagCollection $tags;

    #endregion fields

    #region ctor

    public function __construct()
    {
        $this->tags = new DocCommentTagCollection();
    }

    #endregion ctor

    #region properties

    /**
     * summary getter
     *
     * @return string|null
     */
    public function getSummary(): ?string
    {
        return $this->summary;
    }

    /**
     * summary setter
     *
     * @param string|null $summary
     * @return self
     */
    public function setSummary(?string $summary): self
    {
        $this->summary = $summary;
        return $this;
    }


    /**
     * description getter
     *
     * @return string|null
     */
    public function getDescription(): ?string
    {
        return $this->description;
    }


    /**
     * description  setter
     *
     * @param string|null $description
     * @return self
     */
    public function setDescription(?string $description): self
    {
        $this->description = $description;
        return $this;
    }

    /**
     * tags getter
     *
     * @return DocCommentTagCollection
     */
    public function getTags(): DocCommentTagCollection
    {
        return $this->tags;
    }

    /**
     * tags getter
     *
     * @param DocCommentTagCollection $tags
     * @return self
     */
    public function setTags(DocCommentTagCollection $tags): self
    {
        $this->tags = $tags;
        return $this;
    }



    #endregion properties

    #region methods

    /**
     * @return bool
     */
    public function isEmpty():bool {
        return empty($this->summary) && empty($this->description) && !$this->tags->any();
    }

    /**
     * @param DocCommentTag|string $tag
     * @param string|null $value
     * @return self
     */
    public function addTag($tag, ?string $value = ''):self {
        if($tag instanceof DocCommentTag){
            $this->tags->add($tag);
        }else{
            $this->tags->addTag($tag,$value);
        }
        return $this;
    }

    /**
     * @param string $summaryLine
     * @return self
     */
    public function addSummary(string $summaryLine):self{
        if($this->summary === null){
            $this->summary = '';
        }
        $this->summary .= $summaryLine . PHP_EOL;
        return $this;
    }

    /**
     * @param string $descriptionLine
     * @return self
     */
    public function addDescription(string $descriptionLine):self{
        if($this->description === null){
            $this->description = '';
        }
        $this->description .= $descriptionLine . PHP_EOL;
        return $this;
    }

    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion methods

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            'summary' => $this->summary,
            'description' => $this->description,
            'tags' => $this->tags->select(
                function (DocCommentTag $tag){
                    return array(
                        $tag->key,
                        $tag->value
                    );
                }
            )->toArray(),
        ];
        var_dump($data);
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->summary = $data['summary'];
        $this->description = $data['description'];

        $tags = Enumerable::fromArray($data['tags'])
            ->select(function(array $item) {
                    return new DocCommentTag($item[0],$item[1]);
                });
        var_dump($tags->toArray());
        $this->tags = new DocCommentTagCollection(...$tags);
    }

    #endregion Serializable

    #region IToIndentedString

    /**
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0,string $indentChars = '    '): string
    {
        $indent = str_repeat($indentChars,$indentLevel);
        $resStr = $indent . "/**" . PHP_EOL;

        $resStr .= $indent . " * " . $this->getSummary().PHP_EOL;
        $resStr .= !empty($this->getSummary())?  $indent . " * " . PHP_EOL : '';
        if($this->description != null){
            $lines = explode(PHP_EOL,$this->description);
            foreach ($lines as $line){
                $resStr .= $indent . " * " . $line . PHP_EOL;
            }
            $resStr .= $indent . " * " . PHP_EOL;
        }
        if($this->tags->any()){
            foreach ($this->tags as $tg){
                $resStr .= $indent . " * ". $tg . PHP_EOL;
            }
        }
        $resStr .= $indent . " */ " . PHP_EOL;

        return $resStr;
    }

    #endregion IToIndentedString

}
