<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\PhpLang\UseItem;
use phpDocumentor\Reflection\Types\This;

class UseItemCollection extends TypedListWrapper implements IToIndentedString
{

    #region ctor

    /**
     * IntList constructor.
     * @param UseItem ...$items
     */
    public function __construct(UseItem ...$items)
    {
        parent::__construct(new ArrayList(), UseItem::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * @param UseItem $item
     */
    public function add(UseItem $item): void
    {
        $this->innerList->add($item);
    }

    /**
     * @param UseItem ...$items
     */
    public function addRange(UseItem ...$items): void
    {
        $this->innerList->addRange($items);
    }


    /**
     * @param UseItem $item
     * @return bool
     */
    public function contains(UseItem $item): bool
    {
        return $this->innerList->contains($item);
    }

    /**
     * @param UseItem $item
     * @return bool
     */
    public function remove(UseItem $item): bool
    {
        return $this->innerList->remove($item);
    }

    /**
     * @param UseItem $item
     * @return int
     */
    public function indexOf(UseItem $item): int
    {
        return $this->innerList->indexOf($item);
    }

    /**
     * @param int $index
     * @param UseItem $item
     */
    public function insert(int $index, UseItem $item): void
    {
        $this->innerList->insert($index, $item);
    }

    /**
     * @param int $index
     * @return UseItem
     */
    public function getAt(int $index): UseItem
    {
        return $this->innerList[$index];
    }

    /**
     * @param int $index
     * @param UseItem $item
     */
    public function setAt(int $index, UseItem $item): void
    {
        $this->innerList[$index] = $item;
    }

    /**
     * @param int $index
     * @param UseItem ...$items
     */
    public function insertRange(int $index, UseItem ...$items): void
    {
        $this->innerList->insertRange($index, $items);
    }


    /**
     * @param array|UseItem[] $useItems
     */
    public function syncUseItems(array $useItems)
    {
        if (empty($useItems)) {
            return;
        }

        foreach ($useItems as $useItem) {
            /** @var UseItem $itm */
            $itm = $this->firstOrNull(function (UseItem $item) use ($useItem) {
                return $item->getFullName() == $useItem->getFullName();
            });
            if (!empty($itm)) {
                if ($itm->isAliasName()) {
                    $useItem->setAliasName($itm->getAliasName());
                }
            } else {
                $this->add($useItem);
            }
        }
        $that = $this->toArrayList();

        /** @var ArrayList|UseItem[] $nameConflicts */
        $nameConflicts = $this->where(function (UseItem $item) use ($that) {
            return $that->any(function (UseItem $itm) use ($item) {
                return ($itm->getName() == $item->getName()) &&
                    ($itm->getAliasName() == $item->getAliasName()) &&
                    ($itm->getFullName() != $item->getFullName())
                    ;
            });
        })->toArrayList();


        while ($nameConflicts->any()) {
            $lastName = "";
            foreach ($nameConflicts as $cnfItem) {
                if ($cnfItem->isAliasName()) {
                    if ($cnfItem->getAliasName() === $lastName || $cnfItem->getName() === $lastName) {
                        continue;
                    }
                    $lastName = $cnfItem->getAliasName();
                    /** @var UseItem $lastItem */
                    $lastItem = $this->lastOrNull(function (UseItem $lastItm) use ($lastName) {
                        return $lastItm->getAliasName() == $lastName;
                    });
                    if (is_numeric(substr($lastName, -1))) {
                        $num = intval(substr($lastName, -1)) + 1;
                        $lastName = substr($lastName, 0, -1) . $num;
                    } else {
                        $lastName .= '1';
                    }
                    $lastItem->setAliasName($lastName);

                } else {
                    if ($cnfItem->getName() === $lastName) {
                        continue;
                    }
                    $lastName = $cnfItem->getName();
                    /** @var UseItem $lastItem */
                    $lastItem = $this->lastOrNull(function (UseItem $lastItm) use ($lastName) {
                        return $lastItm->getName() == $lastName;
                    });
                    $lastItem->setAliasName($lastName . '1');


                }
            }
            $nameConflicts = $this->where(function (UseItem $item) use ($that) {
                return $that->any(function (UseItem $itm) use ($item) {
                    return $itm->getName() == $item->getName() &&
                        $itm->getAliasName() == $item->getAliasName() &&
                        $itm->getFullName() != $item->getFullName();
                });
            })->toArrayList();
        }

    }

    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion methods

    /**
     * @param int $indentLevel girinti seviyesi
     * @param string $indentChars girinti karakterleri
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        return $this->aggregate(function (&$lines, UseItem $item) use ($indentLevel, $indentChars) {
            $lines .= $item->toIndentedString($indentLevel, $indentChars) . PHP_EOL;
        }, '');
    }
}
