<?php


namespace Gek\PhpLang;


use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\Infrastructure\Str;

class CodeLine implements IToIndentedString
{

    #region fields

    protected string $code = '';

    protected int $indentLevel = 0;

    #endregion fields

    #region ctor

    public function __construct(string $code = '',int $indentLevel = 0)
    {
        $this->setCode($code)
            ->setIndentLevel($indentLevel);
    }

    #endregion ctor

    #region properties

    /**
     * @return string
     */
    public function  getCode():string {
        return $this->code;
    }

    /**
     * @param string $code
     * @return CodeLine
     */
    public function setCode(string $code):self {
        $this->code = $code;
        return $this;
    }

    /**
     * @return int
     */
    public function getIndentLevel():int{
        return $this->indentLevel;
    }

    public function setIndentLevel(int $indentLevel):self{
        if($indentLevel < 0){
            $indentLevel = 0;
        }
        $this->indentLevel = $indentLevel;
        return $this;
    }

    #endregion properties

    #region method

    /**
     * @param int $add
     * @return int
     */
    public function incrementIndentLevel(int $add = 1):int {
        $this->setIndentLevel($this->indentLevel + $add);
        return $this->getIndentLevel();
    }

    /**
     * @param int $sub
     * @return int
     */
    public function decrementIndentLevel(int $sub = 1):int {
        $this->setIndentLevel($this->indentLevel - $sub);
        return $this->getIndentLevel();
    }

    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion method

    #region utils

    /**
     * @param string $code
     * @return CodeLine
     */
    public static function parseLine(string $code):CodeLine{
        $indent = '    ';
        $indentCount = strlen($indent);
        $indentLevel = 0;
        while (Str::startsWith($code,$indent)){

            $indentLevel++;
            $code = substr($code,$indentCount);
        }
        $codeLine = new CodeLine(trim($code),$indentLevel);
        return $codeLine;
    }

    /**
     * @param string $codes
     * @return array
     */
    public static function parseCodes(string $codes):array {
        $resArr = array();

        $linesArr = explode(PHP_EOL,$codes);
        foreach ($linesArr as $strLine){
            $resArr[] = static::parseLine($strLine);
        }
        return $resArr;
    }

    #endregion utils

    #region IToIndentedString

    /**
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0,string $indentChars = '    '): string
    {
        return str_repeat($indentChars,$this->indentLevel + $indentLevel) . $this->code;
    }

    #endregion IToIndentedString
}
