<?php


namespace Gek\PhpLang;


use Gek\Collections\ArrayList;
use Gek\PhpLang\Contracts\IToIndentedString;

/**
 * PhpFile Sınıfı
 *
 * Php dosyası sınıfı
 *
 * @package Gek\PhpLang
 */
class PhpFile implements IToIndentedString
{
    #region fields

    /**
     * Ad alanı
     * @var string|null
     */
    protected ?string $namespace = null;

    /**
     * kodlar
     * @var ArrayList|IToIndentedString[]
     */
    protected ArrayList $contents;

    #endregion fields

    #region ctor

    /**
     * PhpFile yapıcı metod.
     * @param string|null $namespace >(opsiyonel) ad alanı.
     */
    public function __construct(?string $namespace = null)
    {
        $this->namespace = $namespace;
        $this->contents = new ArrayList();
    }

    #endregion ctor

    #region properties

    /**
     * ad alanını verir.
     * @return string|null ad alanı
     */
    public function getNamespace(): ?string
    {
        return $this->namespace;
    }

    /**
     * ad alanını set eder
     * @param string|null $namespace ad alanı
     * @return PhpFile
     */
    public function setNamespace(?string $namespace): PhpFile
    {
        $this->namespace = $namespace;
        return $this;
    }

    #endregion properties

    #region methods

    /**
     * dosyaya kod ekler
     * @param IToIndentedString|string $content kod içeriği
     * @return $this
     */
    public function addContent($content):self {
        if($content instanceof PhpClass){
            if(empty($this->namespace)){
                $this->namespace = $content->getNamespace();
            }

            $content->setNamespace(null);
        }
        if(is_string($content)){
            $content = new CodeLine($content);
        }
        $this->contents->add($content);
        return $this;
    }

    /**
     * dosyayı metne (php koduna) çevirir.
     * @return string metin (php kodu)
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }



    #endregion methods

    #region IToIndentedString

    /**
     * dosyayı girintili metne (php koduna) çevirir.
     * @param int $indentLevel girinti düzeyi
     * @param string $indentChars girinti karakterleri
     * @return string girintili metin (php kodu)
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $str = '<?php ' . PHP_EOL;
        if(!empty($this->namespace)){
            $str .= 'namespace ' . $this->getNamespace() . ';' . PHP_EOL;
        }
        foreach ($this->contents as $cnt){
            $str .= $cnt->toIndentedString($indentLevel,$indentChars);
            if($cnt instanceof CodeLine){
                $str .= PHP_EOL;
            }
        }
        return $str;
    }

    #endregion IToIndentedString

}