<?php


namespace Gek\PhpLang\DocComments;


use Gek\Infrastructure\Str;
use Gek\PhpLang\Traits\MagicGetSetAware;

/**
 * DocCommentTag Sınıfı
 *
 * docComment tag verisini tutan sınıf.
 *
 * @package Gek\PhpLang\DocComments
 * @property string $key
 * @property string $value
 */
class DocCommentTag implements \Serializable
{
    use MagicGetSetAware;

    #region fields

    /**
     * key (tag adı)
     * @var string
     */
    private string $_key;

    /**
     * value (tag değeri)
     * @var string
     */
    private string $_value;

    #endregion fields

    #region ctor

    /**
     * DocCommentTag yapıcı metod.
     * @param string $key tag adı
     * @param string $value tag değeri
     */
    public function __construct(string $key,string $value = '')
    {
        $this->setKey($key);
        $this->setValue($value);
    }

    #endregion ctor

    #region properties

    /**
     * key getter.
     * @return string key (tag adı)
     */
    public function getKey():string {
        return $this->_key;
    }

    /**
     * key setter.
     * @param string $key key (tag adı)
     * @return $this
     */
    public function setKey(string $key):self {
        $key = $this->fixKey($key);
        if($key === ''){
            throw new \InvalidArgumentException('$key boş.');
        }
        $this->_key = $key;
        return $this;
    }

    /**
     * value getter
     *
     * @return string tag değeri
     */
    public function getValue():string {
        return $this->_value;
    }

    /**
     * value setter
     * @param string $value tag değeri
     * @return $this
     */
    public function setValue(string $value):self {
        $this->_value = $value;
        return $this;
    }



    #endregion properties

    #region methods

    /**
     * docComment tagı metne çevirir.
     * @return string tag metni
     */
    public function __toString()
    {
        return trim(Str::format('@{0} {1}',$this->_key,$this->_value));
    }

    #endregion methods

    #region utils

    /**
     * keyi (tag adını) formal hale getirir.
     * @param string $key tag adı
     * @return string formal hale getirilmiş tag adı
     */
    private function fixKey(string $key):string {
        $key = trim($key);
        $key = str_replace('@','',$key);
        $key = preg_replace("/\n\r\t+/", ' ', $key);
        $key = preg_replace("/\s+/", ' ', $key);
        //$key = preg_replace("/_+/", '_', $key);
        $key = str_replace([' ','_'],'-',$key);
        $key = strtolower($key);
        return $key;
    }
    #endregion utils

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            'k' => $this->_key
        ];
        if($this->_value !== ''){
            $data['v']= $this->_value;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->_key = $data['k'];
        if(isset($data['v'])){
            $this->_value = $data['v'];
        }else{
            $this->_value ='';
        }
    }

    #endregion Serializable
}
