<?php


namespace Gek\PhpLang\DocComments;


use Gek\Infrastructure\Str;
use Gek\PhpLang\Collections\DocCommentTagCollection;
use Gek\PhpLang\Contracts\IToIndentedString;

/**
 * DocComment sınıfı
 *
 * DocComment verilerini barındıran sınıf.
 *
 * @package Gek\PhpLang\DocComments
 */
class DocComment implements \Serializable, IToIndentedString
{

    #region fields

    /**
     * Özet
     * @var string|null
     */
    private ?string $summary = null;

    /**
     * Açıklama
     * @var string|null
     */
    private ?string $description = null;

    /**
     * Tag koleksiyonu
     * @var DocCommentTagCollection
     */
    private DocCommentTagCollection $tags;

    #endregion fields

    #region ctor

    /**
     * DocComment yapıcı metod.
     */
    public function __construct()
    {
        $this->tags = new DocCommentTagCollection();
    }

    #endregion ctor

    #region properties

    /**
     * summary getter
     *
     * @return string|null
     */
    public function getSummary(): ?string
    {
        return $this->summary;
    }

    /**
     * summary setter
     *
     * @param string|null $summary
     * @return self
     */
    public function setSummary(?string $summary): self
    {
        $this->summary = $summary;
        return $this;
    }


    /**
     * description getter
     *
     * @return string|null
     */
    public function getDescription(): ?string
    {
        return $this->description;
    }


    /**
     * description  setter
     *
     * @param string|null $description
     * @return self
     */
    public function setDescription(?string $description): self
    {
        $this->description = $description;
        return $this;
    }

    /**
     * tags getter
     *
     * @return DocCommentTagCollection
     */
    public function getTags(): DocCommentTagCollection
    {
        return $this->tags;
    }

    /**
     * tags setter
     *
     * @param DocCommentTagCollection $tags
     * @return self
     */
    public function setTags(DocCommentTagCollection $tags): self
    {
        $this->tags = $tags;
        return $this;
    }



    #endregion properties

    #region methods

    /**
     * doc commentin boş olup olmadığına bakar.
     * @return bool boşsa true değilse false
     */
    public function isEmpty(): bool
    {
        return empty($this->summary) && empty($this->description) && !$this->tags->any();
    }

    /**
     * yeni tag ekler.
     * @param DocCommentTag|string $tag tag
     * @param string|null $value tag değeri
     * @return $this
     */
    public function addTag($tag, ?string $value = ''): self
    {
        if ($tag instanceof DocCommentTag) {
            $this->tags->add($tag);
        } else {
            $this->tags->addTag($tag, $value);
        }
        return $this;
    }

    /**
     * verilen tagın olup olmadığını kontrol edip yoksa ekler.
     * @param DocCommentTag|string $tag tag
     * @param string|null $value tag değeri
     * @return $this
     */
    public function checkAndAddTag($tag, ?string $value = ''): self
    {
        if (!($tag instanceof DocCommentTag)) {
            $tag = new DocCommentTag($tag, $value);
        }
        $tagStr = $tag->__toString();
        $check = $this->tags->toArrayList()
            ->any(function (DocCommentTag $t) use($tagStr){
                return $t->__toString() == $tagStr;
            });
        if(!$check){
            $this->tags->add($tag);
        }


        return $this;
    }

    /**
     *  Özete metin ekler.
     * @param string $summaryLine yeni metin satırı.
     * @return $this
     */
    public function addSummary(string $summaryLine): self
    {
        if ($this->summary === null) {
            $this->summary = '';
        }
        $this->summary .= $summaryLine . PHP_EOL;
        return $this;
    }

    /**
     * Açıklamaya metin ekler.
     * @param string $descriptionLine açıklama satırı.
     * @return $this
     */
    public function addDescription(string $descriptionLine): self
    {
        if ($this->description === null) {
            $this->description = '';
        }
        $this->description .= $descriptionLine . PHP_EOL;
        return $this;
    }

    /**
     * DocComment i metne (php docComment e) çevirir.
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion methods

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = [
            's' => $this->summary,
            'd' => $this->description,
            't' => $this->tags,
        ];
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->summary = $data['s'];
        $this->description = $data['d'];
        $this->tags = $data['t'];
    }

    #endregion Serializable

    #region IToIndentedString

    /**
     * DocCommenti girintili metne çevirir.
     * @param int $indentLevel
     * @param string $indentChars
     * @return string
     */
    public function toIndentedString(int $indentLevel = 0, string $indentChars = '    '): string
    {
        $indent = str_repeat($indentChars, $indentLevel);
        $resStr = $indent . "/**" . PHP_EOL;

        $resStr .= $indent . " * " . $this->getSummary() . PHP_EOL;
        $resStr .= !empty($this->getSummary()) ? $indent . " * " . PHP_EOL : '';
        if ($this->description != null) {
            $lines = explode(PHP_EOL, $this->description);
            foreach ($lines as $line) {
                $resStr .= $indent . " * " . $line . PHP_EOL;
            }
            $resStr .= $indent . " * " . PHP_EOL;
        }
        if ($this->tags->any()) {
            foreach ($this->tags as $tg) {
                $resStr .= $indent . " * " . $tg . PHP_EOL;
            }
        }
        $resStr .= $indent . " */ " . PHP_EOL;

        return $resStr;
    }

    #endregion IToIndentedString

    #region static

    /**
     * docCommenti parse eder.
     * @param string $docComment doccomment metni
     * @return DocComment DocComment nesnesi
     */
    public static function parse(string $docComment){
        $res = new self();
        if(empty($docComment)){
            return $res;
        }

        $fixedEndLine = str_replace(chr(10), PHP_EOL, $docComment);
        $keys = static::readAllKeys($fixedEndLine);
        $commentLines = array();
        $summaryLines = array();
        $summaryFlag = true;
        $fixedEndLine = str_replace($keys, '', $fixedEndLine);
        $lines = explode(PHP_EOL, $fixedEndLine);
        foreach ($lines as $line) {
            $line = trim($line, " \t\n\r\0\x0B*/");

            if (!empty($line) && !in_array($line, array('*', '/**', '/*', '*/'))) {
                if ($summaryFlag) {
                    $summaryLines[] = $line;
                } else {
                    $commentLines[] = $line;
                }

            } elseif ($summaryFlag && !empty($summaryLines)) {
                $summaryFlag = false;
            }
        }
        $res->setSummary(implode(" ", $summaryLines));
        $res->setDescription(implode(PHP_EOL,$commentLines));
        foreach ($keys as $currentKey) {
            $keyEndPos = Str::indexOf($currentKey, ' ');
            if ($keyEndPos !== false) {
                $key = str_replace('@', '', mb_substr($currentKey, 0, $keyEndPos));
                $key = str_replace('@', '', trim($key));
                $value = trim(mb_substr($currentKey, $keyEndPos));

            } else {
                $key = str_replace('@', '', $currentKey);
                $key = str_replace('@', '', trim($key));
                $value = "";
            }
            $res->addTag($key,$value);
        }
        return $res;
    }

    /**
     * Bütün tag ları array olarak verir.
     *
     * @param string $docComment doccomment metni
     * @return array|string[] tag dizisi
     */
    private static function readAllKeys(string $docComment): array
    {
        $keys = array();
        $flag = 0;
        $currentTerm = '';
        $prev = '';

        for ($i = 0; $i < strlen($docComment); $i++) {
            $char = $docComment[$i];
            if ($char == chr(10)) {
                $char = PHP_EOL;
            }

            if ($char == '@') {
                if ($flag == 1) {
                    $keys[] = trim($currentTerm, " \t\n\r\0\x0B*");
                    $currentTerm = $char;
                } else {
                    $flag = 1;
                    $currentTerm .= $char;
                }
            } elseif ($char == '/' && $prev == '*') {
                if ($flag == 1) {
                    $keys[] = trim(substr($currentTerm, 0, -1), " \t\n\r\0\x0B*");
                    $currentTerm = '';
                    $flag = 0;
                }
            } else {
                if ($flag == 1) {
                    $currentTerm .= $char;
                }
            }
            $prev = $char;
        }
        if ($flag == 1) {
            $keys[] = trim($currentTerm, " \t\n\r\0\x0B*");
        }
        return $keys;
    }


    #endregion static

}
