<?php


namespace Gek\PhpLang\Collections;


use Gek\Collections\ArrayList;
use Gek\Collections\TypedListWrapper;
use Gek\PhpLang\ClassMember;
use Gek\PhpLang\Contracts\IUseCreator;
use Gek\PhpLang\UseItem;

/**
 * ClassMemberCollection sınıfı
 *
 * Sınıf öğeleri koleksiyonu
 *
 *
 * @package Gek\PhpLang\Collections
 */
class ClassMemberCollection extends TypedListWrapper implements IUseCreator, \Serializable
{

    #region ctor

    /**
     * ClassMemberCollection yapıcı metod.
     * @param ClassMember ...$items sınıf öğeleri
     */
    public function __construct(ClassMember ...$items)
    {
        parent::__construct(new ArrayList(), ClassMember::class);
        $this->addRange(...$items);
    }

    #endregion ctor

    #region methods

    /**
     * öğe ekler
     *
     * @param ClassMember $item sınıf öğesi
     */
    public function add(ClassMember $item): void
    {
        $this->innerList->add($item);
    }

    /**
     * Çoklu öğe ekler
     *
     * @param ClassMember ...$items sınıf öğeleri
     */
    public function addRange(ClassMember ...$items): void
    {
        $this->innerList->addRange($items);
    }


    /**
     * verilen öğenin kıleksitonda olup olmadığına bakar
     *
     * @param ClassMember $item sınıf öğesi
     * @return bool varsa true yoksa false
     */
    public function contains(ClassMember $item): bool
    {
        return $this->innerList->contains($item);
    }

    /**
     * öğeyi koleksiyondan kaldırır.
     *
     * @param ClassMember $item sınıf öğesi
     * @return bool öğe kaldırlısa true yoksa false
     */
    public function remove(ClassMember $item): bool
    {
        return $this->innerList->remove($item);
    }

    /**
     * verilen öğenin koleksiyondaki indeksini verir.
     *
     * @param ClassMember $item sınıf öğesi
     * @return int öğe indeksi öğe bulunamazsa -1 döner
     */
    public function indexOf(ClassMember $item): int
    {
        return $this->innerList->indexOf($item);
    }

    /**
     * verilen indekse öğeyi ekler
     *
     * @param int $index eklenecek indeks
     * @param ClassMember $item sınıf öğesi
     */
    public function insert(int $index, ClassMember $item): void
    {
        $this->innerList->insert($index, $item);
    }

    /**
     * verilen indeksteki öğeyi verir.
     *
     * @param int $index öğe indeksi
     * @return ClassMember sınıf öğesi
     */
    public function getAt(int $index): ClassMember
    {
        return $this->innerList[$index];
    }

    /**
     * Verilen indekse öğeyi set eder.
     *
     * @param int $index indeks
     * @param ClassMember $item sınıf öğesi
     */
    public function setAt(int $index, ClassMember $item): void
    {
        $this->innerList[$index] = $item;
    }

    /**
     * verilen indeksten itibaren verilen öğeleri ekler
     * @param int $index indeks
     * @param ClassMember ...$items sınıf öğeleri
     */
    public function insertRange(int $index, ClassMember ...$items): void
    {
        $this->innerList->insertRange($index, $items);
    }

    #endregion methods

    #region IUseCreator

    /**
     * @inheritDoc
     */
    public function getUseArray(): array
    {
        return $this->where(function (ClassMember $item) {
            return ($item instanceof IUseCreator);
        })->aggregate(function (&$arr, IUseCreator $item) {
            $arr = array_merge($arr, $item->getUseArray());
        }, array());
    }

    #endregion IUseCreator

    #region Serializable

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = $this->toArray();
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data =  unserialize($serialized);
        parent::__construct(new ArrayList(), ClassMember::class);
        $this->addRange(...$data);
    }

    #endregion Serializable
}
