<?php


namespace Gek\PhpLang;


use Gek\PhpLang\Contracts\IToIndentedString;
use Gek\Infrastructure\Str;

/**
 * CodeLine sınıfı
 * kod satırını tutar
 * @package Gek\PhpLang
 */
class CodeLine implements IToIndentedString, \Serializable
{

    #region fields

    /**
     * Kod metni
     * @var string
     */
    protected string $code = '';

    /**
     * Girinti düzeyi
     * @var int
     */
    protected int $indentLevel = 0;

    #endregion fields

    #region ctor

    /**
     * CodeLine yapıcı metod.
     * @param string $code kod metni
     * @param int $indentLevel girinti düzeyi
     */
    public function __construct(string $code = '',int $indentLevel = 0)
    {
        $this->setCode($code)
            ->setIndentLevel($indentLevel);
    }

    #endregion ctor

    #region properties

    /**
     * Kod metnini verir.
     *
     * @return string
     */
    public function  getCode():string {
        return $this->code;
    }

    /**
     * Kod metnini set eder.
     *
     * @param string $code kod metni
     * @return $this
     */
    public function setCode(string $code):self {
        $this->code = $code;
        return $this;
    }

    /**
     * Girinti düzeyini verir.
     *
     * @return int girinti seviyesi
     */
    public function getIndentLevel():int{
        return $this->indentLevel;
    }

    /**
     * Girinti seviyesini set eder.
     *
     * @param int $indentLevel girinti seviyesi.
     * @return $this
     */
    public function setIndentLevel(int $indentLevel):self{
        if($indentLevel < 0){
            $indentLevel = 0;
        }
        $this->indentLevel = $indentLevel;
        return $this;
    }

    #endregion properties

    #region method

    /**
     * Girinti seviyesini arttırır.
     *
     * @param int $add artış
     * @return int geçerli girinti seviyesi
     */
    public function incrementIndentLevel(int $add = 1):int {
        $this->setIndentLevel($this->indentLevel + $add);
        return $this->getIndentLevel();
    }

    /**
     * Girinti seviyesini azaltır.
     * @param int $sub azaltılacak sayı
     * @return int geçerli girinti seviyesi
     */
    public function decrementIndentLevel(int $sub = 1):int {
        $this->setIndentLevel($this->indentLevel - $sub);
        return $this->getIndentLevel();
    }

    /**
     * kod satırını metne (php koduna) çevirir.
     * @return string
     */
    public function __toString()
    {
        return $this->toIndentedString();
    }

    #endregion method

    #region utils

    /**
     * Verilen metinden kod satırı oluşturur.
     *
     * @param string $code kod metni
     * @return CodeLine kod satırı.
     */
    public static function parseLine(string $code):CodeLine{
        $indent = '    ';
        $indentCount = strlen($indent);
        $indentLevel = 0;
        while (Str::startsWith($code,$indent)){

            $indentLevel++;
            $code = substr($code,$indentCount);
        }
        $codeLine = new CodeLine(trim($code),$indentLevel);
        return $codeLine;
    }

    /**
     * verilen metinden kod satırları oluşturur.
     * @param string $codes kod metni.
     * @return array Kıd satırları.
     */
    public static function parseCodes(string $codes):array {
        $resArr = array();

        $linesArr = explode(PHP_EOL,$codes);
        foreach ($linesArr as $strLine){
            $resArr[] = static::parseLine($strLine);
        }
        return $resArr;
    }

    #endregion utils

    #region IToIndentedString

    /**
     * kod satırını girintili metne (php koduna) çevirir.
     * @param int $indentLevel girinti düzeyi.
     * @param string $indentChars girinti karakterleri
     * @return string girintili metin (php kodu)
     */
    public function toIndentedString(int $indentLevel = 0,string $indentChars = '    '): string
    {
        return str_repeat($indentChars,$this->indentLevel + $indentLevel) . $this->code;
    }

    #endregion IToIndentedString

    /**
     * String representation of object
     * @link https://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        $data = ['c' => $this->code];
        if($this->indentLevel > 0){
            $data['il'] = $this->indentLevel;
        }
        return serialize($data);
    }

    /**
     * Constructs the object
     * @link https://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $data = unserialize($serialized);
        $this->code = $data['c'];
        $this->indentLevel = isset($data['il']) ? $data['il'] : 0;

    }
}
