<?php
declare(strict_types=1);

namespace Gek\Infrastructure\Tests;


use Gek\Infrastructure\Str;
use Gek\Infrastructure\Tests\Mocups\DummyClassForStr;
use Gek\Infrastructure\Tests\Mocups\DummyClassWithToStringForStr;
use PHPUnit\Framework\TestCase;

/**
 * Class StrTest
 * @package Gek\Infrastructure\Tests
 */
class StrTest extends TestCase
{

    #region Str::format

    /**
     * Str::format temel test
     */
    public function testFormatBasic(): void
    {
        $this->assertEquals(
            'deneme bir, iki',
            Str::format('deneme {0}, {1}', 'bir', 'iki')
        );

    }

    /**
     * Str::format kaçış karakterleri testi
     */
    public function testFormatEscaped(): void
    {
        $this->assertEquals(
            'deneme bir, iki - {',
            Str::format('deneme {0}, {1} - {{', 'bir', 'iki')
        );

        $this->assertEquals(
            'deneme bir, iki - }',
            Str::format('deneme {0}, {1} - }}', 'bir', 'iki')
        );

        $this->assertEquals(
            'deneme bir, iki - {0}',
            Str::format('deneme {0}, {1} - {{0}}', 'bir', 'iki')
        );

        $this->assertEquals(
            'deneme bir, iki - {bir}',
            Str::format('deneme {0}, {1} - {{{0}}}', 'bir', 'iki')
        );

        $this->assertEquals(
            'deneme bir, iki - {bir} {{iki}}',
            Str::format('deneme {0}, {1} - {{{0}}} {{{{{1}}}}}', 'bir', 'iki')
        );
    }

    /**
     * Str::format params yerine array kullanımı testi
     */
    public function testFormatArrayParams(): void
    {
        $this->assertEquals(
            'deneme bir, iki',
            Str::format('deneme {0}, {1}', array('bir', 'iki'))
        );
    }

    /**
     * Str::format karmaşık (karışık ve çoklu) kullanım testi
     */
    public function testFormatComplexOrder(): void
    {
        $this->assertEquals(
            'deneme bir, iki test iki bir',
            Str::format('deneme {0}, {1} test {1} {0}', 'bir', 'iki')
        );
    }

    /**
     * Str::format çok uzun parametre sayısı testi
     */
    public function testFormatVeryLongParameterCount(): void
    {
        $format = "";
        $checkString = "";
        $argumentArr = array();

        for ($i = 0; $i < 5000; $i++) {
            $format .= "{" . $i . "}-";
            $checkString .= "arg" . $i . "-";
            $argumentArr[] = "arg" . $i;
        }
        $this->assertEquals(
            $checkString,
            Str::format($format, $argumentArr)
        );

    }


    public function testFormatArgumentTypes(): void
    {
        $this->assertEquals(
            'deneme true, false',
            Str::format('deneme {0}, {1}', true, false)
        );
        $this->assertEquals(
            'deneme 456, 7854',
            Str::format('deneme {0}, {1}', 456, 7854)
        );
        $this->assertEquals(
            'deneme 5.454, 456456.785',
            Str::format('deneme {0}, {1}', 5.454, 456456.785)
        );
        $this->assertEquals(
            'deneme 5.454, [array]',
            Str::format('deneme {0}, {1}', 5.454, array())
        );
        $this->assertEquals(
            'deneme 5.454, [stdClass]',
            Str::format('deneme {0}, {1}', 5.454, (object)array())
        );

        $dmyCls = new DummyClassForStr();
        $this->assertEquals(
            'deneme 5.454, [Gek\Infrastructure\Tests\Mocups\DummyClassForStr]',
            Str::format('deneme {0}, {1}', 5.454, $dmyCls)
        );

        $dmyClsTs = new DummyClassWithToStringForStr(5, 'test');
        $this->assertEquals(
            'deneme 5.454, foo: 5 bar: test',
            Str::format('deneme {0}, {1}', 5.454, $dmyClsTs)
        );

        $dt = new \DateTime('2020-05-03 20:39:28');

        $this->assertEquals(
            $dt->format("c"),
            Str::format('{0:c}', $dt)
        );
        $this->assertEquals(
            'deneme 5.454, 2020-05-03 20:39:28',
            Str::format('deneme {0}, {1}', 5.454, $dt)
        );
        $this->assertEquals(
            'deneme 5.454, 2020-05-03T20:39:28+00:00',
            Str::format('deneme {0}, {1:c}', 5.454, $dt)
        );
        $this->assertEquals(
            'deneme 5.454, 03.05.2020 saat 20:39:28',
            Str::format('deneme {0}, {1:d.m.Y \s\a\a\t H:i:s}', 5.454, $dt)
        );
    }

    #endregion Str::format

    #region Str::convertEncoding

    public function testConvertEncoding(){
        $file = __DIR__. DIRECTORY_SEPARATOR ."unicode.txt";
        $dt = fopen($file, "rb");
         fread($dt, 2);
        $utf16 = fread($dt, filesize($file));
        fclose($dt);

       //$utf16 = file_get_contents('unicode.txt');
        $utf8 = 'Atakan Çalışkan';

       $this->assertNotEquals(
           $utf8,
           $utf16
       );

        $this->assertEquals(
            $utf8,
            Str::convertEncoding($utf16,'UTF-8','UNICODE')
        );
        $this->assertEquals(
            $utf8,
            Str::convertEncoding($utf16,null,'UNICODE')
        );

        $this->assertEquals(
            $utf8,
            Str::convertEncoding($utf16,null,null)
        );

        $this->assertEquals(
            $utf16,
            Str::convertEncoding($utf8,'UNICODE','UTF-8')
        );


    }

    #endregion Str::convertEncoding

    #region Str::unicodeToEscapedUnicode

    public function testUnicodeToEscapedUnicode(){
        $file = __DIR__. DIRECTORY_SEPARATOR ."unicode.txt";
        $dt = fopen($file, "rb");
        fread($dt, 2);
        $uni = fread($dt, filesize($file));
        fclose($dt);

        $this->assertEquals(
            "\u{0041}\u{0074}\u{0061}\u{006b}\u{0061}\u{006e}\u{0020}\u{00c7}\u{0061}\u{006c}\u{0131}\u{015f}\u{006b}\u{0061}\u{006e}",
            Str::unicodeToEscapedUnicode($uni)
        );
        $this->assertEquals(
            "\\u{0041}\\u{0074}\\u{0061}\\u{006b}\\u{0061}\\u{006e}\\u{0020}\\u{00c7}\\u{0061}\\u{006c}\\u{0131}\\u{015f}\\u{006b}\\u{0061}\\u{006e}",
            Str::unicodeToEscapedUnicode($uni,false)
        );

        $uni = Str::convertEncoding("Atakan Çalışkan","UNICODE");

        $this->assertEquals(
            "\u{0041}\u{0074}\u{0061}\u{006b}\u{0061}\u{006e}\u{0020}\u{00c7}\u{0061}\u{006c}\u{0131}\u{015f}\u{006b}\u{0061}\u{006e}",
            Str::unicodeToEscapedUnicode($uni)
        );
        $this->assertEquals(
            "\\u{0041}\\u{0074}\\u{0061}\\u{006b}\\u{0061}\\u{006e}\\u{0020}\\u{00c7}\\u{0061}\\u{006c}\\u{0131}\\u{015f}\\u{006b}\\u{0061}\\u{006e}",
            Str::unicodeToEscapedUnicode($uni,false)
        );


    }

    #endregion Str::unicodeToEscapedUnicode

    #region Str::ensureMaximumLength

    public function testEnsureMaximumLengthBasic(): void
    {

        $this->assertEquals(
            'Atakan Çalışkan',
            Str::ensureMaximumLength('Atakan Çalışkan dşği asdasdasd', 15)
        );
        $this->assertEquals(
            '国推震山記密企退受及以屈内新士選',
            Str::ensureMaximumLength('国推震山記密企退受及以屈内新士選。語記末正作酵際滴田掲組宿', 16)
        );

    }

    public function testEnsureMaximumLengthMarker(): void
    {

        $this->assertEquals(
            'Atakan Çalışkan...',
            Str::ensureMaximumLength('Atakan Çalışkan dşği asdasdasd', 15,"...")
        );
        $this->assertEquals(
            '国推震山記密企退受及以屈内新士選 ->',
            Str::ensureMaximumLength('国推震山記密企退受及以屈内新士選。語記末正作酵際滴田掲組宿', 16, ' ->')
        );

    }

    public function testEnsureMaximumLengthEncoding(): void
    {

        $this->assertEquals(
            'Atakan ?al??kan',
            Str::ensureMaximumLength('Atakan Çalışkan dşği asdasdasd', 15,null,'ASCII')
        );

        $this->assertEquals(
            implode("",
            [
                hex2bin('0041'), // UICODE  A
                hex2bin('0074'), // UICODE  t
                hex2bin('0061'), // UICODE  a
                hex2bin('006b'), // UICODE  k
                hex2bin('0061'), // UICODE  a
                hex2bin('006e'), // UICODE  n
                hex2bin('0020'), // UICODE
                hex2bin('00c7'), // UICODE  Ç
                hex2bin('0061'), // UICODE  a
                hex2bin('006c'), // UICODE  l
                hex2bin('0131'), // UICODE  ı
                hex2bin('015f'), // UICODE  ş
                hex2bin('006b'), // UICODE  k
                hex2bin('0061'), // UICODE  a
                hex2bin('006e'), // UICODE  n
                hex2bin('002e'), // UICODE  .
                hex2bin('002e'), // UICODE  .
                hex2bin('002e'), // UICODE  .

            ]),
            Str::ensureMaximumLength('Atakan Çalışkan dşği asdasdasd', 15,"...",'UNICODE')
        );


    }

    #endregion Str::ensureMaximumLength

    #region Str::equals

    public function testEquals(){
        $text = "Atakan Çalışkan";
        $uni = Str::convertEncoding($text,'UNICODE');
        $utf7 = Str::convertEncoding($text,'UTF-7');

        $this->assertTrue(
            "200" == "2e2"
        );

        $this->assertFalse(
            Str::equals("200", "2e2")
        );

        $this->assertTrue(
            Str::equals("", null)
        );

        $this->assertTrue(
            Str::equals("", "")
        );
        $this->assertTrue(
            Str::equals("", "",true,true)
        );

        $this->assertTrue(
            Str::equals(null, null)
        );
        $this->assertTrue(
            Str::equals(null, null,true,true)
        );

        $this->assertFalse(
            Str::equals("aa", null)
        );
        $this->assertFalse(
            Str::equals("aa", null,true,true)
        );

        $this->assertTrue(
            Str::equals("12", 12)
        );
        $this->assertFalse(
            Str::equals("12", 12,false,true)
        );

        $this->assertFalse(
            Str::equals(12, "12",false,true)
        );


        $this->assertFalse(
            Str::equals("", null,false,true)
        );

        $this->assertTrue(
            Str::equals("0", false)
        );

        $this->assertFalse(
            Str::equals("0", false,false,true)
        );


        // ignore case

        $this->assertTrue(
            Str::equals(
                $text, "atakan çalışkan",true
            )
        );

        $this->assertTrue(
            Str::equals(
                $uni, Str::convertEncoding("atakan çalışkan","UNICODE"),true,false,
                "UNICODE"
            )
        );

        $this->assertTrue(
            Str::equals(
                $utf7, Str::convertEncoding("atakan çalışkan","UTF-7"),true,false,
                "UTF-7"
            )
        );

        // language
        $this->assertFalse(
            Str::equals(
                "Işıklı", 'ışıklı',true,true,
            )
        );

        $this->assertTrue(
            Str::equals(
                "Işıklı", 'ışıklı',true,
                true,'UTF-8', 'tr'
            )
        );

    }

    #endregion Str::equals

    #region Str::Len


    public function testLen(){
        $this->assertNotEquals(
            15,
            strlen("Atakan Çalışkan")
        );
        $this->assertEquals(
            15,
            Str::len("Atakan Çalışkan")
        );
        $uni = Str::convertEncoding("Atakan Çalışkan",'UNICODE');
        $this->assertNotEquals(
            15,
            strlen($uni)
        );
        $this->assertNotEquals(
            15,
            Str::len($uni)
        );
        $this->assertEquals(
            15,
            Str::len($uni,"UNICODE")
        );

        $utf7 = Str::convertEncoding("Atakan Çalışkan",'UTF-7');
        $this->assertNotEquals(
            15,
            strlen($utf7)
        );
        $this->assertNotEquals(
            15,
            Str::len($utf7)
        );
        $this->assertEquals(
            15,
            Str::len($utf7,"UTF-7")
        );
        $iso = Str::convertEncoding("Atakan Çalışkan","ISO-8859-15");
        $this->assertEquals(
            15,
            strlen($iso)
        );
        $this->assertNotEquals(
            15,
            Str::len($iso)
        );

        $this->assertEquals(
            15,
            Str::len($iso,"ISO-8859-15")
        );
    }

    #endregion Str::Len

    #region Str::startsWith

    public function testStartsWith(){
        $str = "The quick brown fox jumps over the lazy dog.";
        $this->assertFalse(
            Str::startsWith($str,"quick")
        );
        $this->assertFalse(
            Str::startsWith($str,"")
        );
        $this->assertFalse(
            Str::startsWith($str,null)
        );
        $this->assertTrue(
            Str::startsWith($str,"The quick")
        );
        $this->assertFalse(
            Str::startsWith($str,"the quick")
        );
        $this->assertTrue(
            Str::startsWith($str,"the quick",true)
        );

        $str = Str::convertEncoding($str,"UCS-2");
        $this->assertFalse(
            Str::startsWith($str,Str::convertEncoding("quick","UCS-2"),false,"UCS-2")
        );
        $this->assertTrue(
            Str::startsWith($str,Str::convertEncoding("The quick","UCS-2"),false,"UCS-2")

        );
        $this->assertFalse(
            Str::startsWith($str,Str::convertEncoding("the quick","UCS-2"),false,"UCS-2")

        );
        $this->assertTrue(
            Str::startsWith($str,Str::convertEncoding("the quick","UCS-2"),true,"UCS-2")
        );

        $str = "Işıklı dünya";
        $this->assertFalse(
            Str::startsWith($str,"dün")
        );
        $this->assertTrue(
            Str::startsWith($str,"Işık")
        );
        $this->assertFalse(
            Str::startsWith($str,"ışık")
        );
        $this->assertFalse(
            Str::startsWith($str,"ışık",true)
        );
        $this->assertTrue(
            Str::startsWith($str,"ışık",true,null,"tr")
        );

        $str = Str::convertEncoding($str,"UCS-2");
        $this->assertFalse(
            Str::startsWith($str,Str::convertEncoding("dun","UCS-2"),false,"UCS-2")
        );
        $this->assertTrue(
            Str::startsWith($str,Str::convertEncoding("Işık","UCS-2"),false,"UCS-2")

        );
        $this->assertFalse(
            Str::startsWith($str,Str::convertEncoding("ışık","UCS-2"),false,"UCS-2")

        );


    }

    #endregion Str::startsWith

    #region Str::endsWith

    public function testEndsWith(){
        $str = "The quick brown fox jumps over the lazy dog.";
        $this->assertFalse(
            Str::endsWith($str,"lazy")
        );
        $this->assertFalse(
            Str::endsWith($str,"")
        );
        $this->assertFalse(
            Str::endsWith($str,null)
        );
        $this->assertTrue(
            Str::endsWith($str,"lazy dog.")
        );
        $this->assertFalse(
            Str::endsWith($str,"lAzY Dog.")
        );
        $this->assertTrue(
            Str::endsWith($str,"lAzY Dog.",true)
        );

        $str = Str::convertEncoding($str,"UCS-2");
        $this->assertFalse(
            Str::endsWith($str,Str::convertEncoding("lazy","UCS-2"),false,"UCS-2")
        );
        $this->assertTrue(
            Str::endsWith($str,Str::convertEncoding("lazy dog.","UCS-2"),false,"UCS-2")

        );
        $this->assertFalse(
            Str::endsWith($str,Str::convertEncoding("Lazy Dog.","UCS-2"),false,"UCS-2")

        );
        $this->assertTrue(
            Str::endsWith($str,Str::convertEncoding("Lazy Dog.","UCS-2"),true,"UCS-2")
        );

        $str = "Işıklı dünya ışıklı";
        $this->assertFalse(
            Str::endsWith($str,"dünya")
        );
        $this->assertTrue(
            Str::endsWith($str,"ışıklı")
        );
        $this->assertFalse(
            Str::endsWith($str,"Işıklı")
        );
        $this->assertFalse(
            Str::endsWith($str,"Işıklı",true)
        );
        $this->assertTrue(
            Str::endsWith($str,"Işıklı",true,null,"tr")
        );

        $str = Str::convertEncoding($str,"UCS-2");
        $this->assertFalse(
            Str::endsWith($str,Str::convertEncoding("dun","UCS-2"),false,"UCS-2")
        );
        $this->assertTrue(
            Str::endsWith($str,Str::convertEncoding("ışıklı","UCS-2"),false,"UCS-2")

        );
        $this->assertFalse(
            Str::endsWith($str,Str::convertEncoding("Işıklı","UCS-2"),false,"UCS-2")

        );


    }

    #endregion Str::endsWith

    #region Str::toLowerCase

    public function testToLowerCaseBasic(): void
    {
        $engText = 'This is an English sentence.';
        $this->assertEquals(
            'this is an english sentence.',
            Str::toLowerCase($engText)
        );
        $dummyLangext = 'TYKjkh TYYUYUB JHKDAF';
        $this->assertEquals(
            'tykjkh tyyuyub jhkdaf',
            Str::toLowerCase($dummyLangext)
        );
    }

    public function testToLowerCaseEncoding(): void
    {
        $testText = 'ĞÜŞÇ';
        $this->assertEquals(
            'ğüşç',
            Str::toLowerCase($testText, 'UTF-8')
        );
        $this->assertNotEquals(
            'ğüşç',
            Str::toLowerCase($testText, 'ASCII')
        );
        $this->assertEquals(
            '????????',
            Str::toLowerCase($testText, 'ASCII')
        );

    }

    public function testToLowerCaseLanguage(): void
    {
        $testText = 'Işık Ilık İGİNÇ İgi XWQ';
        $resText = 'ışık ılık iginç igi xwq';
        $resTextAscii = '???k ?l?k igin? igi xwq';

        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,null,'tr')
        );

        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,'UTF-8','tr')
        );
        $this->assertEquals(
            $resTextAscii,
            Str::toLowerCase($testText,'ASCII','tr')
        );

        $testText = 'I IS WQX a i Is';
        $resText = 'i is wqx a i is';

        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,null,'en')
        );

        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,'UTF-8','en')
        );
        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,'ASCII','en')
        );



    }

    #endregion Str::toLowerCase

    #region Str::toUpperCase

    public function testToUpperCaseBasic(): void
    {
        $engText = 'This is an English sentence.';
        $this->assertEquals(
            'THIS IS AN ENGLISH SENTENCE.',
            Str::toUpperCase($engText)
        );
        $dummyLangext = 'TYKjkh df jakl';
        $this->assertEquals(
            'TYKJKH DF JAKL',
            Str::toUpperCase($dummyLangext)
        );
    }

    public function testToUpperCaseEncoding(): void
    {
        $testText = 'ğüşç';
        $this->assertEquals(
            'ĞÜŞÇ',
            Str::toUpperCase($testText, 'UTF-8')
        );
        $this->assertNotEquals(
            'ĞÜŞÇ',
            Str::toUpperCase($testText, 'ASCII')
        );
        $this->assertEquals(
            '????????',
            Str::toUpperCase($testText, 'ASCII')
        );

    }

    public function testToUpperCaseLanguage(): void
    {
        $testText = 'Işık Ilık ilginç İlgi xwq';
        $resText = 'IŞIK ILIK İLGİNÇ İLGİ XWQ';
        $resTextAscii = 'I?IK ILIK ?LG?N? ?LG? XWQ';

        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,null,'tr')
        );

        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,'UTF-8','tr')
        );
        $this->assertEquals(
            $resTextAscii,
            Str::toUpperCase($testText,'ASCII','tr')
        );

        $testText = 'i IS is wqx a i Is';
        $resText = 'I IS IS WQX A I IS';

        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,null,'en')
        );

        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,'UTF-8','en')
        );
        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,'ASCII','en')
        );



    }

    #endregion Str::toUpperCase

    #region Str::toTitleCase

    public function testToTitleCaseBasic(): void
    {
        $engText = 'this is an English sentence.';
        $this->assertEquals(
            'This Is An English Sentence.',
            Str::toTitleCase($engText)
        );
        $dummyLangext = 'tYKjkh df JAkl';
        $this->assertEquals(
            'Tykjkh Df Jakl',
            Str::toTitleCase($dummyLangext)
        );
        $engText = 'this IS An English sentence. this is AnOther Sentence.';
        $this->assertEquals(
            'This Is An English Sentence. This Is Another Sentence.',
            Str::toTitleCase($engText)
        );
        $this->assertEquals(
            '',
            Str::toTitleCase('')
        );
        $this->assertEquals(
            '   
            ',
            Str::toTitleCase('   
            ')
        );
        $this->assertEquals(
            " \t \n",
            Str::toTitleCase(" \t \n")
        );
    }

    public function testToTitleCaseEncoding(): void
    {
        $testText = 'ğüşç';
        $this->assertEquals(
            'Ğüşç',
            Str::toTitleCase($testText, 'UTF-8')
        );
        $this->assertNotEquals(
            'Ğüşç',
            Str::toTitleCase($testText, 'ASCII')
        );
        $this->assertEquals(
            '????????',
            Str::toTitleCase($testText, 'ASCII')
        );

    }

    public function testToTitleCaseLanguage(): void
    {

        $testText = 'bu tÜRkçe bir cümLedİr. deĞİl mi?';
        $resText = 'Bu Türkçe Bir Cümledir. Değil Mi?';
        $resTextAscii = 'Bu T?rk?e Bir C?mledir. De?il Mi?';

        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,null,'tr')
        );

        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,'UTF-8','tr')
        );
        $this->assertEquals(
            $resTextAscii,
            Str::toTitleCase($testText,'ASCII','tr')
        );

        $testText = 'i IS is wqx a i Is';
        $resText = 'I Is Is Wqx A I Is';

        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,null,'en')
        );

        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,'UTF-8','en')
        );
        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,'ASCII','en')
        );

    }

    #endregion Str::toTitleCase

    #region Str::ucFirst

    public function testUcFirst(){
        $this->assertEquals(
            '',
            Str::ucFirst('')
        );
        $this->assertEquals(
            'I love this GaMe.',
            Str::ucFirst('i love this GaMe.')
        );
        $this->assertEquals(
            'I love this GaMe.',
            Str::ucFirst('i love this GaMe.', 'ASCII')
        );
        $this->assertEquals(
            'I love this GaMe.',
            Str::ucFirst('i love this GaMe.', 'ASCII', 'en')
        );
        $this->assertNotEquals(
            'I love this GaMe.',
            Str::ucFirst('i love this GaMe.', 'UTF-8', 'tr')
        );
        $this->assertEquals(
            'Işık ile başlayan bu cümlenin İlk harFi büyük Olacak.',
            Str::ucFirst('ışık ile başlayan bu cümlenin İlk harFi büyük Olacak.',null,'tr')
        );
        $this->assertEquals(
            'İç dış ile başlayan bu cümlenin İlk harFi büyük Olacak.',
            Str::ucFirst('iç dış ile başlayan bu cümlenin İlk harFi büyük Olacak.','UTF-8','tr')
        );
        $this->assertEquals(
            '?? d?? ile ba?layan bu c?mlenin ?lk harFi b?y?k Olacak.',
            Str::ucFirst('iç dış ile başlayan bu cümlenin İlk harFi büyük Olacak.','ASCII','tr')
        );
    }

    #endregion Str::ucFirst

    #region Str::lcFirst

    public function testLcFirst(){
        $this->assertEquals(
            '',
            Str::lcFirst('')
        );
        $this->assertEquals(
            'i love this GaMe.',
            Str::lcFirst('I love this GaMe.')
        );
        $this->assertEquals(
            'i love this GaMe.',
            Str::lcFirst('I love this GaMe.', 'ASCII')
        );
        $this->assertEquals(
            'i love this GaMe.',
            Str::lcFirst('I love this GaMe.', 'ASCII', 'en')
        );
        $this->assertNotEquals(
            'i love this GaMe.',
            Str::lcFirst('I love this GaMe.', 'UTF-8', 'tr')
        );
        $this->assertEquals(
            'ışık ile başlayan bu cümlenin İlk harFi küçük Olacak.',
            Str::lcFirst('Işık ile başlayan bu cümlenin İlk harFi küçük Olacak.',null,'tr')
        );
        $this->assertEquals(
            'iç dış ile başlayan bu cümlenin İlk harFi küçük Olacak.',
            Str::lcFirst('İç dış ile başlayan bu cümlenin İlk harFi küçük Olacak.','UTF-8','tr')
        );
        $this->assertEquals(
            'i? d?? ile ba?layan bu c?mlenin ?lk harFi k???k  Olacak.',
            Str::lcFirst('İç dış ile başlayan bu cümlenin İlk harFi küçük  Olacak.','ASCII','tr')
        );
    }

    #endregion Str::lcFirst

    #region Str:contains

    public function testContains(){
        $str = "The quick Brown fox jumps over the lazy dog.";

        $this->assertTrue(Str::contains($str,"Brown"));
        $this->assertFalse(Str::contains($str,"brown"));
        $this->assertTrue(Str::contains($str,"brown",true));
        $this->assertFalse(Str::contains($str,""));
        $this->assertFalse(Str::contains($str,null));

        $str = "Evlerin ışıkları yanıyor.";
        $this->assertTrue(Str::contains($str,"ışık"));
        $this->assertFalse(Str::contains($str,"IŞIK"));
        $this->assertFalse(Str::contains($str,"IŞIK",true));
        $this->assertTrue(Str::contains($str,"IŞIK",true,null,"tr"));


    }

    public function testContainsIgnoreCaseSensitive(){
        $str = "The quick Brown fox jumps over the lazy dog.";
        $this->assertTrue(Str::contains($str,"Brown",true));
        $this->assertTrue(Str::contains($str,"brown",true));

        // türkçe dil belirtmeden
        $str = "Ha bu ışıklı dünya oldu bize karanlık.";
        $this->assertTrue(Str::contains($str,"ışıklı",true));
        $this->assertFalse(Str::contains($str,"IŞIKLI",true));
        // türkçe dil belirterek
        $this->assertTrue(Str::contains($str,"ışıklı",true,null,'tr'));
        $this->assertTrue(Str::contains($str,"IŞIKLI",true,null,'tr'));


    }

    #endregion Str:contains

    #region Str:containsCount

    public function testContainsCount(){
        $str = "The quick Brown fox jumps over the lazy dog.O";

        $this->assertEquals(1,Str::containsCount($str,"The"));
        $this->assertEquals(1,Str::containsCount($str,"the"));
        $this->assertEquals(4,Str::containsCount($str,"o"));
        $this->assertEquals(1,Str::containsCount($str,"O"));

        $this->assertEquals(00,Str::containsCount($str,""));
        $this->assertEquals(00,Str::containsCount($str,null));

    }

    public function testContainsCountIgnoreCaseSensitive(){
        $str = "The quick Brown fox jumps over the lazy dog.O";

        $this->assertEquals(2,Str::containsCount($str,"The",true));
        $this->assertEquals(2,Str::containsCount($str,"the",true));
        $this->assertEquals(5,Str::containsCount($str,"o",true));
        $this->assertEquals(5,Str::containsCount($str,"O",true));

        // türkçe dil belirtmeden
        $str = "Işıksız ışıldak mı olur? Işıldak dediğin ışık saçar.";
        $this->assertEquals(2,Str::containsCount($str,"ışı",true));
        $this->assertEquals(2,Str::containsCount($str,"Işı",true));

        // türkçe dil belirterek
        $this->assertEquals(4,Str::containsCount($str,"ışı",true,null,'tr'));
        $this->assertEquals(4,Str::containsCount($str,"Işı",true,null,'tr'));


    }

    #endregion Str:containsCount

    #region Str::checkEncoding

    public function testCheckEncoding(){
        $text = "Atakan Çalışkan";
        $this->assertFalse(
            Str::checkEncoding($text,"ASCII")
        );
        $this->assertTrue(
            Str::checkEncoding(Str::convertEncoding($text,'ASCII'),"ASCII")
        );

        $text = "Atakan Çalışkan";
        $this->assertFalse(
            Str::checkEncoding($text,"EUC-JP")
        );
        $this->assertTrue(
            Str::checkEncoding(Str::convertEncoding($text,'EUC-JP'),"EUC-JP")
        );
    }

    #endregion Str::checkEncoding

    #region Str::toArray

    public function testToArray(){
        $test = "Işıklı";
        $this->assertEquals(
            ["I","ş","ı","k","l","ı"],
            Str::toArray($test)
        );
        $this->assertEquals(
            ["Işı","klı"],
            Str::toArray($test,null,3)
        );
        $this->assertEquals(
            ["Işık","lı"],
            Str::toArray($test,null,4)
        );

        $test = Str::convertEncoding("Işıklı","UCS-2");
        $this->assertEquals(
            ["\u{0049}","\u{015f}","\u{0131}","\u{006b}","\u{006c}","\u{0131}"],
            Str::toArray(Str::unicodeToEscapedUnicode($test))
        );
        $this->assertEquals(
            [
                hex2bin("0049"),
                hex2bin("015f"),
                hex2bin("0131"),
                hex2bin("006b"),
                hex2bin("006c"),
                hex2bin("0131"),
            ],
            Str::toArray($test,"UCS-2")
        );

    }

    #endregion Str::toArray

    #region Str::indexOf

    public function testIndexOf(){
        $test = "Atakan Çalışkan";
        $this->assertEquals(
            1,
            Str::indexOf($test,"t")
        );
        $this->assertEquals(
            false,
            Str::indexOf($test,"T")
        );
        $this->assertEquals(
            1,
            Str::indexOf($test,"T",true)
        );
        $this->assertEquals(
            10,
            Str::indexOf($test,"ı")
        );
        $this->assertEquals(
            false,
            Str::indexOf($test,"I")
        );
        $this->assertEquals(
            false,
            Str::indexOf($test,"I",true)
        );
        $this->assertEquals(
            10,
            Str::indexOf($test,"I",true,"UTF-8","tr")
        );
        $test = Str::convertEncoding($test,"UCS-2");
        $this->assertEquals(
            10,
            Str::indexOf($test,Str::convertEncoding("ı","UCS-2"),false,"UCS-2")
        );


    }

    #endregion Str::indexOf

    #region Str::lastIndexOf

    public function testLastIndexOf(){
        $test = "Atakan Çalışkan";
        $this->assertEquals(
            13,
            Str::lastIndexOf($test,"a")
        );
        $this->assertEquals(
            false,
            Str::lastIndexOf($test,"K")
        );
        $this->assertEquals(
            12,
            Str::lastIndexOf($test,"K",true)
        );
        $this->assertEquals(
            10,
            Str::lastIndexOf($test,"ı")
        );
        $this->assertEquals(
            false,
            Str::lastIndexOf($test,"I")
        );
        $this->assertEquals(
            false,
            Str::lastIndexOf($test,"I",true)
        );
        $this->assertEquals(
            10,
            Str::lastIndexOf($test,"I",true,"UTF-8","tr")
        );
        $test = Str::convertEncoding($test,"UCS-2");
        $this->assertEquals(
            13,
            Str::lastIndexOf($test,Str::convertEncoding("A","UCS-2"),true,"UCS-2")
        );
        $this->assertEquals(
            0,
            Str::lastIndexOf($test,Str::convertEncoding("A","UCS-2"),false,"UCS-2")
        );


    }

    #endregion Str::lastIndexOf

    #region Str::DetectEncoding

    public function testDetectEncoding(){

        $utf16 = Str::convertEncoding("Atakan Çalışkan","UCS-2");

        $this->assertEquals(
            "UCS-2",
            Str::detectEncoding($utf16,"UCS-2,UTF-7,UTF-8,ASCII")
        );
        $this->assertEquals(
            "UTF-7",
            Str::detectEncoding(Str::convertEncoding("Atakan","UTF-7"),"UCS-2,UTF-7,UTF-8,ASCII")
        );
        $this->assertEquals(
            "UTF-8",
            Str::detectEncoding("Atakan Çalışkan","UCS-2,UTF-7,UTF-8,ASCII")
        );
        $this->assertEquals(
            "UTF-8",
            Str::detectEncoding("Atakan Çalışkan","UCS-2,ASCII,UTF-7,UTF-8")
        );

        $this->assertEquals(
            "ASCII",
            Str::detectEncoding("Ata","UCS-2,ASCII,UTF-7,UTF-8")
        );
    }

    #endregion Str::DetectEncoding

    #region Str::fixSpaces

    public function testFixSpaces(){
        $str = "Atakan \n Çalışkan\t\tdeneme  123";
        $this->assertEquals(
            "Atakan Çalışkan deneme 123",
            Str::fixSpaces($str)
        );
        $this->assertEquals(
            "Atakan   Çalışkan  deneme  123",
            Str::fixSpaces($str," ",false)
        );
        $this->assertEquals(
            "Atakan \n Çalışkan deneme 123",
            Str::fixSpaces($str," ",true," \t\r\0\x0B")
        );
        $this->assertEquals(
            "Atakan-Çalışkan-deneme-123",
            Str::fixSpaces($str, "-")
        );
    }

    #endregion Str::fixSpaces



}
