<?php /** @noinspection PhpUnhandledExceptionInspection */

namespace Gek\Infrastructure\Tests\Math;

use Exception;
use Gek\Infrastructure\IComparable;
use Gek\Infrastructure\Math\Decimal;
use PHPUnit\Framework\TestCase;

class DecimalTest extends TestCase
{

    public function testGetValue(){
        $d = new Decimal();
        $this->assertEquals("0.000000",$d->getValue());

        $d = new Decimal(12.4,3);
        $this->assertEquals("12.400",$d->getValue());

        $d = new Decimal(0.15,6);
        $this->assertEquals("0.150000",$d->getValue());

    }

    public function testGetScale(){

        $d = new Decimal();
        $this->assertEquals(6,$d->getScale());

        $d = new Decimal(12.4,3);
        $this->assertEquals(3,$d->getScale());

        $d = new Decimal(0.15,6);
        $this->assertEquals(6,$d->getScale());

    }

    public function testGetDecPoint(){

        $d = new Decimal();
        $this->assertEquals(".",$d->getDecPoint());

        $d = new Decimal("12,4",3,",");
        $this->assertEquals(",",$d->getDecPoint());

        $d = new Decimal(0.15,6,".");
        $this->assertEquals(".",$d->getDecPoint());

    }

    public function testGetThousandsSep(){

        $d = new Decimal();
        $this->assertEquals(",",$d->getThousandsSep());

        $d = new Decimal("12,4",3,",",".");
        $this->assertEquals(".",$d->getThousandsSep());

        $d = new Decimal(0.15,6,".");
        $this->assertEquals(",",$d->getThousandsSep());

    }

    public function testSerialize(){
        $dec = new Decimal("15665.465",4);
        $s1 =  serialize($dec);
        $unSerDec = unserialize($s1);

        $this->assertEquals(
            $dec,
            $unSerDec
        );

        $this->assertEquals(
            $s1,
            serialize($unSerDec)
        );

    }

    public function testJsonSerialize(){
        $dec = new Decimal("16.5",2);
        $this->assertEquals(
            "\"16.50\"",
            json_encode($dec)
        );

        $testObj = (object)[
            'sayi' => $dec,
            'sayi2' => $dec->add(10.2),
        ];
        $this->assertEquals(
            "{\"sayi\":\"16.50\",\"sayi2\":\"26.70\"}",
            json_encode($testObj)
        );
        $testArr = [
            $dec,
            $dec->add(10.2),
            $dec->mul(2),
            $dec->div(1.5)
        ];
        $this->assertEquals(
            "[\"16.50\",\"26.70\",\"33.00\",\"11.00\"]",
            json_encode($testArr)
        );
    }

    public function testCtor(){
        $sampleDec = new Decimal(1);
        $this->assertInstanceOf(Decimal::class,$sampleDec);
        $this->assertEquals("1.000000",$sampleDec->getValue());
        $sampleDec = new Decimal(1,4);
        $this->assertEquals("1.0000",$sampleDec->getValue());
        $sampleDec = new Decimal(1,4,",");
        $this->assertEquals("1.0000",$sampleDec->getValue());

        $sampleDec = new Decimal("10,000,5858789",6,",",".");
        $this->assertEquals("10000.585879",$sampleDec->getValue());

        $this->expectDeprecationMessage('$value gecerli bir sayisal deger icermiyor.');
        /** @noinspection PhpUnusedLocalVariableInspection */
        $dec = new Decimal("atakan");
    }

    public function testToFloat(){
        $dec = new Decimal("12.1",3);
        $this->assertEquals(
            12.100,
                $dec->toFloat()
        );

        $dec = new Decimal("-12.1",3);
        $this->assertEquals(
            -12.100,
            $dec->toFloat()
        );



        $dec = new Decimal("12.1124",3);
        $this->assertEquals(
            12.112,
            $dec->toFloat()
        );

        $dec = new Decimal("-12.1124",3);
        $this->assertEquals(
            -12.112,
            $dec->toFloat()
        );

        $dec = new Decimal("12.1155",3);
        $this->assertEquals(
            12.116,
            $dec->toFloat()
        );

        $dec = new Decimal("-12.1155",3);
        $this->assertEquals(
            -12.116,
            $dec->toFloat()
        );

        $dec = new Decimal("12.1154",3);
        $this->assertEquals(
            12.115,
            $dec->toFloat()
        );

        $dec = new Decimal("-12.1154",3);
        $this->assertEquals(
            -12.115,
            $dec->toFloat()
        );

        $dec = new Decimal("12.1154",4);
        $this->assertEquals(
            12.1154,
            $dec->toFloat()
        );
        $dec = new Decimal("12.1154",0);
        $this->assertEquals(
            12,
            $dec->toFloat()
        );
        $dec = new Decimal("12.7154",0);
        $this->assertEquals(
            13,
            $dec->toFloat()
        );

        $dec = new Decimal(0,0);
        $this->assertEquals(
            0,
            $dec->toFloat()
        );
        $dec = new Decimal("-0.12",1);
        $this->assertEquals(
            -0.1,
            $dec->toFloat()
        );
        $dec = new Decimal("-0.00",1);
        $this->assertEquals(
            0,
            $dec->toFloat()
        );

    }

    public function testToString(){
        $dec = new Decimal();
        $this->assertEquals(
            "0.000000",
            $dec->__toString()
        );
        $this->assertEquals(
            "0.000000",
            $dec .""
        );
        $dec = new Decimal(12);
        $this->assertEquals(
            "12.000000",
            $dec->__toString()
        );
        $this->assertEquals(
            "12.000000",
            $dec .""
        );

        $dec = new Decimal(1245457.7856786);
        $this->assertEquals(
            "1245457.785679",
            $dec->__toString()
        );
        $this->assertEquals(
            "1245457.785679",
            $dec .""
        );

        $this->assertEquals(
            "1,245,457.785679",
            $dec->toHumanizeString()
        );
        $this->assertEquals(
            "1,245,457.786",
            $dec->toHumanizeString(3)
        );

        $dec = new Decimal(1245457.7856786,3,",",".");
        $this->assertEquals(
            "1245457.786",
            $dec->__toString()
        );
        $this->assertEquals(
            "1245457.786",
            $dec .""
        );

        $this->assertEquals(
            "1.245.457,786",
            $dec->toHumanizeString()
        );
        $this->assertEquals(
            "1.245.457,786000",
            $dec->toHumanizeString(6)
        );

        $dec = new Decimal(-1245457.7856786,3,",",".");
        $this->assertEquals(
            "-1245457.786",
            $dec->__toString()
        );
        $this->assertEquals(
            "-1245457.786",
            $dec .""
        );

        $this->assertEquals(
            "-1.245.457,786",
            $dec->toHumanizeString()
        );
        $this->assertEquals(
            "-1.245.457,79",
            $dec->toHumanizeString(2)
        );



    }

    public function testAdd(){
        $dec = new Decimal("10.752",4);
        $this->assertEquals(
            11.7632,
            $dec->add(1.0112)->toFloat()
        );

        $this->assertEquals(
            10.6999,
            $dec->add(-0.0521)->toFloat()
        );

        $other = new Decimal("1.01126",5);
        $this->assertEquals(
            11.7633,
            $dec->add($other)->toFloat()
        );
        $this->expectExceptionMessage('$other gecersiz bir turde.');
        $dec->add("atakan");
    }

    public function testCompare(){
        $dec = new Decimal("12.7845",3);
        $this->assertEquals(
            0,
            $dec->compare(12.785)
        );
        $this->assertEquals(
            -1,
            $dec->compare(12.786)
        );
        $this->assertEquals(
            1,
            $dec->compare(12.783)
        );
        $this->assertTrue(
            $dec->isGreaterThan(11)
        );
        $this->assertTrue(
            $dec->isGreaterThan("12.782")
        );
        $this->assertFalse(
            $dec->isGreaterThan("12.785")
        );

        $this->assertTrue(
            $dec->isGreaterOrEqualThan("12.785")
        );
        $this->assertTrue(
            $dec->isGreaterOrEqualThan(11)
        );
        $this->assertTrue(
            $dec->isGreaterOrEqualThan("12.782")
        );


        //-*****************
        $this->assertTrue(
            $dec->isLessThan(27)
        );
        $this->assertTrue(
            $dec->isLessThan("12.786")
        );
        $this->assertFalse(
            $dec->isLessThan("12.785")
        );

        $this->assertTrue(
            $dec->isLessOrEqualThan("12.785")
        );
        $this->assertTrue(
            $dec->isLessOrEqualThan(18)
        );
        $this->assertTrue(
            $dec->isLessOrEqualThan("12.786")
        );



    }

    public function testEquals(){
        $dec = new Decimal("12.7845",3);
        $this->assertEquals(
            true,
            $dec->equals(12.785)
        );
        $this->assertEquals(
            false,
            $dec->equals(12.786)
        );
        $this->assertEquals(
            false,
            $dec->equals(12.783)
        );

    }

    public function testDiv(){
        $dec = new Decimal(1250.7547,4);
        $this->assertEquals(
            1059.9616,
            $dec->div(1.18)->toFloat()
        );
        $dec = new Decimal(1250.7547,4);
        $this->assertEquals(
            1058.8847,
            $dec->div("1.1812")->toFloat()
        );

    }

    public function testMod(){
        $dec = new Decimal("10.10",2);
        $this->assertEquals(
            0,
            $dec->mod(1.01)->toFloat()
        );
    }

    public function testMul(){
        $dec = new Decimal(1250.7547,4);
        $this->assertEquals(
            1475.8905,
            $dec->mul(1.18)->toFloat()
        );
        $dec = new Decimal(1250.7547,4);
        $this->assertEquals(
            1477.3914,
            $dec->mul(1.18118)->toFloat()
        );

    }


    public function testPow(){
        $dec = new Decimal("1.01",2);
        $this->assertEquals(
            1.10,
            $dec->pow(10)->toFloat()
        );
        $dec = new Decimal("2",2);
        $this->assertEquals(
            256.00,
            $dec->pow(8)->toFloat()
        );
    }

    public function testSub(){
        $dec = new Decimal("1.4576",4);
        $this->assertEquals(
            1.4575,
            $dec->sub(0.00008)->toFloat()
        );
        $this->assertEquals(
            -0.5425,
            $dec->sub(2.00008)->toFloat()
        );

        $this->assertEquals(
            2.0,
            $dec->sub(-0.5424)->toFloat()
        );
        $other = new Decimal("-0.5424");
        $this->assertEquals(
            2.0,
            $dec->sub($other)->toFloat()
        );

    }

    public function testSqrt(){
        $dec = new Decimal(48454.618618);
        $this->assertEquals(
            220.124098,
            $dec->sqrt()->toFloat()
        );
    }

    public function testRound(){
        $dec = new Decimal(2.55);
        $this->assertEquals(
            3,
            $dec->round()
        );
        $this->assertEquals(
            2.6,
            $dec->round(1)
        );
    }

    public function testFloor(){
        $dec = new Decimal(2.55);
        $this->assertEquals(
            2,
            $dec->floor()
        );
        $dec = new Decimal(3.999999);
        $this->assertEquals(
            3,
            $dec->floor()
        );
    }

    public function testCeil(){
        $dec = new Decimal(2.35);
        $this->assertEquals(
            3,
            $dec->ceil()
        );
        $dec = new Decimal(3.11111);
        $this->assertEquals(
            4,
            $dec->ceil()
        );
    }

    public function testWrap(){
        $dec1 = new Decimal('1.4566');
        $dec2 = Decimal::wrap('1.4566');
        $this->assertEquals(
            $dec1,
            $dec2
        );
        /** @noinspection PhpNonStrictObjectEqualityInspection */
        $this->assertTrue(
            $dec1 == $dec2
        );
        $this->assertFalse(
            $dec1 === $dec2
        );

        $dec1 = new Decimal('1.4566',3);
        $dec2 = Decimal::wrap('1.4566',3);
        $this->assertEquals(
            $dec1,
            $dec2
        );
        $this->assertEquals(
            $dec1->getValue(),
            $dec2->getValue()
        );
        $this->assertEquals(
            $dec1->toFloat(),
            $dec2->toFloat()
        );

    }

    public function testExp(){
        $res = Decimal::exp("1.12");
        $this->assertEquals(
            "1.120000",
            $res->getValue()
        );

        $res = Decimal::exp("1.12",3);
        $this->assertEquals(
            "1.120",
            $res->getValue()
        );
        $res = Decimal::exp("-1.12",3);
        $this->assertEquals(
            "-1.120",
            $res->getValue()
        );
        $res = Decimal::exp("-1.12 + 1.0",3);
        $this->assertEquals(
            "-0.120",
            $res->getValue()
        );

        $res = Decimal::exp("(10.0 + 10.0) * 2",3);
        $this->assertEquals(
            "40.000",
            $res->getValue()
        );

        $res = Decimal::exp("10.0 + (10.0 * 2)",3);
        $this->assertEquals(
            "30.000",
            $res->getValue()
        );

        $res = Decimal::exp("10.0 + 10.0 * 2",3);
        $this->assertEquals(
            "30.000",
            $res->getValue()
        );
        $res = Decimal::exp("2 * 10.0 + 10.0",3);
        $this->assertEquals(
            "40.000",
            $res->getValue()
        );

        $res = Decimal::exp("((2 * 10.0) + 10.0) - (45.00 / 3.0)",3);
        $this->assertEquals(
            "15.000",
            $res->getValue()
        );

        $res = Decimal::exp("11.00 % 2.00",3);
        $this->assertEquals(
            "1.000",
            $res->getValue()
        );



        $this->expectException(Exception::class);

        /** @noinspection PhpUnusedLocalVariableInspection */
        $res = Decimal::exp("iki kere iki",3);


    }

    public function testExpF(){
        $res = Decimal::expF("1.12");
        $this->assertEquals(
            1.120000,
            $res
        );

        $res = Decimal::expF("1.12",3);
        $this->assertEquals(
            1.120,
            $res
        );
        $res = Decimal::expF("-1.12",3);
        $this->assertEquals(
            -1.120,
            $res
        );
        $res = Decimal::expF("-1.12 + 1.0",3);
        $this->assertEquals(
            -0.120,
            $res
        );

        $res = Decimal::expF("(10.0 + 10.0) * 2",3);
        $this->assertEquals(
            40.000,
            $res
        );

        $res = Decimal::expF("10.0 + (10.0 * 2)",3);
        $this->assertEquals(
            30.000,
            $res
        );

        $res = Decimal::expF("10.0 + 10.0 * 2",3);
        $this->assertEquals(
            30.000,
            $res
        );
        $res = Decimal::expF("2 * 10.0 + 10.0",3);
        $this->assertEquals(
            40.000,
            $res
        );


        $res = Decimal::expF("((2 * 10.0)) + (10.0)",3);
        $this->assertEquals(
            30.000,
            $res
        );





        $this->expectException(Exception::class);

        /** @noinspection PhpUnusedLocalVariableInspection */
        $res = Decimal::expF("3 * 2 5",3);



    }

    public function testFloatingPointCompare(){
        $floatCalc = floor((0.1 + 0.7) * 10); // 7
        $decimalCalc = floor(Decimal::expF("(0.1 + 0.7) * 10")); // 8

        $this->assertNotEquals(
            8,
            $floatCalc
        );
        $this->assertNotEquals(
            $floatCalc,
            $decimalCalc
        );
        $this->assertEquals(
            8,
            $decimalCalc
        );
    }

    public function testCompareTo(){

        $dec = new Decimal("12.00",2);

        $this->assertInstanceOf(
            IComparable::class,
            $dec
        );

        $testArr = [
            new Decimal("15.20",2),
            new Decimal("9.10",2),
            new Decimal("12.30",2),
            new Decimal("85.50",2),
            new Decimal("-12.30",2),
            new Decimal("-2.80",2),
            new Decimal("3.80",2),
        ];

        $cmp = function(Decimal $a, Decimal $b)
        {
            return $a->compareTo($b);
        };

        usort($testArr,$cmp);

        $this->assertEquals(
            "-12.30",
            $testArr[0]->getValue()
        );
        $this->assertEquals(
            "85.50",
            $testArr[count($testArr) - 1]->getValue()
        );




    }

}
