<?php /** @noinspection PhpUnhandledExceptionInspection */


namespace Gek\Infrastructure\Math;


use Exception;
use Gek\Infrastructure\Exceptions\GekException;
use Gek\Infrastructure\IComparable;
use Gek\Infrastructure\IEquatable;
use Gek\Infrastructure\IToFloat;
use Gek\Infrastructure\Math\DecimalExpressions\DecimalExpressionParser;
use Gek\Infrastructure\Str;
use JsonSerializable;
use Serializable;

/**
 * Decimal Sınıfı
 *
 * Decimal türünü simule eder
 * @package Gek\Infrastructure\Math
 * @since 1.5.0
 * @see https://www.php.net/manual/en/book.bc.php BCMath php eklentisi
 */
class Decimal implements IComparable, IToFloat, IEquatable, Serializable, JsonSerializable
{

    #region fields

    /**
     * ondalık kısım basamak sayısı
     *
     * @var int
     */
    protected int $scale = 6;

    /**
     * deper
     *
     * @var string
     */
    protected string $value;

    /**
     * ondalık ayracı
     *
     * @var string
     */
    protected string $decPoint = ".";


    /**
     * binlik ayracı
     *
     * @var string
     */
    protected string $thousandsSep = ",";

    #endregion fields

    #region ctor

    /**
     * Decimal yapıcı metod.
     *
     * @param float|int|string $value değer
     * @param int $scale ondalık basamak hassasiyeti
     * @param string $decPoint ondalık ayracı
     * @param string $thousandsSep binlik ayracı
     * @throws Exception
     */
    public function __construct($value = 0.0, int $scale = 6, string $decPoint = '.', string $thousandsSep = ",")
    {
        $this->scale = $scale;
        $this->decPoint = $decPoint;
        $this->thousandsSep = $thousandsSep;

        if (is_string($value)) {
            if ($decPoint != '.') {
                $value = str_replace($decPoint, '.', $value);
            }
            if (Str::containsCount($value, '.') > 1) {
                $pointIndex = Str::lastIndexOf($value, '.');
                $dec = mb_substr($value, $pointIndex);
                $int = mb_substr($value, 0, $pointIndex);
                $value = str_replace('.', '', $int) . '.'
                    . str_replace('.', '', $dec);
            }

            if (is_numeric($value)) {
                $value  = floatval($value);
            }
        }

        if(is_float($value) || is_int($value)) {
            $value = $this->numberToString($value);
        } else {
            throw new Exception('$value gecerli bir sayisal deger icermiyor. $value: ' . $value);
        }
        $this->value = $value;

    }

    #endregion ctor

    #region properties

    /**
     * ondalık basamak hassasiyetini verir.
     *
     * @return int
     */
    public function getScale(): int
    {
        return $this->scale;
    }

    /**
     * ondalık ayracını verir
     *
     * @return string
     */
    public function getDecPoint(): string
    {
        return $this->decPoint;
    }

    /**
     * Binlik ayracını verir
     *
     * @return string
     */
    public function getThousandsSep(): string
    {
        return $this->thousandsSep;
    }

    /**
     * değeri verir
     *
     * @return string
     */
    public function getValue(): string
    {
        return $this->value;
    }

    #endregion properties

    #region utils

    /**
     * Float değeri stringe çevirir.
     *
     * @param float $number
     * @return string
     */
    protected function numberToString(float $number): string
    {
        //$number = round($number,$this->scale);
        return number_format($number, $this->scale, '.', '');
    }

    /**
     * İşlem için verilen diğer değeri işlem için hazırlar
     *
     * @param Decimal|string|float|int $other değer
     * @return string düxeltilmiş değer
     * @throws Exception
     */
    protected function getOtherValue($other)
    {
        if ($other instanceof Decimal) {
            if ($other->scale > $this->scale) {
                $dc = new self($other->getValue(), $this->scale);
                return $dc->value;
            }
            return $other->value;
        }
        if (is_float($other) || is_int($other)) {
            return $this->numberToString($other);
        }
        if (is_string($other) && is_numeric($other)) {
            //$other = floatval($other);
            return $other;
        }
        throw new Exception('$other gecersiz bir turde.');
    }



    #endregion utils

    #region methods

    #region IComparable

    /**
     * Verilen değerle mevcut değeri karşılaştırır
     *
     * @see \Gek\Infrastructure\IComparable IComparable arayüzü
     * @param mixed $other diğer değer
     * @return int
     * @throws Exception
     */
    public function compareTo($other): int
    {
        return $this->compare($other);
    }

    #endregion IComparable

    #region IToFloat

    /**
     * Mevcut değeri float türünde verir.
     *
     * @see IToFloat  IToFloat Arayüzü
     * @return float
     */
    public function toFloat(): float
    {
        return (float)$this->value;
    }

    #endregion IToFloat

    #region IEquatable

    /**
     * Verilen değer ile eşitlik kontrolu yapar.
     *
     * @see \Gek\Infrastructure\IEquatable IEquatable Arayüzü
     * @param float|int|string|Decimal $other diğer değer
     * @return bool eşitse true değilse false
     * @throws Exception
     */
    public function equals($other): bool
    {
        return $this->compare($other) == 0;
    }

    #endregion IEquatable

    #region Serializable

    /**
     * nesnenin serileştirilmiş halini verir.
     *
     * @see https://www.php.net/manual/en/language.oop5.serialization.php Php serileştirme.
     * @see https://www.php.net/manual/en/class.serializable.php Serializable arayüzü
     * @return string serileştirilmiş veri
     */
    public function serialize()
    {
        return serialize([
            $this->value,
            $this->scale,
            $this->decPoint,
            $this->thousandsSep,
        ]);
    }

    /**
     * serileştirilmiş veriden nesneyi doldurur..
     *
     * @see https://www.php.net/manual/en/language.oop5.serialization.php Php serileştirme.
     * @see https://www.php.net/manual/en/class.serializable.php Serializable arayüzü
     * @param string $serialized serileştirilmiş veri.
     */
    public function unserialize($serialized)
    {
        list(
            $this->value,
            $this->scale,
            $this->decPoint,
            $this->thousandsSep
            ) = unserialize($serialized);
    }

    #endregion Serializable

    #region JsonSerializable

    /**
     * değeri json için serileştirir.
     *
     * @see https://www.php.net/manual/en/class.jsonserializable.php JsonSerializable arayüzü.
     * @return mixed|string
     */
    public function jsonSerialize()
    {
        return $this->value;
    }

    #endregion JsonSerializable

    /**
     * Değerin verilen değerden büyük olup olmadığını kontrol eder.
     *
     * @param Decimal|string|float|int $other Diğer değer.
     * @return bool Büyükse true aksi halde false
     * @throws Exception
     */
    public function isGreaterThan($other){
        $res = $this->compare($other);
        return $res > 0;
    }

    /**
     * Değerin verilen değerden büyük veya eşit olup olmadığını kontrol eder.
     *
     * @param Decimal|string|float|int $other Diğer değer
     * @return bool Büyük veya eşitse true değilse false
     * @throws Exception
     */
    public function isGreaterOrEqualThan($other){
        $res = $this->compare($other);
        return $res >= 0;
    }

    /**
     * Değerin verilen değerden küçük olup olmadığını kontrol eder.
     *
     * @param Decimal|string|float|int $other Diğer deüer.
     * @return bool Küçükse true aksi halde false
     * @throws Exception
     */
    public function isLessThan($other){
        $res = $this->compare($other);
        return $res < 0;
    }

    /**
     * Değerin verilen değerden küçük veya eşit olup olmadığını kontrol eder.
     *
     * @param Decimal|string|float|int $other Diğer değer.
     * @return bool Küçük veya eşitse true aksi halde false
     * @throws Exception
     */
    public function isLessOrEqualThan($other){
        $res = $this->compare($other);
        return $res <= 0;
    }


    /**
     * Toplama işlemi
     *
     * Verilen değeri mevcut değer ile toplar.
     *
     * @param float|int|string|Decimal $other Diğer değer
     * @return Decimal Toplama sonucu
     * @throws Exception
     */
    public function add($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcadd($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * Değeri string olarak verir.
     *
     * @see https://www.php.net/manual/en/language.oop5.magic.php#object.tostring Php __toString sihirli yöntemi
     * @return string Değerin string hali
     */
    public function __toString()
    {
        return number_format($this->value, $this->scale, ".", "");
    }

    /**
     * Değeri insanların okuması için stringe çevirir.
     *
     * @param int|null $decimals Ondalık basamak sayısı
     * @return string Değerin string hali
     */
    public function toHumanizeString(?int $decimals = null): string
    {
        if ($decimals === null) {
            $decimals = $this->scale;
        }
        return number_format($this->value, $decimals, $this->decPoint, $this->thousandsSep);
    }

    /**
     * Verilen değeri mevcut değer ile karşılaştırır
     *
     * @see https://www.php.net/manual/en/function.bccomp.php bccomp fonksiyonu
     * @param float|int|string|Decimal $other Diğer değer
     * @return int Karşılaştırma sonucu
     * @throws Exception
     */
    public function compare($other): int
    {
        $otherVal = $this->getOtherValue($other);
        return bccomp($this->value, $otherVal, 14);
    }

    /**
     * Bölme işlemi
     *
     * Mevcut değeri verilen değer e böler.
     *
     * @see https://www.php.net/manual/en/function.bcdiv.php bcdiv php metodu
     * @param float|int|string|Decimal $other Bölen değer.
     * @return Decimal Sonuç
     * @throws Exception
     */
    public function div($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcdiv($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * Bölme işleminden kalanı bulur
     *
     * @see https://www.php.net/manual/en/function.bcmod.php bcmod php fonksiyonu
     * @param float|int|string|Decimal $other Bölen değer.
     * @return Decimal Kalan
     * @throws Exception
     */
    public function mod($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcmod($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * Çarpma işlemi
     *
     * Mevcut değeri verilen değer ile çarpar.
     *
     * @see https://www.php.net/manual/en/function.bcmul.php bcmul php fonksiyonu
     * @param float|int|string|Decimal $other Çarpan değer
     * @return Decimal İşlem sonucu
     * @throws Exception
     */
    public function mul($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcmul($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * Üs alma işlemi yapar
     *
     * @see https://www.php.net/manual/en/function.bcpow.php bcpow php fonksiyonu
     * @param int $exponent Üs değeri
     * @return Decimal İşlem sonucu
     * @throws Exception
     */
    public function pow(int $exponent)
    {

        $resVal = bcpow($this->value, $exponent, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * Çıkartma işlemi yapar.
     *
     * Mevcut değerden verilen değeri çıkartır.
     *
     * @see https://www.php.net/manual/en/function.bcsub.php  bcsub php fonksiyonu
     * @param float|int|string|Decimal $other Çıkarılacak değer.
     * @return Decimal İşlem sonucu.
     * @throws Exception
     */
    public function sub($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcsub($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * Değerin karekökünü döndürür
     *
     * @see https://www.php.net/manual/en/function.bcsqrt.php bcsqrt php işlevi
     * @return Decimal Karakök
     * @throws Exception
     */
    public function sqrt()
    {
        $resVal = bcsqrt($this->value, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * Değerin yuvarlanmış halini verir.
     *
     * @see https://www.php.net/manual/en/function.round.php round php fonksiyonu
     * @param int $scale Ondalık basamak hassasiyeti
     * @return false|float yuvarlanmış değer
     */
    public function round(int $scale = 0)
    {
        return round($this->toFloat(), $scale);
    }

    /**
     * Değeri aşağı yuvarlar.
     *
     * @see https://www.php.net/manual/en/function.floor.php floor php fonksiyonu
     * @return false|float Yuvarlanmış değer
     */
    public function floor()
    {
        return floor($this->toFloat());
    }

    /**
     * Değeri yukarı yuvarlar.
     *
     * @see https://www.php.net/manual/en/function.ceil.php ceil php fonksiyon
     * @return false|float Yuvarlanmış değer
     */
    public function ceil()
    {
        return ceil($this->toFloat());
    }

    #endregion methods

    #region statics

    /**
     * Verilen değeri Decimal türünde verir.
     *
     * @see \Gek\Infrastructure\Math\Decimal::__construct Decimal yapıcı method
     * @param self|float|int|string $val Değer
     * @param int|null $scale Ondalık basamak hassasiyetş
     * @return static Decimal değer
     * @throws Exception
     */
    public static function wrap($val, ?int $scale = null): self
    {
        if ($scale !== null) {
            return new self($val, $scale);
        }
        return new self($val);
    }

    /**
     * Verilen ifadeyi işler ve sonucu verir.
     *
     * @param string $exp ifade
     * @param int|null $scale ondalık basamak hassasiyeti
     * @return float float sonuç
     * @throws GekException
     */
    public static function expF(string $exp, ?int $scale = null): float
    {

        return static::exp($exp, $scale)->toFloat();
    }

    /**
     * Verilen ifadeyi işler ve sonucu verir.
     *
     * @param string $exp ifade
     * @param int|null $scale ondalık basamak hassasiyeti
     * @return Decimal Decimal değer
     * @throws GekException
     */
    public static function exp(string $exp, ?int $scale = null)
    {

        return DecimalExpressionParser::parseValue($exp, $scale);
    }

    #endregion statics

}