<?php


namespace Gek\Infrastructure;

use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Exceptions\NotImplementedException;
use ReflectionException;

/**
 * EnumType sınıfı
 *
 * Enum yapısını simule eder.
 *
 * @package Gek\Infrastructure\Enums
 * @since 1.2.0
 */
abstract class EnumType extends Enum implements IToInt
{
    /**
     * Enum değeri
     *
     * @var int
     */
    private int $value;

    /**
     * EnumType yapıcı method
     * .
     * @param int|string $value Enum değeri ya da Enum key (adı)
     * @throws InvalidArgumentException değer geçersizse
     * @throws ReflectionException
     */
    public function __construct($value)
    {
        ClassHelper::checkReflectSupport();
        $this->setValue($value);
    }

    /**
     * Enum adından )key= yeni bir nesne örneği oluşturur.
     *
     * @param $name
     * @param $arguments
     * @return mixed
     * @throws NotImplementedException
     * @throws ReflectionException
     */
    public static function __callStatic($name, $arguments)
    {
        if (self::checkEnumKey($name)) {
            $class = get_called_class();
            return new $class($name);
        }
        throw new NotImplementedException();
    }

    /**
     * Enum değerini döndürür
     *
     * @return int Enum değeri
     */
    public function getValue(): int
    {
        return $this->value;
    }

    /**
     * Enum değerini set eder.
     *
     * @param int|string $val Enum değeri ya da Enum key (adı)
     * @throws InvalidArgumentException değer geçersizse
     * @throws ReflectionException
     */
    public function setValue($val): void
    {
        if (is_int($val)) {
            if (!self::checkEnumValue($val)) {
                throw new InvalidArgumentException(
                    Str::format('Set edilmeye çalışılan değer ("{0}") Enum tipinin değerleri arasında bulunamadı.', $val),
                    0,
                    null,
                    '$val'
                );
            }
            $this->value = $val;
        } elseif (is_string($val)) {
            if (!self::checkEnumKey($val)) {
                throw new InvalidArgumentException(
                    Str::format('Set edilmeye çalışılan değer ("{0}") Enum tipinin değerleri arasında bulunamadı.', $val),
                    0,
                    null,
                    '$val'
                );
            }
            $this->value = (int)self::getEnumValue($val);
        } else {
            throw new InvalidArgumentException(
                'Geçersiz parametre tipi. Parametre tipi int veya string olmalı.',
                0,
                null,
                '$val'
            );
        }
    }

    /**
     * Enum adını (key) döndürür.
     *
     * @return string
     * @throws ReflectionException
     */
    public function __toString()
    {
        $values = self::getEnumValues();
        foreach ($values as $key => $val) {
            if ($this->value == $val) {
                return $key;
            }
        }
        return '';
    }

    /**
     * Enum değerini verir.
     *
     * @return int Enum değeri
     */
    public function toInt(): int
    {
        return $this->value;
    }

}
