<?php


namespace Gek\Infrastructure;

use Gek\Infrastructure\Reflections\DocCommentObject;
use ReflectionException;

/**
 * Enum sınıfı
 * @package Gek\Infrastructure\Enums
 */
abstract class Enum
{


    /**
     * 
     * @var array
     */
    private static array $constArray = array();

    /**
     * 
     * @var array
     */
    private static array $docComArray = array();

    /**
     * Enum değerini döndürür.
     *
     * @param string $key enum anahtarı (adı)
     * @return int|null enum değeri
     * @throws ReflectionException
     */
    public static function getEnumValue(string $key): ?int
    {
        $constants = self::getEnumValues();
        return isset($constants[$key]) ? $constants[$key] : null;
    }

    /**
     * Enum değerlerini dizi olarak döndürür. Enum keu (adı) => enum değeri
     *
     * @return array|int[]  enum değerleri
     * @throws ReflectionException
     */
    public static function getEnumValues(): array
    {
        $className = get_called_class();
        return self::getConstantsArray($className);
    }

    /**
     * 
     * @param string $className
     * @return array
     * @throws ReflectionException
     */
    private static function getConstantsArray(string $className): array
    {
        if (!isset(self::$constArray[$className])) {
            $reflect = ClassHelper::reflectClass($className);
            self::$constArray[$className] = $reflect->getConstants();
        }
        return self::$constArray[$className];
    }

    /**
     * Enum key in mevcut olup olmadığını döndürür.
     * @param string $key Enum key (Adı)
     * @return bool Mevcutsa true değilse false
     * @throws ReflectionException
     */
    public static function checkEnumKey(string $key): bool
    {
        $constants = self::getEnumValues();
        return isset($constants[$key]);
    }

    /**
     * Enum değerinin mevcut olup olmadığını döndürür.
     *
     * @param int $value Enum değeri
     * @return bool Değer mevcutsa true değilse false
     * @throws ReflectionException
     */
    public static function checkEnumValue(int $value): bool
    {
        $constants = self::getEnumValues();
        return in_array($value, $constants, true);
    }

    /**
     * Enum sabitlerinin docComment lerinde oluşan bir dizi döndürür.
     *
     * @return array|DocCommentObject[]
     * @throws ReflectionException
     */
    public static function getEnumDocComentObjects(): array
    {
        $className = get_called_class();
        if (!isset(self::$docComArray[$className])) {
            $reflect = ClassHelper::reflectClass($className);
            $reflectConstArray = $reflect->getReflectionConstants();
            $docComArr = array();
            foreach ($reflectConstArray as $refConst) {
                $docComArr[$refConst->getName()] = DocCommentObject::parse($refConst->getDocComment());
            }

            self::$docComArray[$className] = $docComArr;
        }
        return self::$docComArray[$className];

    }


}
