<?php


namespace Gek\Infrastructure;


use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Exceptions\NotImplementedException;
use ReflectionException;

/**
 * ConstClassType Sınıfı
 * @package Gek\Infrastructure\Enums
 * @since 1.4.0
 */
abstract class ConstClassType extends ConstClass
{
    /**
     * sabit değeri
     *
     * 
     * @var mixed
     */
    private $value;

    /**
     * ConstClassType yapıcı method
     * .
     * @param mixed $value değer
     * @throws InvalidArgumentException
     * @throws ReflectionException
     */
    public function __construct($value)
    {
        ClassHelper::checkReflectSupport();
        $this->setValue($value);
    }

    /**
     * sabit adından yeni bir nesne örneği oluşturur.
     *
     * @see https://www.php.net/manual/en/language.oop5.overloading.php#object.callstatic php __callStatic sihirli yöntemi
     * @param $name
     * @param $arguments
     * @return mixed
     * @throws NotImplementedException sabit ismi hatalıysa
     * @throws ReflectionException
     */
    public static function __callStatic($name, $arguments)
    {
        if (self::checkConstKey($name)) {
            $class = get_called_class();
            return new $class($name);
        }
        throw new NotImplementedException();
    }

    /**
     * Nesneni tuttuğu değeri döndürür.
     *
     * @return mixed
     */
    public function getValue()
    {
        return $this->value;
    }

    /**
     * Nesne değerini set eder
     *
     * @param mixed $val değer
     * @throws InvalidArgumentException değer sabit adları veya sabit değerleri içinde bulunmuyorsa
     * @throws ReflectionException
     */
    public function setValue($val): void
    {
        if (!static::checkConstValue($val) && !static::checkConstKey($val)) {
            throw new InvalidArgumentException(
                Str::format('Set edilmeye çalışılan değer ("{0}") {1} tipinin değerleri / anahtarları arasında bulunamadı.', $val, get_called_class()),
                0,
                null,
                '$val'
            );
        }
        if (static::checkConstValue($val)) {
            $this->value = $val;
        } else {
            $this->value = self::getConstValue($val);
        }

    }

    /**
     * Sabit adını  döndürür.
     *
     * @return string Sabit adı
     * @throws ReflectionException
     */
    public function __toString()
    {
        $values = self::getConstValues();
        foreach ($values as $key => $val) {
            if ($this->value === $val) {
                return $key;
            }
        }
        return '';
    }

}