<?php


namespace Gek\Infrastructure;


use Gek\Infrastructure\Reflections\DocCommentObject;
use ReflectionException;

/**
 * ConstClass Sınıfı
 *
 * String değerlerden oluşan sabitler tutar (Enum gibi ama string)
 *
 * @package Gek\Infrastructure\Enums
 * @since 1.4.0
 */
abstract class ConstClass
{

    /**
     * önbellek için sınıf isimlerine göre sabitleri  tutan array
     *
     * 
     * @var array
     */
    private static array $constArray = array();

    /**
     * önbellek için sınıf isimlerine göre sabitlerin docBlock larını tutatn aray
     *
     * 
     * @var array
     */
    private static array $docComArray = array();

    /**
     * Varilen sabit ismine göre sabitin değerini verir
     *
     * @param string $key Sbit ismi
     * @return mixed|null sabit değeri ya da bulunamazsa null döner
     * @throws ReflectionException
     */
    public static function getConstValue(string $key)
    {
        $constants = self::getConstValues();
        return isset($constants[$key]) ? $constants[$key] : null;
    }

    /**
     * Sabitleri dizi olarak göndürür
     *
     * @return array sabitlerin olduğu dizi. sabit adı => sabit değeri
     * @throws ReflectionException
     */
    public static function getConstValues():array
    {
        $className = get_called_class();
        return self::getConstantsArray($className);
    }

    /**
     * Sınıf ismine göre sabitleri dizi olarak döndürür
     *
     * 
     * @param string $className sınıf adı
     * @return array sabitlerin olduğu dizi sabit adı => sabit değeri
     * @throws ReflectionException
     */
    private static function getConstantsArray(string $className): array
    {
        if (!isset(self::$constArray[$className])) {
            $reflect = ClassHelper::reflectClass($className);
            self::$constArray[$className] = $reflect->getConstants();
        }
        return self::$constArray[$className];
    }

    /**
     * Sabit adının olup olmadığını kontrol eder
     *
     * @param string $key
     * @return bool
     * @throws ReflectionException
     */
    public static function checkConstKey(string $key): bool
    {
        $constants = self::getConstValues();
        return isset($constants[$key]);
    }

    /**
     * Sabit değerinin olup olmadığını kontrol eder.
     *
     * @param mixed $value
     * @return bool
     * @throws ReflectionException
     */
    public static function checkConstValue($value): bool
    {
        $constants = self::getConstValues();
        return in_array($value, $constants, true);
    }

    /**
     * Sabitlerin DocCommentlerini dizi olarak verir.
     *
     * @return array<string,DocCommentObject>|DocCommentObject[]
     * @throws ReflectionException
     */
    public static function getConstDocCommentObjects(): array
    {
        $className = get_called_class();
        if (!isset(self::$docComArray[$className])) {
            $reflect = ClassHelper::reflectClass($className);
            $reflectConstArray = $reflect->getReflectionConstants();
            $docComArr = array();
            foreach ($reflectConstArray as $refConst) {
                $docComArr[$refConst->getName()] = DocCommentObject::parse($refConst->getDocComment());
            }

            self::$docComArray[$className] = $docComArr;
        }
        return self::$docComArray[$className];

    }

}