<?php


namespace Gek\Infrastructure\Tests\Reflections;


use Gek\Infrastructure\Reflections\DocCommentObject;
use PHPUnit\Framework\TestCase;

/**
 * Class DocCommentObjectTest
 * @package Gek\Infrastructure\Tests\Reflections
 */
class DocCommentObjectTest extends TestCase
{

    public function testParseMethodNewInstance(): void
    {
        $docCom1 = DocCommentObject::parse(null);
        $docCom2 = DocCommentObject::parse(null);
        $this->assertFalse($docCom1 === $docCom2);
    }

    public function testNullDocComment(): void
    {
        $docCom = DocCommentObject::parse(null);
        $this->assertEquals(null, $docCom->getDocComment());
        $this->assertTrue(empty($docCom->getComments()));
        $this->assertTrue(empty($docCom->getKeys()));
        $this->assertTrue(empty($docCom->getSummary()));
    }

    public function testEmptyDocComment(): void
    {
        $docCom = DocCommentObject::parse('');
        $this->assertEquals('', $docCom->getDocComment());
        $this->assertTrue(empty($docCom->getComments()));
        $this->assertTrue(empty($docCom->getKeys()));
        $this->assertTrue(empty($docCom->getSummary()));
    }

    public function testWhiteSpacesDocComment(): void
    {
        $docCom = DocCommentObject::parse('     ');
        $this->assertEquals('     ', $docCom->getDocComment());
        $this->assertTrue(empty($docCom->getComments()));
        $this->assertTrue(empty($docCom->getKeys()));
        $this->assertTrue(empty($docCom->getSummary()));
    }

    public function testParseDocComments(): void
    {
        $testDocComStr = <<<DOCBLOCK
/**
 * This is a summary.
 * Second line summary.
 *
 * This is a multiline Description
 * that contains a code block.
 *
 *     See here: a CodeBlock
 * @var string|null
 * @required
 * @label sample
 * @dbField
 * @property string test1
 * @property string test2
 */
DOCBLOCK;
        $docCom = DocCommentObject::parse($testDocComStr);
        $this->assertEquals($testDocComStr, $docCom->getDocComment());
        $this->assertCount(3, $docCom->getComments());
        $this->assertCount(2, $docCom->getSummary());

        $this->assertEquals(
            'This is a summary.',
            $docCom->getSummary()[0]
        );
        $this->assertEquals(
            'Second line summary.',
            $docCom->getSummary()[1]
        );
        $this->assertEquals(
            'This is a multiline Description',
            $docCom->getComments()[0]
        );
        $this->assertEquals(
            'that contains a code block.',
            $docCom->getComments()[1]
        );
        $this->assertEquals(
            'See here: a CodeBlock',
            $docCom->getComments()[2]
        );

        $this->assertCount(
            5,
            $docCom->getKeys()
        );

        $this->assertEquals(
            'string|null',
            $docCom->getKey('var')
        );
        $this->assertEquals(
            'string|null',
            $docCom->getKey(' @var ')
        );
        $this->assertEquals(
            true,
            $docCom->getKey('required')
        );
        $this->assertEquals(
            'sample',
            $docCom->getKey(" \t @LaBEl  ")
        );
        $this->assertEquals(
            true,
            $docCom->getKey(" \t @dbfield  ")
        );
        $this->assertEquals(
            true,
            $docCom->getKey(" \t @dBFiElD  ")
        );
        $this->assertEquals(
            ['string test1','string test2'],
            $docCom->getKey(" \t @property  ")
        );
        $this->assertEquals(
            null,
            $docCom->getKey('invalid')
        );

        $this->assertEquals(
            [
                'var' => 'string|null',
                'required' => true,
                'label' => 'sample',
                'dbfield' => true,
                'property' => ['string test1','string test2']
            ],
            $docCom->getKeys()
        );

        $testDocComStr2 = <<<DOCBLOCK
/**
 * This is a summary.
 * @var string|null
 * @required
 * @label sample test
 * @key value
 */
DOCBLOCK;

        $docCom->setDocComment($testDocComStr2);

        $this->assertEquals(
            $testDocComStr2,
            $docCom->getDocComment()
        );
        $this->assertEquals(
            1,
            count($docCom->getSummary())
        );
        $this->assertCount(
            0,
            $docCom->getComments()
        );
        $this->assertEquals(
            ['This is a summary.'],
            $docCom->getSummary()
        );
        $this->assertCount(
            4,
            $docCom->getKeys()
        );
        $this->assertEquals(
            [
                'var' => 'string|null',
                'required' => true,
                'label' => 'sample test',
                'key' => 'value'
            ],
            $docCom->getKeys()
        );

    }


}
