<?php


namespace Gek\Infrastructure;

use DateTime;
use Transliterator;
use function mb_convert_case;
use function mb_detect_encoding;
use function mb_strlen;
use function mb_substr;
use function method_exists;
use function strval;
use function ucfirst;

/**
 * Class Str
 * @package Gek\Infrastructure
 */
class Str
{
    private const FORMAT_START_CHAR = '{';
    private const FORMAT_END_CHAR = '}';
    private const FORMAT_DATE_TIME_FORMAT = 'Y-m-d H:i:s';

    private const CASE_UPPER = MB_CASE_UPPER;
    private const CASE_LOWER = MB_CASE_LOWER;
    private const CASE_TITLE = MB_CASE_TITLE;


    /**
     * @param string $format
     * @param mixed|array ...$params
     * @return string
     */
    public static function format(string $format, ...$params): string
    {
        $res = '';
        $last_therm = '';
        $prev_char = '';
        $therm_flag = false;
        $sChar = self::FORMAT_START_CHAR;
        $eChar = self::FORMAT_END_CHAR;
        $repeatEndCharCount = 0;

        if (count($params) == 1 && is_array($params[0])) {
            $params = $params[0];
        }

        for ($i = 0; $i < strlen($format); $i++) {
            $cur_char = $format[$i];

            if ($cur_char === $sChar) {
                if ($therm_flag === true) {
                    $therm_flag = false;
                    if ($prev_char === $sChar) {
                        $res .= $sChar;
                    }
                    $last_therm = '';
                } else {
                    $therm_flag = true;
                    $last_therm .= $cur_char;
                }

            } elseif ($cur_char === $eChar) {
                if ($therm_flag === true) {
                    $last_therm .= $cur_char;
                    $num = str_replace(array($sChar, $eChar), '', $last_therm);
                    $argFormat = null;
                    if (false !== $indexOf = self::indexOf($num, ":")) {
                        $argFormat = mb_substr($num, $indexOf + 1);
                        $num = mb_substr($num, 0, $indexOf);


                    }
                    if (is_numeric($num) && $num < count($params)) {
                        $val = $params[intval($num)];
                        if (is_bool($val)) {
                            $val = $val ? 'true' : 'false';
                        } elseif (is_array($val)) {
                            $val = '[array]';
                        } elseif (is_object($val)) {
                            if ($val instanceof DateTime) {
                                if ($argFormat !== null) {
                                    $val = $val->format($argFormat);
                                } else {
                                    $val = $val->format(self::FORMAT_DATE_TIME_FORMAT);
                                }
                            } elseif (method_exists($val, '__toString')) {
                                $val = strval($val);
                            } else {

                                $val = '[' . get_class($val) . ']';

                            }
                        }
                        $res .= $val;//strval($val);
                    }
                    $last_therm = '';
                    $therm_flag = false;
                } elseif ($prev_char === $eChar) {
                    if ($repeatEndCharCount == 0) {
                        $res .= $eChar;
                        $repeatEndCharCount++;
                    } else {
                        $repeatEndCharCount = 0;
                    }

                }
            } else {
                if ($therm_flag === true) {
                    $last_therm .= $cur_char;
                } else {
                    $res .= $cur_char;
                }

            }
            $prev_char = $cur_char;
        }
        return $res;
    }

    /*
    public static function binaryEquals($str1, $str2)
    {


        if (!is_string($str1)) {
            if (is_array($str1)) {
                $str1 = pack('C*', ...$str1);
            } else {
                $str1 = pack('C*', $str1);
            }
        }

        if(!is_string($str2)){
            if(is_array($str2)){
                $str2 = pack('C*',...$str2);
            }else{
                $str2 = pack('C*',$str2);
            }
        }

        return $str1 == $str2;

    }
    */

    /**
     * @param string $haystack
     * @param string $needle
     * @param bool $ignoreCaseSensitive
     * @param string|null $encoding
     * @param string|null $language
     * @return false|int
     */
    public static function indexOf(string $haystack, string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null)
    {
        if ($ignoreCaseSensitive) {
            $haystack = static::toLowerCase($haystack, $encoding, $language);
            $needle = static::toLowerCase($needle, $encoding, $language);
            return !empty($encoding) ?
                mb_stripos($haystack, $needle, 0, $encoding) :
                mb_stripos($haystack, $needle);
        }
        return !empty($encoding) ?
            mb_strpos($haystack, $needle, 0, $encoding) :
            mb_strpos($haystack, $needle);
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    public static function toLowerCase(string $str, ?string $encoding = null, ?string $language = null)
    {
        return self::convertCase($str, self::CASE_LOWER, $encoding, $language);
    }

    /**
     * @param string $str
     * @param int $mode
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    protected static function convertCase(string $str, int $mode, ?string $encoding = null, ?string $language = null): string
    {
        if (empty($str)) {
            return $str;
        }
        if ($language !== null) {
            $id = '';
            switch ($mode) {
                case self::CASE_LOWER:
                    $id = '-Lower';
                    break;
                case self::CASE_UPPER:
                    $id = '-Upper';
                    break;
                case self::CASE_TITLE:
                    $id = '-Title';
                    break;
            }
            $translitator = Transliterator::create($language . $id);
            if (empty($translitator)) {
                $translitator = Transliterator::create('Any' . $id);
            }
            if (!empty($translitator)) {
                $result = $translitator->transliterate($str);
                if ($encoding === null) {
                    $encoding = mb_detect_encoding($str);
                    $encoding = ($encoding === false) ? 'UTF-8' : $encoding;
                }
                return mb_convert_encoding($result, $encoding);
            }
        }
        if ($encoding === null) {
            $encoding = mb_detect_encoding($str);
            $encoding = ($encoding === false) ? null : $encoding;
        }
        $result = mb_convert_case($str, $mode, $encoding);
        return $result;
    }

    /**
     * @param string $str
     * @param bool $eval
     * @return string
     */
    public static function unicodeToEscapedUnicode(string $str, bool $eval = true)
    {
        $res = "";
        $charArr = static::toArray($str, 'UNICODE');
        foreach ($charArr as $chr) {
            if ($eval) {
                eval('$res .= "\u{' . bin2hex($chr) . '}";');
            } else {
                $res .= '\u{' . bin2hex($chr) . "}";
            }


        }
        return $res;
    }

    public static function toArray(string $str, ?string $encoding = null, int $splitLength = 1): array
    {
        if (empty($encoding)) {
            return mb_str_split($str, $splitLength);
        }
        return mb_str_split($str, $splitLength, $encoding);
    }

    /**
     * @param string $str
     * @param int $maxLength
     * @param string|null $trimMarker
     * @param string|null $encoding
     * @return string
     */
    public static function ensureMaximumLength(string $str, int $maxLength, ?string $trimMarker = null, ?string $encoding = null): string
    {
        if (!empty($encoding)) {
            $fromEnc = static::detectEncoding($str);
            if ($encoding != $fromEnc) {
                $str = static::convertEncoding($str, $encoding, $fromEnc);
            }
            if (!empty($trimMarker)) {
                $trimMarker = static::convertEncoding($trimMarker, $encoding);
            }
        }
        $isTrim = false;
        if (!empty($encoding)) {
            if (mb_strlen($str, $encoding) > $maxLength) {
                $str = mb_substr($str, 0, $maxLength, $encoding);
                $isTrim = true;
            }
        } else {
            if (mb_strlen($str) > $maxLength) {
                $str = mb_substr($str, 0, $maxLength);
                $isTrim = true;
            }
        }
        if ($isTrim && !empty($trimMarker)) {
            $str .= $trimMarker;
        }

        return $str;
    }

    /**
     * @param string $str
     * @param string|array|string[]|null $encodingList
     * @param bool|null $strict
     * @return false|string
     */
    public static function detectEncoding(string $str, $encodingList = null, ?bool $strict = null)
    {
        if (!empty($encodingList)) {
            $tmp = mb_detect_order();
            mb_detect_order($encodingList);
        }
        $res = mb_detect_encoding($str, $encodingList, $strict);
        if (isset($tmp)) {
            mb_detect_order($tmp);
        }
        if (false == $res) {
            if (!empty($encodingList)) {
                if (false == is_array($encodingList)) {
                    $encodingList = explode(",", $encodingList);
                }
                foreach ($encodingList as $enc) {
                    if (static::checkEncoding($str, $enc)) {
                        return $enc;
                    }
                }
            }
        }
        return $res;
    }

    /**
     * @param string $str
     * @param string $encoding
     * @return bool
     */
    public static function checkEncoding(string $str, string $encoding): bool
    {
        return mb_check_encoding($str, $encoding);
    }

    /**
     * @param string $str
     * @param string|null $toEncoding
     * @param string|null $fromEncoding
     * @return string
     */
    public static function convertEncoding(string $str, ?string $toEncoding = null, ?string $fromEncoding = null)
    {
        if (empty($toEncoding)) {
            $toEncoding = mb_internal_encoding();
        }
        if (empty($fromEncoding)) {
            $fromEncoding = static::detectEncoding($str, ['UCS-2', 'UTF-7', 'UTF-8', 'ASCII'], true);
        }
        // if (!empty($fromEncoding)) {
        return mb_convert_encoding($str, $toEncoding, $fromEncoding);
        //}
        // return mb_convert_encoding($str, $toEncoding);

    }

    /**
     * @param string $haystack
     * @param string $needle
     * @param bool $ignoreCaseSensitive
     * @param string|null $encoding
     * @param string|null $language
     * @return bool
     */
    public static function startsWith(string $haystack, ?string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null): bool
    {
        if ($needle === null) {
            return false;
        }
        $length = static::len($needle, $encoding);
        if ($length == 0) {
            return false;
        }
        $opr1 = !empty($encoding) ?
            mb_substr($haystack, 0, $length, $encoding) :
            mb_substr($haystack, 0, $length);
        return static::equals($opr1, $needle, $ignoreCaseSensitive, true, $encoding, $language);
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @return int
     */
    public static function len(string $str, ?string $encoding = null): int
    {
        if (empty($encoding)) {
            $encoding = static::getDefaultEncoding();
        }
        return mb_strlen($str, $encoding);
    }

    /**
     * @return string
     */
    protected static function getDefaultEncoding(): string
    {
        return mb_internal_encoding();
    }

    /**
     * @param mixed $str1
     * @param mixed $str2
     * @param bool $ignoreCaseSensitive
     * @param bool $strictMode
     * @param string|null $encoding
     * @param string|null $language
     * @return bool
     */
    public static function equals($str1, $str2, bool $ignoreCaseSensitive = false, bool $strictMode = false, ?string $encoding = null, ?string $language = null): bool
    {

        if (empty($str1) || empty($str2)) {
            if ($strictMode) {
                if ($str1 === null && $str2 === null) {
                    return true;
                }
                if ($str1 === '' && $str2 === '') {
                    return true;
                }
                return false;
            } else {
                if (empty($str1) && empty($str2)) {
                    return true;
                } else {
                    return false;
                }
            }
        }

        if ($strictMode && (false == is_string($str1) || false == is_string($str2))) {
            if (is_string($str1) || is_string($str2)) {
                return false;
            }
        }


        $opr1 = $ignoreCaseSensitive ?
            static::toLowerCase($str1, $encoding, $language) :
            $str1;
        $opr2 = $ignoreCaseSensitive ?
            static::toLowerCase($str2, $encoding, $language) :
            $str2;


        return strcmp($opr1, $opr2) === 0;
    }

    /**
     * @param string $haystack
     * @param string $needle
     * @param bool $ignoreCaseSensitive
     * @param string|null $encoding
     * @param string|null $language
     * @return bool
     */
    public static function endsWith(string $haystack, ?string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null): bool
    {
        if ($needle === null) {
            return false;
        }
        $length = static::len($needle, $encoding);
        if ($length == 0) {
            return false;
        }
        $opr1 = !empty($encoding) ?
            mb_substr($haystack, -$length, null, $encoding) :
            mb_substr($haystack, -$length);
        return static::equals($opr1, $needle, $ignoreCaseSensitive, true, $encoding, $language);
    }

    public static function containsCount(string $haystack, ?string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null): int
    {
        if ($needle === null) {
            return 0;
        }
        if (empty($encoding)) {
            $encoding = static::getDefaultEncoding();
        }

        if (static::len($needle, $encoding) == 0) {
            return 0;
        }

        if ($ignoreCaseSensitive) {
            $haystack = static::toLowerCase($haystack, $encoding, $language);
            $needle = static::toLowerCase($needle, $encoding, $language);
        }

        return mb_substr_count($haystack, $needle, $encoding);
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    public static function toTitleCase(string $str, ?string $encoding = null, ?string $language = null): string
    {
        return self::convertCase($str, self::CASE_TITLE, $encoding, $language);
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    public static function ucFirst(string $str, ?string $encoding = null, ?string $language = null): string
    {
        if (empty(trim($str))) {
            return $str;
        }
        if ($encoding === null && $language == null) {
            return ucfirst($str);
        }
        $firstChar = mb_substr($str, 0, 1);
        $firstChar = self::toUpperCase($firstChar, $encoding, $language);
        $otherChars = mb_substr($str, 1);
        if ($encoding !== null) {
            $otherChars = mb_convert_encoding($otherChars, $encoding);
        }
        return $firstChar . $otherChars;
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    public static function toUpperCase(string $str, ?string $encoding = null, ?string $language = null)
    {
        return self::convertCase($str, self::CASE_UPPER, $encoding, $language);
    }

    /**
     * @param string $haystack
     * @param string $needle
     * @param bool $ignoreCaseSensitive
     * @param string|null $encoding
     * @param string|null $language
     * @return false|int
     */
    public static function lastIndexOf(string $haystack, string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null)
    {
        if ($ignoreCaseSensitive) {
            $haystack = static::toLowerCase($haystack, $encoding, $language);
            $needle = static::toLowerCase($needle, $encoding, $language);
            return !empty($encoding) ?
                mb_strripos($haystack, $needle, 0, $encoding) :
                mb_strripos($haystack, $needle);
        }
        return !empty($encoding) ?
            mb_strrpos($haystack, $needle, 0, $encoding) :
            mb_strrpos($haystack, $needle);
    }

    #region utils

    /**
     * @param string $str
     * @param string $space
     * @param bool $onyOneSpace
     * @param string $convertChars
     * @return string
     */
    public static function fixSpaces(string $str, string $space = " ", bool $onyOneSpace = true, string $convertChars = " \t\n\r\0\x0B")
    {
        if ($convertChars !== '' && $convertChars !== null) {
            $searchChars = self::toArray($convertChars);
            $str = str_replace($searchChars, $space, $str);
        }
        if ($onyOneSpace) {
            $searchSpace = str_repeat($space, 2);
            while (self::contains($str, $searchSpace)) {
                $str = str_replace($searchSpace, $space, $str);
            }

        }
        return $str;
    }

    /**
     * @param string $haystack
     * @param string|null $needle
     * @param bool $ignoreCaseSensitive
     * @param string|null $encoding
     * @param string|null $language
     * @return bool
     */
    public static function contains(string $haystack, ?string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null): bool
    {
        if ($needle === null) {
            return false;
        }

        if (static::len($needle, $encoding) == 0) {
            return false;
        }

        if ($ignoreCaseSensitive) {
            $haystack = static::toLowerCase($haystack, $encoding, $language);
            $needle = static::toLowerCase($needle, $encoding, $language);
        }


        return mb_strpos($haystack, $needle, $encoding) !== false;
    }

    #endregion utils


}
