<?php /** @noinspection PhpUnhandledExceptionInspection */


namespace Gek\Infrastructure\Math\DecimalExpressions;


use Gek\Infrastructure\Exceptions\GekException;
use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Math\Decimal;
use Gek\Infrastructure\Str;
use Throwable;


class DecimalExpressionParser
{
    public static function parseValue(string $decimalExpStr, ?int $scale = null): Decimal
    {
        try {
            $tokenArr = self::tokenize($decimalExpStr);

            $node = self::createNodeTree($tokenArr, $scale);
            return $node->calculateValue();
        } catch (Throwable $exp) {
            throw new GekException('Parse Error.' . $exp->getMessage(), 0, $exp);
        }
    }

    private static function tokenize($str, &$i = 0, $isSub = false): array
    {
        $numFlag = false;
        $curNum = '';
        $arr = array();
        $strArr = static::strToArray($str);

        for (; $i < count($strArr); $i++) {

            if (ctype_space($strArr[$i])) {

                if ($numFlag) {
                    $numFlag = false;
                    $arr[] = floatval($curNum);
                    $curNum = '';
                }
            } elseif (ctype_digit($strArr[$i]) || $strArr[$i] == '.') {

                if ($numFlag) {
                    $curNum .= $strArr[$i];
                } else {
                    $curNum = $strArr[$i];
                    $numFlag = true;
                }
            } elseif (in_array($strArr[$i], array('+', '-', '*', '/', '%', ')', '('))) {
                if ($numFlag) {
                    $numFlag = false;
                    $arr[] = floatval($curNum);
                    $curNum = '';
                }
                if ($strArr[$i] == '(') {
                    $i++;
                    $arr[] = self::tokenize($str, $i, true);
                } elseif ($strArr[$i] == ')') {
                    if ($isSub) {
                        $i++;
                        if (!empty($curNum)) {
                            $arr[] = floatval($curNum);
                        }
                        return array_reverse($arr);
                    }
                } elseif ($strArr[$i] == '-') {
                    $isNegativeSign = empty($arr) ||
                        in_array($arr[count($arr) - 1], array('+', '-', '*', '/', '%', '('));
                    if ($isNegativeSign) {
                        $curNum = $strArr[$i];
                        $numFlag = true;
                    } else {
                        $arr[] = $strArr[$i];
                    }
                } else {
                    $arr[] = $strArr[$i];
                }
            } else {
                throw new InvalidArgumentException(
                    Str::format('Sytax error.  $str: "{0}" , $str[$i] : "{1}" ', $str, $str[$i]),
                    0,
                    null,
                    '$str'
                );
            }
        }
        if ($curNum !== null && $curNum !== '') {
            $arr[] = floatval($curNum);
        }
        return array_reverse($arr);
    }

    private static function strToArray(string $str, string $encoding = "UTF-8"): array
    {
        $len = mb_strlen($str, $encoding);
        $result = [];
        for ($i = 0; $i < $len; $i++) {
            $result[] = mb_substr($str, $i, 1, $encoding);
        }
        return $result;
    }


    private static function createNodeTree(array $revArr, ?int $scale = null): ?DecimalExpression
    {
        $lastExp = null;
        foreach ($revArr as $token) {
            if (is_numeric($token)) {
                $exp = new DecimalValueExpression(Decimal::wrap($token, $scale));
                if ($lastExp === null) {
                    $lastExp = $exp;
                } else {
                    switch (true) {
                        case ($lastExp instanceof DecimalValueExpression):
                        case ($lastExp instanceof DecimalGroupExpression):
                            throw new GekException('Syntax Error');
                            break;
                        case ($lastExp instanceof DecimalAddExpression):
                        case ($lastExp instanceof DecimalSubExpression):
                        case ($lastExp instanceof DecimalDivExpression):
                        case ($lastExp instanceof DecimalModExpression):
                        case ($lastExp instanceof DecimalMulExpression):
                            $lastExp->left = $exp;
                            break;
                        default:
                            throw new GekException('Bilinmeyen tür.');
                    }
                }
            } elseif (is_string($token)) {
                if ($lastExp == null || !in_array($token, ['+', '-', '*', '/', '%'])) {
                    throw new GekException('Syntax Error : ' . $token);
                }
                $exp = null;
                switch ($token) {
                    case '+':
                        $exp = new DecimalAddExpression(null, $lastExp);
                        break;
                    case '-':
                        $exp = new DecimalSubExpression(null, $lastExp);
                        break;
                    case '*':
                        $exp = new DecimalMulExpression(null, $lastExp);
                        break;
                    case '/':
                        $exp = new DecimalDivExpression(null, $lastExp);
                        break;
                    case '%':
                        $exp = new DecimalModExpression(null, $lastExp);
                        break;
                }
                $lastExp = $exp;
            } elseif (is_array($token)) {
                $exp = self::createNodeTree($token, $scale);
                $exp = new DecimalGroupExpression($exp);
                if ($lastExp === null) {
                    $lastExp = $exp;
                } else {
                    switch (true) {
                        case ($lastExp instanceof DecimalValueExpression):
                            throw new GekException('Syntax Error');
                            break;
                        case ($lastExp instanceof DecimalGroupExpression):
                            self::handleGroup($lastExp, $exp);
                            break;
                        case ($lastExp instanceof DecimalAddExpression):
                        case ($lastExp instanceof DecimalSubExpression):
                        case ($lastExp instanceof DecimalDivExpression):
                        case ($lastExp instanceof DecimalModExpression):
                        case ($lastExp instanceof DecimalMulExpression):
                            $lastExp->left = $exp;
                            break;
                        default:
                            throw new GekException('Bilinmeyen tür.');
                    }
                }
            }
        }
        return $lastExp;
    }

    private static function handleGroup(DecimalGroupExpression $groupExp, DecimalExpression $exp)
    {
        $lastExp = $groupExp->startExp;
        switch (true) {
            case ($lastExp instanceof DecimalValueExpression):
                throw new GekException('Syntax Error in group');
                break;
            case ($lastExp instanceof DecimalGroupExpression):
                self::handleGroup($lastExp, $exp);
                break;
            case ($lastExp instanceof DecimalAddExpression):
            case ($lastExp instanceof DecimalSubExpression):
            case ($lastExp instanceof DecimalDivExpression):
            case ($lastExp instanceof DecimalModExpression):
            case ($lastExp instanceof DecimalMulExpression):
                $lastExp->left = $exp;
                break;
            default:
                throw new GekException('Bilinmeyen tür.');
        }
    }

}