<?php /** @noinspection PhpUnhandledExceptionInspection */


namespace Gek\Infrastructure\Math;


use Exception;
use Gek\Infrastructure\Exceptions\GekException;
use Gek\Infrastructure\Math\DecimalExpressions\DecimalExpressionParser;
use Gek\Infrastructure\Str;

class Decimal
{
    #region fields

    protected int $scale = 6;

    protected string $value;

    protected string $decPoint = ".";


    protected string $thousandsSep = ",";

    #endregion fields

    #region ctor

    /**
     * FloatWrapperForCurrency constructor.
     * @param float|int|string $value
     * @param int $scale
     * @param string $decPoint
     * @param string $thousandsSep
     * @throws Exception
     */
    public function __construct($value = 0.0, int $scale = 6, string $decPoint = '.', string $thousandsSep = ",")
    {
        $this->scale = $scale;
        $this->decPoint = $decPoint;
        $this->thousandsSep = $thousandsSep;

        if (is_string($value)) {
            if ($decPoint != '.') {
                $value = str_replace($decPoint, '.', $value);
                if (Str::containsCount($value, '.') > 1) {
                    $pointIndex = Str::lastIndexOf($value, '.');
                    $dec = mb_substr($value, $pointIndex);
                    $int = mb_substr($value, 0, $pointIndex);
                    $value = str_replace('.', '', $int) . '.'
                        . str_replace('.', '', $dec);

                }
            }
            if (is_numeric($value)) {

                $value = floatval($value);
            }
        }

        if (is_float($value) || is_int($value)) {
            $value = $this->numberToString($value);
        } else {
            throw new Exception('$value gecerli bir sayisal deger icermiyor.');
        }
        $this->value = $value;

    }

    #endregion ctor

    #region properties

    /**
     * @param float $number
     * @return string
     */
    protected function numberToString(float $number): string
    {
        //$number = round($number,$this->scale);
        return number_format($number, $this->scale, '.', '');
    }

    /**
     * @param self|float|int|string $val
     * @param int|null $scale
     * @return static
     * @throws Exception
     */
    public static function wrap($val, ?int $scale = null): self
    {
        if ($scale !== null) {
            return new self($val, $scale);
        }
        return new self($val);
    }

    /**
     * @param string $exp
     * @param int|null $scale
     * @return float
     * @throws GekException
     */
    public static function expF(string $exp, ?int $scale = null): float
    {

        return static::exp($exp, $scale)->toFloat();
    }

    /**
     * @param string $exp
     * @param int|null $scale
     * @return Decimal
     * @throws GekException
     */
    public static function exp(string $exp, ?int $scale = null)
    {

        return DecimalExpressionParser::parseValue($exp, $scale);
    }


    #endregion properties

    #region methods

    /**
     * @return int
     */
    public function getScale(): int
    {
        return $this->scale;
    }

    /**
     * @return string
     */
    public function getDecPoint(): string
    {
        return $this->decPoint;
    }

    /**
     * @return string
     */
    public function getThousandsSep(): string
    {
        return $this->thousandsSep;
    }

    /**
     * @param float|int|string|Decimal $other
     * @return Decimal
     * @throws Exception
     */
    public function add($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcadd($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    protected function getOtherValue($other)
    {
        if ($other instanceof Decimal) {
            if ($other->scale > $this->scale) {
                $dc = new self($other->getValue(), $this->scale);
                return $dc->value;
            }
            return $other->value;
        }
        if (is_float($other) || is_int($other)) {
            return $this->numberToString($other);
        }
        if (is_string($other) && is_numeric($other)) {
            $other = floatval($other);
            return $this->numberToString($other);
        }
        throw new Exception('$other gecersiz bir turde.');
    }

    /**
     * @return string
     */
    public function getValue(): string
    {
        return $this->value;
    }

    /**
     * @param float|int|string|Decimal $otger
     * @return bool
     * @throws Exception
     */
    public function equals($otger): bool
    {
        return $this->compare($otger) == 0;
    }

    /**
     * @param float|int|string|Decimal $other
     * @return int
     * @throws Exception
     */
    public function compare($other): int
    {
        $otherVal = $this->getOtherValue($other);
        return bccomp($this->value, $otherVal, $this->scale);
    }

    /**
     * @param float|int|string|Decimal $other
     * @return Decimal
     * @throws Exception
     */
    public function div($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcdiv($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * @param float|int|string|Decimal $other
     * @return Decimal
     * @throws Exception
     */
    public function mod($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcmod($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * @param float|int|string|Decimal $other
     * @return Decimal
     * @throws Exception
     */
    public function mul($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcmul($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * @param int $exponent
     * @return Decimal
     * @throws Exception
     */
    public function pow(int $exponent)
    {

        $resVal = bcpow($this->value, $exponent, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * @param float|int|string|Decimal $other
     * @return Decimal
     * @throws Exception
     */
    public function sub($other)
    {
        $otherVal = $this->getOtherValue($other);
        $resVal = bcsub($this->value, $otherVal, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }

    /**
     * @return Decimal
     * @throws Exception
     */
    public function sqrt()
    {
        $resVal = bcsqrt($this->value, $this->scale);
        return new self($resVal, $this->scale, $this->decPoint);
    }



    #endregion methods

    #region utils

    /**
     * @return string
     */
    public function __toString()
    {
        return number_format($this->toFloat(), $this->scale, $this->decPoint, $this->thousandsSep);
    }

    /**
     * @return float
     */
    public function toFloat(): float
    {
        return floatval($this->value);
    }

    #endregion utils

    #region statics

    /**
     * @param int $scale
     * @return false|float
     */
    public function round(int $scale = 0)
    {
        return round($this->toFloat(), $scale);
    }

    /**
     * @return false|float
     */
    public function floor()
    {
        return floor($this->toFloat());
    }

    /**
     * @return false|float
     */
    public function ceil()
    {
        return ceil($this->toFloat());
    }

    #endregion statics


}