<?php


namespace Gek\Infrastructure\FlagExp;


use Error;
use Exception;
use Gek\Infrastructure\Exceptions\GekException;
use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Str;

/**
 * Class FlagExpressionParser
 * @package Gek\Infrastructure\FlagExp
 */
class FlagExpressionParser
{

    /**
     * @param string $flagExpString
     * @return int
     * @throws GekException
     */
    public static function parseValue(string $flagExpString):int {
        try{
            $tokenArr = self::tokenize($flagExpString);
            $node = self::createNodeTree($tokenArr);
            return $node->calculateValue();
        }catch (Exception|Error $exp){
            throw new GekException('Parse Error.',0,$exp);
        }
    }

    /**
     * @param $str
     * @param int $i
     * @param bool $isSub
     * @return array
     * @throws InvalidArgumentException
     */
    private static function tokenize($str, &$i = 0, $isSub = false): array
    {
        $numFlag = false;
        $curNum = '';
        $arr = array();

        for (; $i < strlen($str); $i++) {
            if (ctype_space($str[$i])) {
                if ($numFlag) {
                    $numFlag = false;
                    $arr[] = intval($curNum);
                    $curNum = '';
                }
            }elseif (ctype_digit($str[$i])) {
                if ($numFlag) {
                    $curNum .= $str[$i];
                } else {
                    $curNum = $str[$i];
                    $numFlag = true;
                }
            } elseif (in_array($str[$i], array('|', '&', '^', '~', ')', '('))) {
                if ($numFlag) {
                    $numFlag = false;
                    $arr[] = intval($curNum);
                    $curNum = '';
                }
                if ($str[$i] == '(') {
                    $i++;
                    $arr[] = self::tokenize($str, $i, true);
                } elseif ($str[$i] == ')') {
                    if ($isSub) {
                        $i++;
                        if (!empty($curNum)) {
                            $arr[] = intval($curNum);
                        }
                        return array_reverse($arr);
                    }
                } else {
                    $arr[] = $str[$i];
                }
            }else{
                throw new InvalidArgumentException(
                    Str::format('Sytax error.  $str: "{0}" , $str[$i] : "{1}" ', $str, $str[$i]),
                    0,
                    null,
                    '$str'
                );
            }
        }
        if (!empty($curNum)) {
            $arr[] = intval($curNum);
        }
        return array_reverse($arr);
    }


    /**
     * @param array $revArr
     * @return FlagExpression|null
     * @throws GekException
     */
    private static function createNodeTree(array $revArr):?FlagExpression
    {
        $lastExp = null;
        foreach ($revArr as $token) {
            if (is_int($token)) {
                $exp = new IntValueExpression($token);
                if ($lastExp === null) {
                    $lastExp = $exp;
                } else {
                    switch (true) {
                        case ($lastExp instanceof IntValueExpression):
                            throw new GekException('Syntax Error');
                            break;
                        case ($lastExp instanceof BitwiseAndExpression):
                        case ($lastExp instanceof BitwiseOrExpression):
                        case ($lastExp instanceof BitwiseXorExpression):
                        case ($lastExp instanceof BitwiseNotExpression):
                            $lastExp->left = $exp;
                            break;
                        default:
                            throw new GekException('Bilinmeyen tür.');
                    }
                }
            } elseif (is_string($token)) {
                if ($lastExp == null || !in_array($token, ['|', '&', '^', '~'])) {
                    throw new GekException('Syntax Error : ' . $token);
                }
                switch ($token) {
                    case '|':
                        $exp = new BitwiseOrExpression(null, $lastExp);
                        break;
                    case '&':
                        $exp = new BitwiseAndExpression(null, $lastExp);
                        break;
                    case '^':
                        $exp = new BitwiseXorExpression(null, $lastExp);
                        break;
                    case '~':
                        $exp = new BitwiseNotExpression($lastExp);
                        break;
                }
                $lastExp = $exp;
            }elseif (is_array($token)){
                $exp = self::createNodeTree($token);
                if ($lastExp === null) {
                    $lastExp = $exp;
                } else {
                    switch (true) {
                        case ($lastExp instanceof IntValueExpression):
                            throw new GekException('Syntax Error');
                            break;
                        case ($lastExp instanceof BitwiseAndExpression):
                        case ($lastExp instanceof BitwiseOrExpression):
                        case ($lastExp instanceof BitwiseXorExpression):
                        case ($lastExp instanceof BitwiseNotExpression):
                            $lastExp->left = $exp;
                            break;
                        default:
                            throw new GekException('Bilinmeyen tür.');
                    }
                }
            }
        }
        return $lastExp;
    }

}
