<?php


namespace Gek\Infrastructure\Events;


use Closure;
use Countable;
use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use ReflectionException;
use ReflectionFunction;
use ReflectionNamedType;
use function count;
use function md5;
use function spl_object_hash;

/**
 * Class EventHandler
 * @package Gek\Infrastructure\Events
 */
class EventHandler implements Countable
{
    #region fields


    /**
     * @var array|Closure[]
     */
    private array $_eventArray = array();

    #endregion fields

    #region ctor


    /**
     * EventHandler constructor.
     * @param string|null $eventArgumentsClass
     */
    public function __construct(?string $eventArgumentsClass = null)
    {

    }

    #endregion ctor

    #region methods

    /**
     * Olay dinleyiciyi ekler
     * @param callable $listener dinleyici
     * @param bool $checkCallableSing
     * @throws InvalidArgumentException
     * @throws ReflectionException
     */
    public function add(callable $listener, $checkCallableSing = false): void
    {
        $key = $this->createKey($listener);
        if(!($listener instanceof Closure)){
            $listener = Closure::fromCallable($listener);
        }
        if($checkCallableSing){
            $reflect = new ReflectionFunction($listener);
            $callableCheck = $reflect->getNumberOfParameters() >= 2;
            if($callableCheck){
                $param1 = $reflect->getParameters()[0];
                if($param1->hasType()){
                    $type1 = $param1->getType();
                    if($type1 instanceof ReflectionNamedType){
                        $callableCheck = $type1->getName() == 'object';
                    }
                }
            }
            if($callableCheck){
                $param2 = $reflect->getParameters()[1];
                if($param2->hasType()){
                    $type2 = $param2->getType();
                    if($type2 instanceof ReflectionNamedType){
                        $eventArgName = $type2->getName();
                        $callableCheck = ($eventArgName == EventArguments::class) ||
                            is_subclass_of($eventArgName,EventArguments::class);
                    }
                }
            }

            if(!$callableCheck){
                throw new InvalidArgumentException('$listener parametre tipleri geçerli değil.');
            }
        }
        $this->_eventArray[$key] = $listener;
    }

    /**
     * olay dinleyiciyi siler
     * @param callable $listener
     */
    public function remove(callable $listener): void
    {
        $key = $this->createKey($listener);
        if (isset($this->_eventArray[$key])) {
            unset($this->_eventArray[$key]);
        }
    }

    /**
     * Bütün olay dinleyicileri temizler
     */
    public function clear(): void
    {
        $this->_eventArray = array();
    }

    /**
     * Olayları tetikler
     * @param object $sender
     * @param EventArguments $eventArgs
     */
    public function invoke(object $sender, EventArguments $eventArgs): void
    {
        foreach ($this->_eventArray as $fn) {
            $fn($sender, $eventArgs);
        }
    }

    /**
     * Olay listesi boşsa true diğer durumlarda false döndürür.
     * @return bool
     */
    public function isEmpty(): bool
    {
        return $this->count() == 0;
    }

    /**
     * {@inheritDoc}
     * @return int
     */
    public function count(): int
    {
        return count($this->_eventArray);
    }

    #endregion methods

    #region utils

    /**
     * Benzersiz nesne hash'i oluşturur.
     * @param callable $listener
     * @return string
     */
    protected function createKey(callable $listener):string
    {
        if ($listener instanceof Closure) {
            return spl_object_hash($listener);
        }

        if(is_string($listener) && strpos($listener,'::') !== false){
            $listener = explode('::',$listener);
        }
        if (is_array($listener)) {

            $key = is_object($listener[0]) ?
                spl_object_hash($listener[0]) :
                md5($listener[0]);
            $key .= md5($listener[1]);
            return $key;
        }



        return md5($listener);

    }

    #endregion utils

}
