<?php


namespace Gek\Infrastructure;


use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Exceptions\GekException;
use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Exceptions\NotImplementedException;
use Gek\Infrastructure\FlagExp\FlagExpressionParser;

/**
 * Class FlagEnumType
 * @package Gek\Infrastructure
 */
abstract class FlagEnumType extends FlagEnum implements IToInt
{
    private int $value;

    /**
     * FlagEnumType constructor.
     * @param $value
     * @param bool $validateValue
     * @throws GekException
     * @throws \ReflectionException
     */
    public function __construct($value, bool $validateValue = true)
    {
        $this->setValue($value, $validateValue);
    }

    /**
     * @param string|int $value
     * @param bool $validateValue
     * @throws \ReflectionException
     * @throws Exceptions\GekException
     */
    public function setValue($value, bool $validateValue = true): void
    {
        if (is_int($value)) {
            $this->value = $value;
        } elseif (is_string($value)) {
            $enumVals = self::getEnumValues();
            $keys = array_keys($enumVals);
            $invalidKeys = array();
            $rs = preg_split('/[\|\&\~\^\)\(\t ]/', $value, -1, PREG_SPLIT_NO_EMPTY);
            $rs = array_filter($rs, function ($item) {
                return !ctype_digit($item);
            });

            foreach ($rs as $ky){
                if(!in_array($ky,$keys)){
                    $invalidKeys[] = $ky;
                }
            }
            if (!empty($invalidKeys)) {
                throw new InvalidArgumentException(
                    Str::format('Geçersiz flag key. ("{0}")', implode(' , ', $invalidKeys)),
                    0,
                    null,
                    '$value'
                );
            }

            $value = str_replace($keys, array_values($enumVals), $value);
            $value = Str::toUpperCase($value);
            $all = $this::allFlags();
            $value = str_replace(['ALL', 'ALLFLAGS', 'ALLFLAG'], $all, $value);
            $int = FlagExpressionParser::parseValue($value);
            $this->value = $int;
        } else {
            throw new InvalidArgumentException(
                Str::format('$value ("{0}") parametresi türü yanlış. String ya da int olmalı.', $value)
                , 0, null, '$value'
            );
        }

        if ($validateValue) {
            $allInt = self::allFlags();
            if ($this->value == 0 || ($allInt | $this->value) != $allInt) {
                throw new ArgumentOutOfRangeException(
                    Str::format('value değeri ({0}) flag değerini aşıyor.', $value),
                    0, null, '$value'
                );
            }
        }
    }

    /**
     * @return int
     */
    public function getValue(): int
    {
        return $this->value;
    }

    /**
     * @return int
     */
    public function toInt(): int
    {
        return $this->value;
    }

    /**
     * @return string
     * @throws \ReflectionException
     */
    public function __toString()
    {

        $valEnums = self::getEnumValues();
        $strArr = array();
        foreach ($valEnums as $key => $val) {
            if (($this->value & $val) == $val) {
                $strArr[] = $key;
            }
        }
        return implode(' | ', $strArr);
    }

    /**
     * @param $name
     * @param $arguments
     * @return mixed
     * @throws NotImplementedException
     * @throws \ReflectionException
     */
    public static function __callStatic($name, $arguments)
    {
        if (self::checkEnumKey($name)) {
            $class = get_called_class();
            return new $class($name);
        }
        throw new NotImplementedException();
    }
}
