<?php


namespace Gek\Infrastructure;

use DateTime;
use Transliterator;
use function mb_convert_case;
use function mb_detect_encoding;
use function mb_strlen;
use function mb_substr;
use function method_exists;
use function strval;
use function ucfirst;

/**
 * Class Str
 * @package Gek\Infrastructure
 */
class Str
{
    private const FORMAT_START_CHAR = '{';
    private const FORMAT_END_CHAR = '}';
    private const FORMAT_DATE_TIME_FORMAT  = 'Y-m-d H:i:s';

    private const CASE_UPPER = MB_CASE_UPPER;
    private const CASE_LOWER = MB_CASE_LOWER;
    private const CASE_TITLE = MB_CASE_TITLE;


    /**
     * @param string $format
     * @param mixed|array ...$params
     * @return string
     */
    public static function format(string $format, ...$params): string
    {
        $res = '';
        $last_therm = '';
        $prev_char = '';
        $therm_flag = false;
        $sChar = self::FORMAT_START_CHAR;
        $eChar = self::FORMAT_END_CHAR;
        $repeatEndCharCount = 0;

        if (count($params) == 1 && is_array($params[0])) {
            $params = $params[0];
        }

        for ($i = 0; $i < strlen($format); $i++) {
            $cur_char = $format[$i];

            if ($cur_char === $sChar) {
                if ($therm_flag === true) {
                    $therm_flag = false;
                    if ($prev_char === $sChar) {
                        $res .= $sChar;
                    }
                    $last_therm = '';
                } else {
                    $therm_flag = true;
                    $last_therm .= $cur_char;
                }

            } elseif ($cur_char === $eChar) {
                if ($therm_flag === true) {
                    $last_therm .= $cur_char;
                    $num = str_replace(array($sChar, $eChar), '', $last_therm);
                    if (strrpos($num, ':') !== false) {
                        $num = explode(':', $num);
                        $num = $num[0];
                    }
                    if (is_numeric($num) && $num < count($params)) {
                        $val = $params[intval($num)];
                        if (is_bool($val)) {
                            $val = $val ? 'true' : 'false';
                        } elseif (is_array($val)) {
                            $val = '[array]';
                        }elseif (is_object($val)){
                            if(method_exists($val,'__toString')){
                                $val  = strval($val);
                            }else{
                                if($val instanceof DateTime){
                                    $val = $val->format(self::FORMAT_DATE_TIME_FORMAT);
                                }else{
                                    $val = '['.get_class($val).']';
                                }
                            }
                        }
                        $res .= strval($val);
                    }
                    $last_therm = '';
                    $therm_flag = false;
                } elseif ($prev_char === $eChar) {
                    if ($repeatEndCharCount == 0) {
                        $res .= $eChar;
                        $repeatEndCharCount++;
                    } else {
                        $repeatEndCharCount = 0;
                    }

                }
            } else {
                if ($therm_flag === true) {
                    $last_therm .= $cur_char;
                } else {
                    $res .= $cur_char;
                }

            }
            $prev_char = $cur_char;
        }
        return $res;
    }

    /**
     * @param string $str
     * @param int $maxLength
     * @return string
     */
    public static function ensureMaximumLength(string $str, int $maxLength):string{
        if(mb_strlen($str) > $maxLength){
            $str = mb_substr($str,0,$maxLength);
        }
        return $str;
    }


    /**
     * @param string $haystack
     * @param string $needle
     * @param bool $ignoreCaseSensitive
     * @return bool
     */
    public static function startsWith(string $haystack, ?string $needle,  bool $ignoreCaseSensitive = false):bool {
        if($needle === null){
            return false;
        }
        if($ignoreCaseSensitive){
            $haystack = mb_strtolower($haystack,'UTF-8');
            $needle = mb_strtolower($needle,'UTF-8');
        }

        $length = mb_strlen($needle);
        if($length == 0){
            return false;
        }
        return mb_substr($haystack,0,$length) == $needle;
    }

    /**
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    public static function endsWith(string $haystack, ?string $needle):bool {
        if($needle === null){
            return false;
        }
        $length = mb_strlen($needle);
        if($length == 0){
            return false;
        }
        return mb_substr($haystack,-$length) == $needle;
    }

    /**
     * @param string $haystack
     * @param string|null $needle
     * @return bool
     */
    public static function contains(string $haystack, ?string $needle):bool {
        if($needle === null){
            return false;
        }

        if(mb_strlen($needle) == 0){
            return false;
        }

        return mb_strpos($haystack,$needle) !== false;
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    public static function toLowerCase(string $str, ?string $encoding = null, ?string $language = null){
        return self::convertCase($str,self::CASE_LOWER,$encoding,$language);
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    public static function toUpperCase(string $str, ?string $encoding = null, ?string $language = null){
        return self::convertCase($str,self::CASE_UPPER,$encoding,$language);
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    public static function toTitleCase(string $str, ?string $encoding = null, ?string $language = null):string {
        return self::convertCase($str,self::CASE_TITLE,$encoding,$language);
    }

    /**
     * @param string $str
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    public static function ucFirst(string $str, ?string $encoding = null, ?string $language = null):string {
        if(empty(trim($str))){
            return $str;
        }
        if($encoding === null && $language == null){
            return ucfirst($str);
        }
        $firstChar = mb_substr($str,0,1);
        $firstChar = self::toUpperCase($firstChar,$encoding,$language);
        $otherChars = mb_substr($str,1);
        if($encoding !== null){
            $otherChars = mb_convert_encoding($otherChars,$encoding);
        }
        return $firstChar . $otherChars;
    }



    #region utils

    /**
     * @param string $str
     * @param int $mode
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     */
    private static function convertCase(string $str,int $mode, ?string $encoding = null, ?string  $language = null){
        if(empty($str)){
            return $str;
        }
        if($language !== null){
            $id  = '';
            switch ($mode){
                case self::CASE_LOWER:
                    $id = '-Lower';
                    break;
                case self::CASE_UPPER:
                    $id = '-Upper';
                    break;
                case self::CASE_TITLE:
                    $id = '-Title';
                    break;
            }
            $translitator =  Transliterator::create($language.$id);
            if(empty($translitator)){
                $translitator =  Transliterator::create('Any'.$id);
            }
            if(!empty($translitator)){
                $result =  $translitator->transliterate($str);
                if($encoding === null){
                    $encoding = mb_detect_encoding($str);
                    $encoding = ($encoding === false) ? 'UTF-8' : $encoding;
                }
                return mb_convert_encoding($result,$encoding);
            }
        }
        if($encoding === null){
            $encoding = mb_detect_encoding($str);
            $encoding = ($encoding === false) ? null : $encoding;
        }
        $result = mb_convert_case($str,$mode,$encoding);
        return $result;
    }

    #endregion utils


}
