<?php
declare(strict_types=1);

namespace Gek\Infrastructure\Tests;


use Gek\Infrastructure\Str;
use Gek\Infrastructure\Tests\Mocups\DummyClassForStr;
use Gek\Infrastructure\Tests\Mocups\DummyClassWithToStringForStr;
use PHPUnit\Framework\TestCase;

/**
 * Class StrTest
 * @package Gek\Infrastructure\Tests
 */
class StrTest extends TestCase
{


    #region Str::format

    /**
     * Str::format temel test
     */
    public function testFormatBasic(): void
    {
        $this->assertEquals(
            'deneme bir, iki',
            Str::format('deneme {0}, {1}', 'bir', 'iki')
        );
    }

    /**
     * Str::format kaçış karakterleri testi
     */
    public function testFormatEscaped(): void
    {
        $this->assertEquals(
            'deneme bir, iki - {',
            Str::format('deneme {0}, {1} - {{', 'bir', 'iki')
        );

        $this->assertEquals(
            'deneme bir, iki - }',
            Str::format('deneme {0}, {1} - }}', 'bir', 'iki')
        );

        $this->assertEquals(
            'deneme bir, iki - {0}',
            Str::format('deneme {0}, {1} - {{0}}', 'bir', 'iki')
        );

        $this->assertEquals(
            'deneme bir, iki - {bir}',
            Str::format('deneme {0}, {1} - {{{0}}}', 'bir', 'iki')
        );

        $this->assertEquals(
            'deneme bir, iki - {bir} {{iki}}',
            Str::format('deneme {0}, {1} - {{{0}}} {{{{{1}}}}}', 'bir', 'iki')
        );
    }

    /**
     * Str::format params yerine array kullanımı testi
     */
    public function testFormatArrayParams(): void
    {
        $this->assertEquals(
            'deneme bir, iki',
            Str::format('deneme {0}, {1}', array('bir', 'iki'))
        );
    }

    /**
     * Str::format karmaşık (karışık ve çoklu) kullanım testi
     */
    public function testFormatComplexOrder(): void
    {
        $this->assertEquals(
            'deneme bir, iki test iki bir',
            Str::format('deneme {0}, {1} test {1} {0}', 'bir', 'iki')
        );
    }

    /**
     * Str::format çok uzun parametre sayısı testi
     */
    public function testFormatVeryLongParameterCount(): void
    {
        $format = "";
        $checkString = "";
        $argumentArr = array();

        for ($i = 0; $i < 5000; $i++) {
            $format .= "{" . $i . "}-";
            $checkString .= "arg" . $i . "-";
            $argumentArr[] = "arg" . $i;
        }
        $this->assertEquals(
            $checkString,
            Str::format($format, $argumentArr)
        );

    }


    public function testFormatArgumentTypes(): void
    {
        $this->assertEquals(
            'deneme true, false',
            Str::format('deneme {0}, {1}', true, false)
        );
        $this->assertEquals(
            'deneme 456, 7854',
            Str::format('deneme {0}, {1}', 456, 7854)
        );
        $this->assertEquals(
            'deneme 5.454, 456456.785',
            Str::format('deneme {0}, {1}', 5.454, 456456.785)
        );
        $this->assertEquals(
            'deneme 5.454, [array]',
            Str::format('deneme {0}, {1}', 5.454, array())
        );
        $this->assertEquals(
            'deneme 5.454, [stdClass]',
            Str::format('deneme {0}, {1}', 5.454, (object)array())
        );

        $dmyCls = new DummyClassForStr();
        $this->assertEquals(
            'deneme 5.454, [Gek\Infrastructure\Tests\Mocups\DummyClassForStr]',
            Str::format('deneme {0}, {1}', 5.454, $dmyCls)
        );

        $dmyClsTs = new DummyClassWithToStringForStr(5, 'test');
        $this->assertEquals(
            'deneme 5.454, foo: 5 bar: test',
            Str::format('deneme {0}, {1}', 5.454, $dmyClsTs)
        );

        $dt = new \DateTime('2020-05-03 20:39:28');
        $this->assertEquals(
            'deneme 5.454, 2020-05-03 20:39:28',
            Str::format('deneme {0}, {1}', 5.454, $dt)
        );
    }

    #endregion Str::format

    #region Str::ensureMaximumLength

    public function testEnsureMaximumLengthBasic(): void
    {

        $this->assertEquals(
            'Atakan Çalışkan',
            Str::ensureMaximumLength('Atakan Çalışkan dşği asdasdasd', 15)
        );
        $this->assertEquals(
            '国推震山記密企退受及以屈内新士選',
            Str::ensureMaximumLength('国推震山記密企退受及以屈内新士選。語記末正作酵際滴田掲組宿', 16)
        );

    }

    #endregion Str::ensureMaximumLength

    #region Str::toLowerCase

    public function testToLowerCaseBasic(): void
    {
        $engText = 'This is an English sentence.';
        $this->assertEquals(
            'this is an english sentence.',
            Str::toLowerCase($engText)
        );
        $dummyLangext = 'TYKjkh TYYUYUB JHKDAF';
        $this->assertEquals(
            'tykjkh tyyuyub jhkdaf',
            Str::toLowerCase($dummyLangext)
        );
    }

    public function testToLowerCaseEncoding(): void
    {
        $testText = 'ĞÜŞÇ';
        $this->assertEquals(
            'ğüşç',
            Str::toLowerCase($testText, 'UTF-8')
        );
        $this->assertNotEquals(
            'ğüşç',
            Str::toLowerCase($testText, 'ASCII')
        );
        $this->assertEquals(
            '????????',
            Str::toLowerCase($testText, 'ASCII')
        );

    }

    public function testToLowerCaseLanguage(): void
    {
        $testText = 'Işık Ilık İGİNÇ İgi XWQ';
        $resText = 'ışık ılık iginç igi xwq';
        $resTextAscii = '???k ?l?k igin? igi xwq';

        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,null,'tr')
        );

        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,'UTF-8','tr')
        );
        $this->assertEquals(
            $resTextAscii,
            Str::toLowerCase($testText,'ASCII','tr')
        );

        $testText = 'I IS WQX a i Is';
        $resText = 'i is wqx a i is';

        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,null,'en')
        );

        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,'UTF-8','en')
        );
        $this->assertEquals(
            $resText,
            Str::toLowerCase($testText,'ASCII','en')
        );



    }

    #endregion Str::toLowerCase

    #region Str::toUpperCase

    public function testToUpperCaseBasic(): void
    {
        $engText = 'This is an English sentence.';
        $this->assertEquals(
            'THIS IS AN ENGLISH SENTENCE.',
            Str::toUpperCase($engText)
        );
        $dummyLangext = 'TYKjkh df jakl';
        $this->assertEquals(
            'TYKJKH DF JAKL',
            Str::toUpperCase($dummyLangext)
        );
    }

    public function testToUpperCaseEncoding(): void
    {
        $testText = 'ğüşç';
        $this->assertEquals(
            'ĞÜŞÇ',
            Str::toUpperCase($testText, 'UTF-8')
        );
        $this->assertNotEquals(
            'ĞÜŞÇ',
            Str::toUpperCase($testText, 'ASCII')
        );
        $this->assertEquals(
            '????????',
            Str::toUpperCase($testText, 'ASCII')
        );

    }

    public function testToUpperCaseLanguage(): void
    {
        $testText = 'Işık Ilık ilginç İlgi xwq';
        $resText = 'IŞIK ILIK İLGİNÇ İLGİ XWQ';
        $resTextAscii = 'I?IK ILIK ?LG?N? ?LG? XWQ';

        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,null,'tr')
        );

        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,'UTF-8','tr')
        );
        $this->assertEquals(
            $resTextAscii,
            Str::toUpperCase($testText,'ASCII','tr')
        );

        $testText = 'i IS is wqx a i Is';
        $resText = 'I IS IS WQX A I IS';

        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,null,'en')
        );

        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,'UTF-8','en')
        );
        $this->assertEquals(
            $resText,
            Str::toUpperCase($testText,'ASCII','en')
        );



    }

    #endregion Str::toUpperCase

    #region Str::toTitleCase

    public function testToTitleCaseBasic(): void
    {
        $engText = 'this is an English sentence.';
        $this->assertEquals(
            'This Is An English Sentence.',
            Str::toTitleCase($engText)
        );
        $dummyLangext = 'tYKjkh df JAkl';
        $this->assertEquals(
            'Tykjkh Df Jakl',
            Str::toTitleCase($dummyLangext)
        );
        $engText = 'this IS An English sentence. this is AnOther Sentence.';
        $this->assertEquals(
            'This Is An English Sentence. This Is Another Sentence.',
            Str::toTitleCase($engText)
        );
        $this->assertEquals(
            '',
            Str::toTitleCase('')
        );
        $this->assertEquals(
            '   
            ',
            Str::toTitleCase('   
            ')
        );
        $this->assertEquals(
            " \t \n",
            Str::toTitleCase(" \t \n")
        );
    }

    public function testToTitleCaseEncoding(): void
    {
        $testText = 'ğüşç';
        $this->assertEquals(
            'Ğüşç',
            Str::toTitleCase($testText, 'UTF-8')
        );
        $this->assertNotEquals(
            'Ğüşç',
            Str::toTitleCase($testText, 'ASCII')
        );
        $this->assertEquals(
            '????????',
            Str::toTitleCase($testText, 'ASCII')
        );

    }

    public function testToTitleCaseLanguage(): void
    {

        $testText = 'bu tÜRkçe bir cümLedİr. deĞİl mi?';
        $resText = 'Bu Türkçe Bir Cümledir. Değil Mi?';
        $resTextAscii = 'Bu T?rk?e Bir C?mledir. De?il Mi?';

        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,null,'tr')
        );

        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,'UTF-8','tr')
        );
        $this->assertEquals(
            $resTextAscii,
            Str::toTitleCase($testText,'ASCII','tr')
        );

        $testText = 'i IS is wqx a i Is';
        $resText = 'I Is Is Wqx A I Is';

        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,null,'en')
        );

        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,'UTF-8','en')
        );
        $this->assertEquals(
            $resText,
            Str::toTitleCase($testText,'ASCII','en')
        );

    }

    #endregion Str::toTitleCase

    #region Str::ucFirst

    public function testUcFirst(){
        $this->assertEquals(
            'I love this GaMe.',
            Str::ucFirst('i love this GaMe.')
        );
        $this->assertEquals(
            'I love this GaMe.',
            Str::ucFirst('i love this GaMe.', 'ASCII')
        );
        $this->assertEquals(
            'I love this GaMe.',
            Str::ucFirst('i love this GaMe.', 'ASCII', 'en')
        );
        $this->assertNotEquals(
            'I love this GaMe.',
            Str::ucFirst('i love this GaMe.', 'UTF-8', 'tr')
        );
        $this->assertEquals(
            'Işık ile başlayan bu cümlenin İlk harFi büyük Olacak.',
            Str::ucFirst('ışık ile başlayan bu cümlenin İlk harFi büyük Olacak.',null,'tr')
        );
        $this->assertEquals(
            'İç dış ile başlayan bu cümlenin İlk harFi büyük Olacak.',
            Str::ucFirst('iç dış ile başlayan bu cümlenin İlk harFi büyük Olacak.','UTF-8','tr')
        );
        $this->assertEquals(
            '?? d?? ile ba?layan bu c?mlenin ?lk harFi b?y?k Olacak.',
            Str::ucFirst('iç dış ile başlayan bu cümlenin İlk harFi büyük Olacak.','ASCII','tr')
        );
    }

    #endregion Str::ucFirst


}
