<?php


namespace Gek\Infrastructure\Tests;


use Gek\Infrastructure\ClassHelper;
use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Tests\Mocups\DummyReflectableClass;
use PHPUnit\Framework\TestCase;
use ReflectionClass;
use ReflectionFunction;
use ReflectionObject;


/**
 * Class ClassHelperTest
 * @package Gek\Infrastructure\Tests
 */
class ClassHelperTest extends TestCase
{

    public function testGetDocCommentObject(): void
    {
        $reflectClass = $this->getReflectedClass();

        // class doc comments
        $classDocCom = ClassHelper::getDocCommentObject($reflectClass);

        $this->assertEquals(
            $reflectClass->getDocComment(),
            $classDocCom->getDocComment()
        );
        // property
        $propReflect = $reflectClass->getProperty('strProp');
        $propDocCom = ClassHelper::getDocCommentObject($propReflect);
        $this->assertEquals(
            $propReflect->getDocComment(),
            $propDocCom->getDocComment()
        );

        // const
        $constReflect = $reflectClass->getReflectionConstant('SAMPLE_CONST');
        $constDocCom = ClassHelper::getDocCommentObject($constReflect);
        $this->assertEquals(
            $constReflect->getDocComment(),
            $constDocCom->getDocComment()
        );

        // method
        $methodReflect = $reflectClass->getMethod('sampleFn');
        $methodDocCom = ClassHelper::getDocCommentObject($methodReflect);
        $this->assertEquals(
            $methodReflect->getDocComment(),
            $methodDocCom->getDocComment()
        );

        // function
        $fnReflect = $this->getReflectedFn();
        $fnDocCom = ClassHelper::getDocCommentObject($fnReflect);
        $this->assertEquals(
            $fnReflect->getDocComment(),
            $fnDocCom->getDocComment()
        );

        // object
        $obj = new DummyReflectableClass();
        $objectReflect = new ReflectionObject($obj);
        $objectDocCom = ClassHelper::getDocCommentObject($objectReflect);

        $this->assertEquals(
            $objectReflect->getDocComment(),
            $objectDocCom->getDocComment()
        );

        $this->expectException(InvalidArgumentException::class);
        $invalidReflector =  $methodReflect->getParameters()[0];
        $invalidDocCom = ClassHelper::getDocCommentObject($invalidReflector);

    }

    public function testGetMetaObject():void {
        $obj = new DummyReflectableClass();

        $this->assertEquals(
            (object)[
                'strProp' => 'strProp'
            ],
            ClassHelper::getMetaObject($obj)
        );
        $obj->appendProp = 10;
        $this->assertNotEquals(
            (object)[
                'strProp' => 'strProp'
            ],
            ClassHelper::getMetaObject($obj)
        );
        $this->assertEquals(
            (object)[
                'strProp' => 'strProp',
                'appendProp' => 'appendProp'
            ],
            ClassHelper::getMetaObject($obj)
        );

        $stdObj = new \stdClass();
        $stdObj->propA = 456;
        $stdObj->propB = 5.78;
        $stdObj->propC = true;

        $this->assertEquals(
            (object)[
                'propA' => 'propA',
                'propB' => 'propB',
                'propC' => 'propC'
            ],
            ClassHelper::getMetaObject($stdObj)
        );

        $arrayObject = (object)array(
            'propD' => 456,
            'propE' => false,
            'propF' => 4.8,
        );

        $this->assertEquals(
            (object)[
                'propD' => 'propD',
                'propE' => 'propE',
                'propF' => 'propF'
            ],
            ClassHelper::getMetaObject($arrayObject)
        );

    }

    public function testGetMetaObjectClassName():void {

        $this->assertEquals(
            (object)[
                'strProp' => 'strProp'
            ],
            ClassHelper::getMetaObjectClassName(DummyReflectableClass::class)
        );

        $this->expectException(\ReflectionException::class);
        ClassHelper::getMetaObjectClassName('invalid class name');

    }

    public function testReflectObject():void {
        $stdObj = new \stdClass();
        $stdObj->propA = 1;
        $reflectStdObj = new ReflectionObject($stdObj);


        $this->assertEquals(
            $reflectStdObj,
            ClassHelper::reflectObject($stdObj)
        );
        $this->assertEquals(
            $reflectStdObj->getProperties(),
            ClassHelper::reflectObject($stdObj)->getProperties()
        );

        $stdObj->propB = 2;
        $this->assertEquals(
            $reflectStdObj->getProperties(),
            ClassHelper::reflectObject($stdObj)->getProperties()
        );

    }

    public function testReflectClass():void {
        $reflectClass = new ReflectionClass(DummyReflectableClass::class);
        $this->assertEquals(
            $reflectClass,
            ClassHelper::reflectClass(DummyReflectableClass::class)
        );
        $this->expectException(\ReflectionException::class);
        ClassHelper::reflectClass('invalid class name');
    }


    /**
     * @return ReflectionClass
     * @throws \ReflectionException
     */
    private function getReflectedClass(): ReflectionClass{
         return new ReflectionClass(DummyReflectableClass::class);
    }

    /**
     * @return ReflectionFunction
     * @throws \ReflectionException
     */
    private function getReflectedFn(): ReflectionFunction{
        return new ReflectionFunction('dummyFn');
    }
}
