<?php


namespace Gek\Infrastructure\Reflections;

use function mb_strtolower;

/**
 * Class DocCommentObject
 * @package Gek\Infrastructure\Reflections
 */
class DocCommentObject
{

    #region fields

    /**
     * @var string|null
     */
    protected ?string $rawDocComment;

    /**
     * @var array
     */
    protected array $comments = array();

    /**
     * @var array
     */
    protected array $keys = array();

    #endregion fields

    #region ctor

    /**
     * DocCommentObject constructor.
     * @param string|null $docComment
     */
    public function __construct(?string $docComment)
    {
        $this->setDocComment($docComment);
    }

    #endregion ctor

    #region methods

    /**
     * @param string $key
     * @return string|bool|null
     */
    public function getKey(string $key)
    {
        $key = $this->fixKey($key);
        if (isset($this->keys[$key])) {
            return $this->keys[$key];
        }
        return null;
    }

    /**
     * @return array
     */
    public function getComments(): array
    {
        return $this->comments;
    }

    /**
     * @return string|null
     */
    public function getDocComment(): ?string
    {
        return $this->rawDocComment;
    }

    /**
     * @param string|null $docComment
     */
    public function setDocComment(?string $docComment):void{
        $this->rawDocComment = $docComment;
        if(empty(trim($docComment))){
            $this->keys = array();
            $this->comments = array();
            return;
        }
        $fixedEndLine = str_replace(chr(10), PHP_EOL, $docComment);
        $keys = $this->readAllKeys($fixedEndLine);
        $commentLines = array();
        $fixedEndLine = str_replace($keys, '', $fixedEndLine);
        $lines = explode(PHP_EOL, $fixedEndLine);
        foreach ($lines as $line) {
            $line = trim($line," \t\n\r\0\x0B*/");
            if (!empty($line) && !in_array($line, array('*', '/**', '/*', '*/'))) {
                $commentLines[] = $line;
            }
        }
        $this->keys = $this->parseKeysData($keys);
        $this->comments = $commentLines;
    }

    /**
     * @return array
     */
    public function getKeys(): array
    {
        return $this->keys;
    }

    #endregion methods

    #region utils

    /**
     * @param string $key
     * @return string
     */
    private function fixKey(string $key): string
    {
        return mb_strtolower(str_replace('@', '', trim($key)));
    }

    /**
     * @param string $docComment
     * @return array<string>
     */
    private function readAllKeys(string $docComment): array
    {
        $keys = array();
        $flag = 0;
        $currentTerm = '';
        $prev = '';

        for ($i = 0; $i < strlen($docComment); $i++) {
            $char = $docComment[$i];
            if ($char == chr(10)) {
                $char = PHP_EOL;
            }

            if ($char == '@') {
                if ($flag == 1) {
                    $keys[] = trim($currentTerm, " \t\n\r\0\x0B*");
                    $currentTerm = $char;
                } else {
                    $flag = 1;
                    $currentTerm .= $char;
                }
            } elseif ($char == '/' && $prev == '*') {
                if ($flag == 1) {
                    $keys[] = trim(substr($currentTerm, 0, -1), " \t\n\r\0\x0B*");
                    $currentTerm = '';
                    $flag = 0;
                }
            } else {
                if ($flag == 1) {
                    $currentTerm .= $char;
                }
            }
            $prev = $char;
        }
        if ($flag == 1) {
            $keys[] = trim($currentTerm, " \t\n\r\0\x0B*");
        }
        return $keys;
    }

    /**
     * @param array<string> $allKeys
     * @return array
     */
    private function parseKeysData(array $allKeys)
    {
        $resArray = array();
        foreach ($allKeys as $currentKey) {
            $keyEndPos = mb_strpos($currentKey, ' ');
            if ($keyEndPos !== false) {
                $key = str_replace('@', '', mb_substr($currentKey,0,$keyEndPos));
                $value = trim(mb_substr($currentKey,$keyEndPos));
                $resArray[$key] = $value;
            }else{
                $key = str_replace('@', '', $currentKey);
                $resArray[$key] = true;
            }
        }
        return $resArray;
    }

    #endregion utils

    #region static
    /**
     * @param string|null $docComment
     * @return static
     */
    public static function parse(?string $docComment):self {
        static $instance = null;
        if($instance === null){
            $instance = new self(null);
        }
        $newInstance = clone $instance;
        $newInstance->setDocComment($docComment);
        return $newInstance;
    }

    #endregion static

}
