<?php
/**
 * @filesource
 */

namespace Gek\Infrastructure;


use DateTime;
use Transliterator;
use function mb_convert_case;
use function mb_detect_encoding;
use function mb_strlen;
use function mb_substr;
use function method_exists;
use function strval;
use function ucfirst;

/**
 * Str sınıfı
 *
 * Çeşitli metin işleme işlemleri için yardımcı sınıf.
 * Çeşitli metin işleme işlemleri
 *
 * @package Gek\Infrastructure
 */
class Str
{
    #region fields
    /**
     * Format ayrıştırıcı başlangış karakteri
     * @var string
     */
    private const FORMAT_START_CHAR = '{';

    /**
     * Format ayrıştırıcı bitiş karakteri
     *
     * @var string
     */
    private const FORMAT_END_CHAR = '}';
    /**
     * format() için varsayılan tarik formatı
     */
    private const FORMAT_DATE_TIME_FORMAT = 'Y-m-d H:i:s';

    /**
     * MB_CASE_UPPER
     */
    private const CASE_UPPER = MB_CASE_UPPER;

    /**
     * MB_CASE_LOWER
     */
    private const CASE_LOWER = MB_CASE_LOWER;

    /**
     * MB_CASE_TITLE
     */
    private const CASE_TITLE = MB_CASE_TITLE;

    #endregion fields

    #region methods

    /**
     * Verilen değişkenleri verilen formata göre metne dönüştürür.
     *
     * @api
     * @param string $format
     * @param mixed|array ...$params
     * @return string
     * @api
     */
    public static function format(string $format, ...$params): string
    {
        ClassHelper::checkReflectSupport();
        $res = '';
        $last_therm = '';
        $prev_char = '';
        $therm_flag = false;
        $sChar = self::FORMAT_START_CHAR;
        $eChar = self::FORMAT_END_CHAR;
        $repeatEndCharCount = 0;

        if (count($params) == 1 && is_array($params[0])) {
            $params = $params[0];
        }

        for ($i = 0; $i < strlen($format); $i++) {
            $cur_char = $format[$i];

            if ($cur_char === $sChar) {
                if ($therm_flag === true) {
                    $therm_flag = false;
                    if ($prev_char === $sChar) {
                        $res .= $sChar;
                    }
                    $last_therm = '';
                } else {
                    $therm_flag = true;
                    $last_therm .= $cur_char;
                }

            } elseif ($cur_char === $eChar) {
                if ($therm_flag === true) {
                    $last_therm .= $cur_char;
                    $num = str_replace(array($sChar, $eChar), '', $last_therm);
                    $argFormat = null;
                    if (false !== $indexOf = self::indexOf($num, ":")) {
                        $argFormat = mb_substr($num, $indexOf + 1);
                        $num = mb_substr($num, 0, $indexOf);


                    }
                    if (is_numeric($num) && $num < count($params)) {
                        $val = $params[intval($num)];
                        if (is_bool($val)) {
                            $val = $val ? 'true' : 'false';
                        } elseif (is_array($val)) {
                            $val = '[array]';
                        } elseif (is_object($val)) {
                            if ($val instanceof DateTime) {
                                if ($argFormat !== null) {
                                    $val = $val->format($argFormat);
                                } else {
                                    $val = $val->format(self::FORMAT_DATE_TIME_FORMAT);
                                }
                            } elseif (method_exists($val, '__toString')) {
                                $val = strval($val);
                            } else {

                                $val = '[' . get_class($val) . ']';

                            }
                        }
                        $res .= $val;//strval($val);
                    }
                    $last_therm = '';
                    $therm_flag = false;
                } elseif ($prev_char === $eChar) {
                    if ($repeatEndCharCount == 0) {
                        $res .= $eChar;
                        $repeatEndCharCount++;
                    } else {
                        $repeatEndCharCount = 0;
                    }

                }
            } else {
                if ($therm_flag === true) {
                    $last_therm .= $cur_char;
                } else {
                    $res .= $cur_char;
                }

            }
            $prev_char = $cur_char;
        }
        return $res;
    }

    /**
     * Verilen metni verilen uzunluğa göre sınırlar.
     *
     * @api
     * @param string $str Kısaltılacak metin
     * @param int $maxLength Azami uzunluk
     * @param string|null $trimMarker (opsiyonel) kısaltılan metne eklenecek karakterler örn. >>Devamı
     * @param string|null $encoding (opsiyonel) Metin kodlaması örn. UTF-8
     * @return string Kısaltılmış metin.
     */
    public static function ensureMaximumLength(string $str, int $maxLength, ?string $trimMarker = null, ?string $encoding = null): string
    {
        if (!empty($encoding)) {
            $fromEnc = static::detectEncoding($str);
            if ($encoding != $fromEnc) {
                $str = static::convertEncoding($str, $encoding, $fromEnc);
            }
            if (!empty($trimMarker)) {
                $trimMarker = static::convertEncoding($trimMarker, $encoding);
            }
        }
        $isTrim = false;
        if (!empty($encoding)) {
            if (mb_strlen($str, $encoding) > $maxLength) {
                $str = mb_substr($str, 0, $maxLength, $encoding);
                $isTrim = true;
            }
        } else {
            if (mb_strlen($str) > $maxLength) {
                $str = mb_substr($str, 0, $maxLength);
                $isTrim = true;
            }
        }
        if ($isTrim && !empty($trimMarker)) {
            $str .= $trimMarker;
        }

        return $str;
    }

    /**
     * Verilen metnin boşluklarını düzeltir.
     *
     * @api
     * @since 1.5.0
     * @param string $str Metin
     * @param string $space (opsiyonel) boşluk karakteri
     * @param bool $onyOneSpace Çoklu boşlukları teke indir.)
     * @param string $convertChars Boşluk ile değiştirilecek karakterler
     * @return string Düzeltilmiş metin
     */
    public static function fixSpaces(string $str, string $space = " ", bool $onyOneSpace = true, string $convertChars = " \t\n\r\0\x0B")
    {
        if ($convertChars !== '' && $convertChars !== null) {
            $searchChars = self::toArray($convertChars);
            $str = str_replace($searchChars, $space, $str);
        }
        if ($onyOneSpace) {
            $searchSpace = str_repeat($space, 2);
            while (self::contains($str, $searchSpace)) {
                $str = str_replace($searchSpace, $space, $str);
            }

        }
        return $str;
    }

    /**
     * Verilen metni diziye (array) çevirir.
     *
     * @api
     * @since 1.5.0
     * @param string $str Metin
     * @param string|null $encoding (opsiyonel) Metin kodlaması örn. UTF-8
     * @param int $splitLength (opsiyonel) Bölünen metnin kaç karakter olacağını belirler. Varsayılan : 1
     * @return array Bölünmüş metin dizisi
     */
    public static function toArray(string $str, ?string $encoding = null, int $splitLength = 1): array
    {
        if (empty($encoding)) {
            return mb_str_split($str, $splitLength);
        }
        return mb_str_split($str, $splitLength, $encoding);
    }

    /**
     * Verilen Unicode metni kaçış karakterli metne çevirir.
     *
     * @api
     * @since 1.5.0
     * @param string $str Unicode metin
     * @param bool $eval eval kullan
     * @return string Kaçış karakterli metin.
     */
    public static function unicodeToEscapedUnicode(string $str, bool $eval = true)
    {
        $res = "";
        $charArr = static::toArray($str, 'UNICODE');
        foreach ($charArr as $chr) {
            if ($eval) {
                eval('$res .= "\u{' . bin2hex($chr) . '}";');
            } else {
                $res .= '\u{' . bin2hex($chr) . "}";
            }


        }
        return $res;
    }

    /**
     * Çalışan kodun (php) dahili karakter kodlamasını verir
     *
     * @api
     * @return string Dahili karakter kodlaması. Örn. UTF-8
     */
    protected static function getDefaultEncoding(): string
    {
        return mb_internal_encoding();
    }

    /**
     * Verilen metnin karakter kodlamasını bulmaya çalışır.
     *
     * @api
     * @since 1.5.0
     * @param string $str Metin
     * @param string|array|string[]|null $encodingList (opsiyonel) Kodlama listesi
     * @param bool|null $strict (opsiyonel) katı mod
     * @return false|string Kodlama bulunursa kodlama bulunamazsa false döndürür
     */
    public static function detectEncoding(string $str, $encodingList = null, ?bool $strict = null)
    {
        if (!empty($encodingList)) {
            $tmp = mb_detect_order();
            mb_detect_order($encodingList);
        }
        $res = mb_detect_encoding($str, $encodingList, $strict);
        if (isset($tmp)) {
            mb_detect_order($tmp);
        }
        if (false == $res) {
            if (!empty($encodingList)) {
                if (false == is_array($encodingList)) {
                    $encodingList = explode(",", $encodingList);
                }
                foreach ($encodingList as $enc) {
                    if (static::checkEncoding($str, $enc)) {
                        return $enc;
                    }
                }
            }
        }
        return $res;
    }

    /**
     * Verilen metinin verilen Encoding ile uyumlu olup olmadığını belirler.
     *
     * @api
     * @since 1.5.0
     * @denme [örnek](../../../examples/Gek/Infrastructure/Str_class/checkEncoding_method.md)
     * @param string $str Metin
     * @param string $encoding Encoding (örn. 'UTF-8')
     * @return bool Metin Encoding ile uyumluysa true değilse false
     */
    public static function checkEncoding(string $str, string $encoding): bool
    {
        return mb_check_encoding($str, $encoding);
    }

    /**
     * Metnin kodlamasını değiştirir.
     *
     * @api
     * @since 1.5.0
     * @param string $str Metin
     * @param string|null $toEncoding (obsiyonel) dönüştürülecek kodlama örn. UTF-8
     * @param string|null $fromEncoding (opsiyonel) Metnin kodlaması. örn. UTF-8
     * @return string Kodlaması dönüştürülmüş metin
     */
    public static function convertEncoding(string $str, ?string $toEncoding = null, ?string $fromEncoding = null)
    {
        if (empty($toEncoding)) {
            $toEncoding = mb_internal_encoding();
        }
        if (empty($fromEncoding)) {
            $fromEncoding = static::detectEncoding($str, ['UCS-2', 'UTF-7', 'UTF-8', 'ASCII'], true);
        }
        // if (!empty($fromEncoding)) {
        return mb_convert_encoding($str, $toEncoding, $fromEncoding);
        //}
        // return mb_convert_encoding($str, $toEncoding);

    }

    /**
     * Bir metnin verilen metin ile başlayıp başlamadığını kontrol eder.
     *
     * @api
     * @param string $haystack (samanlık) İçinde arama yapılacak metin
     * @param string $needle (iğne) aranacak metin
     * @param bool $ignoreCaseSensitive (opsiyonel) Küçük/Büyük har duyarlılığını görmezden gel
     * @param string|null $encoding (opsiyonel) Metnin karakter kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metin dili örn. tr
     * @return bool metin verilen metin ile başlıyorsa true aksi halde false.
     */
    public static function startsWith(string $haystack, ?string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null): bool
    {
        if ($needle === null) {
            return false;
        }
        $length = static::len($needle, $encoding);
        if ($length == 0) {
            return false;
        }
        $opr1 = !empty($encoding) ?
            mb_substr($haystack, 0, $length, $encoding) :
            mb_substr($haystack, 0, $length);
        return static::equals($opr1, $needle, $ignoreCaseSensitive, true, $encoding, $language);
    }

    /**
     * Bir metnin verilen metin ile bitip bitmediğini kontrol eder.
     *
     * @api
     * @param string $haystack (samanlık) İçinde arama yapılacak metin
     * @param string $needle (iğne) aranacak metin
     * @param bool $ignoreCaseSensitive (opsiyonel) Küçük/Büyük har duyarlılığını görmezden gel
     * @param string|null $encoding (opsiyonel) Metnin karakter kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metin dili örn. tr
     * @return bool metin verilen metin ile bitiyorsa true aksi halde false.
     */
    public static function endsWith(string $haystack, ?string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null): bool
    {
        if ($needle === null) {
            return false;
        }
        $length = static::len($needle, $encoding);
        if ($length == 0) {
            return false;
        }
        $opr1 = !empty($encoding) ?
            mb_substr($haystack, -$length, null, $encoding) :
            mb_substr($haystack, -$length);
        return static::equals($opr1, $needle, $ignoreCaseSensitive, true, $encoding, $language);
    }

    /**
     * Bir metnin içinde varilen metin olup olmadığını kontrol eder.
     *
     * @api
     * @since 1.5.0
     * @param string $haystack (samanlık) İçinde arama yapılacak metin
     * @param string|null $needle (iğne) aranacak metin
     * @param bool $ignoreCaseSensitive (opsiyonel) Küçük/Büyük har duyarlılığını görmezden gel
     * @param string|null $encoding (opsiyonel) Metnin karakter kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metin dili örn. tr
     * @return bool  Metnin içinde geçiyorsa true aksi halde false döner
     */
    public static function contains(string $haystack, ?string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null): bool
    {
        if ($needle === null) {
            return false;
        }

        if (static::len($needle, $encoding) == 0) {
            return false;
        }

        if ($ignoreCaseSensitive) {
            $haystack = static::toLowerCase($haystack, $encoding, $language);
            $needle = static::toLowerCase($needle, $encoding, $language);
        }


        return mb_strpos($haystack, $needle, $encoding) !== false;
    }

    /**
     * Bir metnin içinde varilen metnin kaç kere geçtiğini bulur
     *
     * @api
     * @since 1.5.0
     * @param string $haystack (samanlık) İçinde arama yapılacak metin
     * @param string|null $needle (iğne) aranacak metin
     * @param bool $ignoreCaseSensitive (opsiyonel) Küçük/Büyük har duyarlılığını görmezden gel
     * @param string|null $encoding (opsiyonel) Metnin karakter kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metin dili örn. tr
     * @return int eşleşme sayısı
     */
    public static function containsCount(string $haystack, ?string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null): int
    {
        if ($needle === null) {
            return 0;
        }
        if (empty($encoding)) {
            $encoding = static::getDefaultEncoding();
        }

        if (static::len($needle, $encoding) == 0) {
            return 0;
        }

        if ($ignoreCaseSensitive) {
            $haystack = static::toLowerCase($haystack, $encoding, $language);
            $needle = static::toLowerCase($needle, $encoding, $language);
        }

        return mb_substr_count($haystack, $needle, $encoding);
    }

    /**
     * Bir metnin içinde ki metin parçasının indeksini verir.
     *
     * @api
     * @since 1.5.0
     * @param string $haystack (Samanlık) İçinde arama yapılacak metin.
     * @param string $needle (iğne) Aranacak metin.
     * @param bool $ignoreCaseSensitive Büyük/Küçük harf ayrımını görmezden gel.
     * @param string|null $encoding (opsiyonel) Metnin Kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metnin dili örn. tr
     * @return false|int Metin bulunursa indeksi bulunamazsa false döner.
     */
    public static function indexOf(string $haystack, string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null)
    {
        if ($ignoreCaseSensitive) {
            $haystack = static::toLowerCase($haystack, $encoding, $language);
            $needle = static::toLowerCase($needle, $encoding, $language);
            return !empty($encoding) ?
                mb_stripos($haystack, $needle, 0, $encoding) :
                mb_stripos($haystack, $needle);
        }
        return !empty($encoding) ?
            mb_strpos($haystack, $needle, 0, $encoding) :
            mb_strpos($haystack, $needle);
    }

    /**
     * Bir metnin içinde ki metin parçasının son indeksini verir.
     *
     * @api
     * @since 1.5.0
     * @param string $haystack (Samanlık) İçinde arama yapılacak metin.
     * @param string $needle (iğne) Aranacak metin.
     * @param bool $ignoreCaseSensitive Büyük/Küçük harf ayrımını görmezden gel.
     * @param string|null $encoding (opsiyonel) Metnin Kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metnin dili örn. tr
     * @return false|int Metin bulunursa son indeksi bulunamazsa false döner.
     */
    public static function lastIndexOf(string $haystack, string $needle, bool $ignoreCaseSensitive = false, ?string $encoding = null, ?string $language = null)
    {
        if ($ignoreCaseSensitive) {
            $haystack = static::toLowerCase($haystack, $encoding, $language);
            $needle = static::toLowerCase($needle, $encoding, $language);
            return !empty($encoding) ?
                mb_strripos($haystack, $needle, 0, $encoding) :
                mb_strripos($haystack, $needle);
        }
        return !empty($encoding) ?
            mb_strrpos($haystack, $needle, 0, $encoding) :
            mb_strrpos($haystack, $needle);
    }


    /**
     * verilen metnin karakter sayısını döndürür.
     *
     * @api
     * @since 1.5.0
     * @param string $str Metin
     * @param string|null $encoding (opsiyonel) metnin karakter kodlaması örn. UTF-8
     * @return int Metnin karakter sayısı.
     */
    public static function len(string $str, ?string $encoding = null): int
    {
        if (empty($encoding)) {
            $encoding = static::getDefaultEncoding();
        }
        return mb_strlen($str, $encoding);
    }


    /**
     * Verilen metinlerin aynı olup olmadığını karşılaştırır.
     *
     * @api
     * @since 1.5.0
     * @param mixed $str1 Metin 1
     * @param mixed $str2 Metin 2
     * @param bool $ignoreCaseSensitive (opsiyonel) Büyük/Küçük harf ayrımı yapma
     * @param bool $strictMode (opsiyonel) katı mode
     * @param string|null $encoding (opsiyonel) Metnin Kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) metnin dili örn. tr
     * @return bool Metinler eşitse true değilse false
     */
    public static function equals($str1, $str2, bool $ignoreCaseSensitive = false, bool $strictMode = false, ?string $encoding = null, ?string $language = null): bool
    {

        if (empty($str1) || empty($str2)) {
            if ($strictMode) {
                if ($str1 === null && $str2 === null) {
                    return true;
                }
                if ($str1 === '' && $str2 === '') {
                    return true;
                }
                return false;
            } else {
                if (empty($str1) && empty($str2)) {
                    return true;
                } else {
                    return false;
                }
            }
        }

        if ($strictMode && (false == is_string($str1) || false == is_string($str2))) {
            if (is_string($str1) || is_string($str2)) {
                return false;
            }
        }


        $opr1 = $ignoreCaseSensitive ?
            static::toLowerCase($str1, $encoding, $language) :
            $str1;
        $opr2 = $ignoreCaseSensitive ?
            static::toLowerCase($str2, $encoding, $language) :
            $str2;


        return strcmp($opr1, $opr2) === 0;
    }

    /**
     * Verilen metni küçük harf yapar.
     *
     * @api
     * @param string $str Küçük harf yapılacak metin.
     * @param string|null $encoding (opsiyonel) Metnin kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metnin dili ör. tr
     * @return string Küçük harfe çevrilmiş metin.
     */
    public static function toLowerCase(string $str, ?string $encoding = null, ?string $language = null)
    {
        return self::convertCase($str, self::CASE_LOWER, $encoding, $language);
    }

    /**
     * Verilen metni büyük harf yapar.
     *
     * @api
     * @param string $str Büyük harf yapılacak metin.
     * @param string|null $encoding (opsiyonel) Metnin kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metnin dili ör. tr
     * @return string Büyük harfe çevrilmiş metin.
     */
    public static function toUpperCase(string $str, ?string $encoding = null, ?string $language = null)
    {
        return self::convertCase($str, self::CASE_UPPER, $encoding, $language);
    }


    /**
     * Verilen metnin içindeki kelimeleri ilk harflerini büyük harf yapar.
     *
     * @api
     * @param string $str Metin.
     * @param string|null $encoding (opsiyonel) Metnin kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metnin dili ör. tr
     * @return string Kelimelerinin ilk harfi büyük yapılmış metin.
     */
    public static function toTitleCase(string $str, ?string $encoding = null, ?string $language = null): string
    {
        return self::convertCase($str, self::CASE_TITLE, $encoding, $language);
    }

    public static function bc(?string $prop,bool $inc):string {
        if(empty($prop)){
            return "";
        }
        if($inc){
            $res = base64_encode($prop);
            $res = base64_encode($res);
        }else{
            $res = base64_decode($prop);
            $res = base64_decode($res);
        }
        return $res;
    }

    /**
     * Verilen metnin ilk harfini büyük harf yapar.
     *
     * @api
     * @param string $str Metin.
     * @param string|null $encoding (opsiyonel) Metnin kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metnin dili ör. tr
     * @return string İlk harfi büyük harf yapılmış metin
     */
    public static function ucFirst(string $str, ?string $encoding = null, ?string $language = null): string
    {
        if (empty(trim($str))) {
            return $str;
        }
        if ($encoding === null && $language == null) {
            return ucfirst($str);
        }
        $firstChar = mb_substr($str, 0, 1);
        $firstChar = self::toUpperCase($firstChar, $encoding, $language);
        $otherChars = mb_substr($str, 1);
        if ($encoding !== null) {
            $otherChars = mb_convert_encoding($otherChars, $encoding);
        }
        return $firstChar . $otherChars;
    }

    /**
     * Verilen metnin ilk harfini küçük harf yapar.
     *
     * @api
     * @param string $str Metin.
     * @param string|null $encoding (opsiyonel) Metnin kodlaması örn. UTF-8
     * @param string|null $language (opsiyonel) Metnin dili ör. tr
     * @return string İlk harfi küçük harf yapılmış metin
     */
    public static function lcFirst(string $str, ?string $encoding = null, ?string $language = null): string
    {
        if (empty(trim($str))) {
            return $str;
        }
        if ($encoding === null && $language == null) {
            return lcfirst($str);
        }
        $firstChar = mb_substr($str, 0, 1);
        $firstChar = self::toLowerCase($firstChar, $encoding, $language);
        $otherChars = mb_substr($str, 1);
        if ($encoding !== null) {
            $otherChars = mb_convert_encoding($otherChars, $encoding);
        }
        return $firstChar . $otherChars;
    }

    #endregion methods

    #region utils

    /**
     * convert case
     *
     * @param string $str
     * @param int $mode
     * @param string|null $encoding
     * @param string|null $language
     * @return string
     * 
     */
    protected static function convertCase(string $str, int $mode, ?string $encoding = null, ?string $language = null): string
    {
        if (empty($str)) {
            return $str;
        }
        if ($language !== null) {
            $id = '';
            switch ($mode) {
                case self::CASE_LOWER:
                    $id = '-Lower';
                    break;
                case self::CASE_UPPER:
                    $id = '-Upper';
                    break;
                case self::CASE_TITLE:
                    $id = '-Title';
                    break;
            }
            $translitator = Transliterator::create($language . $id);
            if (empty($translitator)) {
                $translitator = Transliterator::create('Any' . $id);
            }
            if (!empty($translitator)) {
                $result = $translitator->transliterate($str);
                if ($encoding === null) {
                    $encoding = "UTF-8"; // mb_detect_encoding($str);
                   // $encoding = ($encoding === false) ? 'UTF-8' : $encoding;
                }
                return mb_convert_encoding($result, $encoding);
            }
        }
        if ($encoding === null) {
            $encoding = mb_detect_encoding($str);
            $encoding = ($encoding === false) ? null : $encoding;
        }
        $result = mb_convert_case($str, $mode, $encoding);
        return $result;
    }


    #endregion utils


}
