<?php


namespace Gek\Infrastructure\Reflections;

use function mb_strtolower;

/**
 * DocCommentObject Sınıfı
 *
 * DocComentleri yapılandırılmış nesne olarak tutmak için kullanılır.
 *
 * @package Gek\Infrastructure\Reflections
 */
class DocCommentObject
{

    #region fields

    /**
     * DocComment ham metyin hali
     *
     * 
     * @var string|null
     */
    protected ?string $rawDocComment;

    /**
     * Özet (summary) sanırları
     *
     * 
     * @var array|string[]
     */
    protected array $summaryLines = array();

    /**
     * Açıklama satırları.
     *
     * @var array|string[]
     */
    protected array $comments = array();

    /**
     * Etiketler
     *
     * @var array|string[]
     */
    protected array $keys = array();

    #endregion fields

    #region ctor

    /**
     * DocCommentObject yapıcı method.
     * .
     * @param string|null $docComment docComment metni
     */
    public function __construct(?string $docComment)
    {
        $this->setDocComment($docComment);
    }

    #endregion ctor

    #region methods

    /**
     * docCommnt metnini set eder.
     *
     * @param string|null $docComment
     */
    public function setDocComment(?string $docComment): void
    {
        $this->rawDocComment = $docComment;
        if (empty(trim($docComment))) {
            $this->keys = array();
            $this->summaryLines = array();
            $this->comments = array();
            return;
        }
        $fixedEndLine = str_replace(chr(10), PHP_EOL, $docComment);
        $keys = $this->readAllKeys($fixedEndLine);
        $commentLines = array();
        $summaryLines = array();
        $summaryFlag = true;
        $fixedEndLine = str_replace($keys, '', $fixedEndLine);
        $lines = explode(PHP_EOL, $fixedEndLine);
        foreach ($lines as $line) {
            $line = trim($line, " \t\n\r\0\x0B*/");

            if (!empty($line) && !in_array($line, array('*', '/**', '/*', '*/'))) {
                if ($summaryFlag) {
                    $summaryLines[] = $line;
                } else {
                    $commentLines[] = $line;
                }

            } elseif ($summaryFlag && !empty($summaryLines)) {
                $summaryFlag = false;
            }
        }
        $this->keys = $this->parseKeysData($keys);
        $this->comments = $commentLines;
        $this->summaryLines = $summaryLines;
    }


    /**
     * verilen etiketin değerini döndürür.
     *
     * @param string $key etiket (tag)
     * @return string|bool|null|array etiket mevcutsa değeri aksi halde null
     */
    public function getKey(string $key)
    {
        $key = $this->fixKey($key);
        if (isset($this->keys[$key])) {
            return $this->keys[$key];
        }
        return null;
    }

    /**
     * bütün etiketleri (tags) döndürür.
     *
     * @return array
     */
    public function getKeys(): array
    {
        return $this->keys;
    }

    /**
     * Açıklama satırlarını döndürür.
     *
     * @return array açıklama satırları dizisi
     */
    public function getComments(): array
    {
        return $this->comments;
    }

    /**
     * Özet satırlarını döndürür.
     *
     * @return array özet satırları dizisi.
     */
    public function getSummary(): array
    {
        return $this->summaryLines;
    }

    /**
     * docComeent metnini döndürür.
     *
     * @return string|null docComment metni
     */
    public function getDocComment(): ?string
    {
        return $this->rawDocComment;
    }


    #endregion methods

    #region utils


    /**
     * bğtğn etigetleri oku
     *
     * 
     * @param string $docComment docComment metni
     * @return array|string[} etiket dizisi
     */
    private function readAllKeys(string $docComment): array
    {
        $keys = array();
        $flag = 0;
        $currentTerm = '';
        $prev = '';

        for ($i = 0; $i < strlen($docComment); $i++) {
            $char = $docComment[$i];
            if ($char == chr(10)) {
                $char = PHP_EOL;
            }

            if ($char == '@') {
                if ($flag == 1) {
                    $keys[] = trim($currentTerm, " \t\n\r\0\x0B*");
                    $currentTerm = $char;
                } else {
                    $flag = 1;
                    $currentTerm .= $char;
                }
            } elseif ($char == '/' && $prev == '*') {
                if ($flag == 1) {
                    $keys[] = trim(substr($currentTerm, 0, -1), " \t\n\r\0\x0B*");
                    $currentTerm = '';
                    $flag = 0;
                }
            } else {
                if ($flag == 1) {
                    $currentTerm .= $char;
                }
            }
            $prev = $char;
        }
        if ($flag == 1) {
            $keys[] = trim($currentTerm, " \t\n\r\0\x0B*");
        }
        return $keys;
    }

    /**
     * etiketlerin değerlerini ayrıştır.
     *
     * 
     * @param array|string[] $allKeys ham etiketler
     * @return array değerleri ayrıştırılmış etiketler
     */
    private function parseKeysData(array $allKeys)
    {
        $resArray = array();
        foreach ($allKeys as $currentKey) {
            $keyEndPos = mb_strpos($currentKey, ' ');
            if ($keyEndPos !== false) {
                $key = str_replace('@', '', mb_substr($currentKey, 0, $keyEndPos));
                $key = $this->fixKey($key);
                $value = trim(mb_substr($currentKey, $keyEndPos));

            } else {
                $key = str_replace('@', '', $currentKey);
                $key = $this->fixKey($key);
                $value = true;
            }
            if (isset($resArray[$key])) {
                if (is_array($resArray[$key])) {
                    $resArray[$key][] = $value;
                } else {
                    $cacheVal = $resArray[$key];
                    $newVal = [$cacheVal, $value];
                    $resArray[$key] = $newVal;
                }
            } else {
                $resArray[$key] = $value;
            }
        }
        return $resArray;
    }

    /**
     * Etiket (tag) formatını düzeltir.
     *
     * 
     * @param string $key etiket
     * @return string düzeltilmiş etiket
     */
    private function fixKey(string $key): string
    {
        return mb_strtolower(str_replace('@', '', trim($key)));
    }

    #endregion utils


    #region static

    /**
     * Verilen docComment metninden yeni bir DocCommentObject nesnesi oluşturur.
     *
     * @param string|null $docComment docComment metni
     * @return static DocCommentObject nesnesi
     */
    public static function parse(?string $docComment): self
    {
        static $instance = null;
        if ($instance === null) {
            $instance = new self(null);
        }
        $newInstance = clone $instance;
        $newInstance->setDocComment($docComment);
        return $newInstance;
    }


    #endregion static

}
