<?php


namespace Gek\Infrastructure;


use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Exceptions\GekException;
use Gek\Infrastructure\Exceptions\InvalidArgumentException;
use Gek\Infrastructure\Exceptions\NotImplementedException;
use Gek\Infrastructure\FlagExp\FlagExpressionParser;
use ReflectionException;

/**
 * FlagEnumType Sınıfı
 *
 * Flag Enum yapısını simule eder.
 *
 * @package Gek\Infrastructure\Enums
 * @since 1.3.0
 */
abstract class FlagEnumType extends FlagEnum implements IToInt
{

    /**
     * Bayrak (flag) değeri
     * @var int
     */
    private int $value;

    /**
     * FlagEnumType yapıcı method
     * .
     * @param string|int $value Bayrak değeri ve ya bayrak adı yada string expression
     * @param bool $validateValue değeri doğrula
     * @throws GekException
     * @throws ReflectionException
     * @throws InvalidArgumentException
     * @throws ArgumentOutOfRangeException
     */
    public function __construct($value, bool $validateValue = true)
    {
        $this->setValue($value, $validateValue);
    }

    /**
     * @param $name
     * @param $arguments
     * @return mixed
     * @throws NotImplementedException
     * @throws ReflectionException
     */
    public static function __callStatic($name, $arguments)
    {
        if (self::checkEnumKey($name)) {
            $class = get_called_class();
            return new $class($name);
        }
        throw new NotImplementedException();
    }

    /**
     * @return int
     */
    public function getValue(): int
    {
        return $this->value;
    }

    /**
     * Bayrak (flag) değerini set eder.
     *
     * @param string|int $value bayrak değeri vaya bayrad adı yada string ifade
     * @param bool $validateValue
     * @throws ReflectionException
     * @throws InvalidArgumentException geçersiz değer
     * @throws ArgumentOutOfRangeException değer sınırlar arasında değil
     * @throws GekException
     */
    public function setValue($value, bool $validateValue = true): void
    {
        if (is_int($value)) {
            $this->value = $value;
        } elseif (is_string($value)) {
            $enumVals = self::getEnumValues();
            $keys = array_keys($enumVals);
            $invalidKeys = array();
            $rs = preg_split('/[|&~^)(\t ]/', $value, -1, PREG_SPLIT_NO_EMPTY);
            $rs = array_filter($rs, function ($item) {
                return !ctype_digit($item);
            });

            foreach ($rs as $ky) {
                if (!in_array($ky, $keys)) {
                    $invalidKeys[] = $ky;
                }
            }
            if (!empty($invalidKeys)) {
                throw new InvalidArgumentException(
                    Str::format('Geçersiz flag key. ("{0}")', implode(' , ', $invalidKeys)),
                    0,
                    null,
                    '$value'
                );
            }

            $value = str_replace($keys, array_values($enumVals), $value);
            $value = Str::toUpperCase($value);
            $all = $this::allFlags();
            $value = str_replace(['ALL', 'ALLFLAGS', 'ALLFLAG'], $all, $value);
            $int = FlagExpressionParser::parseValue($value);
            $this->value = $int;
        } else {
            throw new InvalidArgumentException(
                Str::format('$value ("{0}") parametresi türü yanlış. String ya da int olmalı.', $value)
                , 0, null, '$value'
            );
        }

        if ($validateValue) {
            $allInt = self::allFlags();
            if ($this->value == 0 || ($allInt | $this->value) != $allInt) {
                throw new ArgumentOutOfRangeException(
                    Str::format('value değeri ({0}) flag değerini aşıyor.', $value),
                    0, null, '$value'
                );
            }
        }
    }

    /**
     * Bayrak (flag) değerini verir.
     *
     * @return int Bayrak değeri
     */
    public function toInt(): int
    {
        return $this->value;
    }

    /**
     * Bayrak (falg) değerinin string expression (ifadesini) verir.
     *
     * @return string Bayrak string expression
     * @throws ReflectionException
     */
    public function __toString()
    {

        $valEnums = self::getEnumValues();
        $strArr = array();
        foreach ($valEnums as $key => $val) {
            if (($this->value & $val) == $val) {
                $strArr[] = $key;
            }
        }
        return implode(' | ', $strArr);
    }
}
