<?php


namespace Gek\FileFinder\Tests;


use Gek\Collections\Enumerable;
use Gek\FileFinder\FileFinder;
use Gek\FileFinder\FileInfo;
use PHPUnit\Framework\TestCase;

class FileFinderTest extends TestCase
{
    protected static $tmpDir;
    protected static $files;

    public static function setUpBeforeClass(): void
    {
        // self::$tmpDir = realpath(sys_get_temp_dir()).\DIRECTORY_SEPARATOR.'filefinder';
        self::$tmpDir = __DIR__ . \DIRECTORY_SEPARATOR . 'filefinder';

        self::$files = [
            '.git/',
            '.foo/',
            '.foo/.bar',
            '.foo/bar',
            '.bar',
            'test.py',
            'foo/',
            'foo/bar.tmp',
            'test.php',
            'toto/',
            'toto/.git/',
            'foo bar',
            'qux_0_1.php',
            'qux_2_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux/',
            'qux/baz_1_2.py',
            'qux/baz_100_1.py',
        ];

        self::$files = self::toAbsolute(self::$files);

        if (is_dir(self::$tmpDir)) {
            self::tearDownAfterClass();
        } else {
            mkdir(self::$tmpDir);
        }

        foreach (self::$files as $file) {
            if (\DIRECTORY_SEPARATOR === $file[\strlen($file) - 1]) {
                mkdir($file);
            } else {
                touch($file);
            }
        }

        file_put_contents(self::toAbsolute('test.php'), str_repeat(' ', 800));
        file_put_contents(self::toAbsolute('test.py'), str_repeat(' ', 2000));

        file_put_contents(self::toAbsolute('.gitignore'), '*.php');

        touch(self::toAbsolute('foo/bar.tmp'), strtotime('2005-10-15'));
        touch(self::toAbsolute('test.php'), strtotime('2005-10-15'));

    }

    public static function tearDownAfterClass(): void
    {
        $paths = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator(self::$tmpDir, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($paths as $path) {
            if ($path->isDir()) {
                if ($path->isLink()) {
                    @unlink($path);
                } else {
                    @rmdir($path);
                }
            } else {
                @unlink($path);
            }
        }
    }

    protected static function toAbsolute($files = null)
    {
        /*
         * Without the call to setUpBeforeClass() property can be null.
         */
        if (!self::$tmpDir) {
            //self::$tmpDir = realpath(sys_get_temp_dir()).\DIRECTORY_SEPARATOR.'symfony_finder';
            self::$tmpDir = __DIR__ . \DIRECTORY_SEPARATOR . 'filefinder';
        }

        if (\is_array($files)) {
            $f = [];
            foreach ($files as $file) {
                if (\is_array($file)) {
                    $f[] = self::toAbsolute($file);
                } else {
                    $f[] = self::$tmpDir . \DIRECTORY_SEPARATOR . str_replace('/', \DIRECTORY_SEPARATOR, $file);
                }
            }

            return $f;
        }

        if (\is_string($files)) {
            return self::$tmpDir . \DIRECTORY_SEPARATOR . str_replace('/', \DIRECTORY_SEPARATOR, $files);
        }

        return self::$tmpDir;
    }

    protected static function toAbsoluteFixtures($files)
    {
        $f = [];
        foreach ($files as $file) {
            $f[] = realpath(__DIR__ . \DIRECTORY_SEPARATOR . 'Fixtures' . \DIRECTORY_SEPARATOR . $file);
        }

        return $f;
    }

    protected function assertIterator($expected, \Traversable $iterator)
    {
        // set iterator_to_array $use_key to false to avoid values merge
        // this made FinderTest::testAppendWithAnArray() fail with GnuFinderAdapter
        $values = array_map(function (\SplFileInfo $fileinfo) {
            return str_replace('/', \DIRECTORY_SEPARATOR, $fileinfo->getPathname());
        }, iterator_to_array($iterator, false));

        $expected = array_map(function ($path) {
            return str_replace('/', \DIRECTORY_SEPARATOR, $path);
        }, $expected);

        sort($values);
        sort($expected);

        $this->assertEquals($expected, array_values($values));
    }

    public function testCreate()
    {
        $this->assertInstanceOf(FileFinder::class, FileFinder::create());
    }

    public function testOnlyDirs()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyDirs());
        $this->assertIterator($this->toAbsolute(['foo', 'qux', 'toto']), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->onlyDirs();
        $finder->onlyFiles();
        $finder->onlyDirs();
        $this->assertIterator($this->toAbsolute(['foo', 'qux', 'toto']), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testOnlyFiles()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles());
        $this->assertIterator($this->toAbsolute(['foo/bar.tmp',
            'test.php',
            'test.py',
            'foo bar',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->onlyFiles();
        $finder->onlyDirs();
        $finder->onlyFiles();
        $this->assertIterator($this->toAbsolute(['foo/bar.tmp',
            'test.php',
            'test.py',
            'foo bar',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testRemoveTrailingSlash()
    {
        $finder = FileFinder::create();

        $expected = $this->toAbsolute([
            'foo/bar.tmp',
            'test.php',
            'test.py',
            'foo bar',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]);
        $in = self::$tmpDir . '//';

        $this->assertIterator($expected, $finder->in($in)->onlyFiles()->getIterator());
    }

    public function testSymlinksNotResolved()
    {
        if ('\\' === \DIRECTORY_SEPARATOR) {
            $this->markTestSkipped('symlinks are not supported on Windows');
        }

        $finder = FileFinder::create();

        symlink($this->toAbsolute('foo'), $this->toAbsolute('baz'));
        $expected = $this->toAbsolute(['baz/bar.tmp']);
        $in = self::$tmpDir . '/baz/';
        try {
            $this->assertIterator($expected, $finder->in($in)->onlyFiles()->getIterator());
            unlink($this->toAbsolute('baz'));
        } catch (\Exception $e) {
            unlink($this->toAbsolute('baz'));
            throw $e;
        }
    }

    public function testBackPathNotNormalized()
    {
        $finder = FileFinder::create();

        $expected = $this->toAbsolute(['foo/../foo/bar.tmp']);
        $in = self::$tmpDir . '/foo/../foo/';
        $this->assertIterator($expected, $finder->in($in)->onlyFiles()->getIterator());
    }

    public function testDepth()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->maxDepth(0));
        $this->assertIterator($this->toAbsolute(['foo',
            'test.php',
            'test.py',
            'toto',
            'foo bar',
            'qux',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->minDepth(1));
        $this->assertIterator($this->toAbsolute([
            'foo/bar.tmp',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
        ]), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->maxDepth(0)->minDepth(1);
        $this->assertIterator([], $finder->in(self::$tmpDir)->getIterator());

    }

    public function testNameFilter()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->nameFilter('*.php'));
        $this->assertIterator($this->toAbsolute([
            'test.php',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->nameFilter('test.ph*');
        $finder->nameFilter('test.py');
        $this->assertIterator($this->toAbsolute(['test.php', 'test.py']), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->nameFilter('~^test~i');
        $this->assertIterator($this->toAbsolute(['test.php', 'test.py']), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->nameFilter('~\\.php$~i');
        $this->assertIterator($this->toAbsolute([
            'test.php',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->nameFilter('test.p{hp,y}');
        $this->assertIterator($this->toAbsolute(['test.php', 'test.py']), $finder->in(self::$tmpDir)->getIterator());


    }

    public function testNameFilterWithArrayParam()
    {
        $finder = FileFinder::create();
        $finder->nameFilter(...['test.php', 'test.py']);
        $this->assertIterator($this->toAbsolute(['test.php', 'test.py']), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->nameFilter('test.php', 'test.py');
        $this->assertIterator($this->toAbsolute(['test.php', 'test.py']), $finder->in(self::$tmpDir)->getIterator());

    }

    public function testNotNameFilter()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->notNameFilter('*.php'));
        $this->assertIterator($this->toAbsolute([
            'foo',
            'foo/bar.tmp',
            'test.py',
            'toto',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
        ]), $finder->in(self::$tmpDir)->getIterator());


        $finder = FileFinder::create();
        $finder->notNameFilter('*.php');
        $finder->notNameFilter('*.py');
        $this->assertIterator($this->toAbsolute([
            'foo',
            'foo/bar.tmp',
            'toto',
            'foo bar',
            'qux',
        ]), $finder->in(self::$tmpDir)->getIterator());


        $finder = FileFinder::create();
        $finder->nameFilter('test.ph*');
        $finder->nameFilter('test.py');
        $finder->notNameFilter('*.php');
        $finder->notNameFilter('*.py');
        $this->assertIterator([], $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->nameFilter('test.ph*');
        $finder->nameFilter('test.py');
        $finder->notNameFilter('*.p{hp,y}');
        $this->assertIterator([], $finder->in(self::$tmpDir)->getIterator());
    }

    public function testNotNameFilterWithArrayParam()
    {
        $finder = FileFinder::create();
        $finder->notNameFilter(...['*.php', '*.py']);
        $this->assertIterator($this->toAbsolute([
            'foo',
            'foo/bar.tmp',
            'toto',
            'foo bar',
            'qux',
        ]), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->notNameFilter('*.php', '*.py');
        $this->assertIterator($this->toAbsolute([
            'foo',
            'foo/bar.tmp',
            'toto',
            'foo bar',
            'qux',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function getRegexNameFilterTestData()
    {
        return [
            ['~.*t\\.p.+~i'],
            ['~t.*s~i'],
        ];
    }

    /**
     * @dataProvider getRegexNameFilterTestData
     */
    public function testRegexNameFilter($regex)
    {
        $finder = FileFinder::create();
        $finder->nameFilter($regex);
        $this->assertIterator($this->toAbsolute([
            'test.py',
            'test.php',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testSizeFilter()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles()->sizeFilter('< 1K')->sizeFilter('> 500'));
        $this->assertIterator($this->toAbsolute(['test.php']), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testSizeFilterWithArrayParam()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles()->sizeFilter(...['< 1K', '> 500']));
        $this->assertIterator($this->toAbsolute(['test.php']), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles()->sizeFilter('< 1K', '> 500'));
        $this->assertIterator($this->toAbsolute(['test.php']), $finder->in(self::$tmpDir)->getIterator());

    }

    public function testDateFilter()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles()->dateFilter('until last month'));
        $this->assertIterator($this->toAbsolute(['foo/bar.tmp', 'test.php']), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testDateFilterWithArrayParam()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles()->dateFilter(...['>= 2005-10-15', 'until last month']));
        $this->assertIterator($this->toAbsolute(['foo/bar.tmp', 'test.php']), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles()->dateFilter('>= 2005-10-15', 'until last month'));
        $this->assertIterator($this->toAbsolute(['foo/bar.tmp', 'test.php']), $finder->in(self::$tmpDir)->getIterator());

    }

    public function testExcludeDirs()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->excludeDirs('foo'));
        $this->assertIterator($this->toAbsolute([
            'test.php',
            'test.py',
            'toto',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testIgnoreDotFiles()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->ignoreDotFiles(false));
        $this->assertIterator($this->toAbsolute([
            '.gitignore',
            '.git',
            '.bar',
            '.foo',
            '.foo/.bar',
            '.foo/bar',
            'foo',
            'foo/bar.tmp',
            'test.php',
            'test.py',
            'toto',
            'toto/.git',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $finder->ignoreDotFiles(false)->ignoreDotFiles(false);
        $this->assertIterator($this->toAbsolute([
            '.gitignore',
            '.git',
            '.bar',
            '.foo',
            '.foo/.bar',
            '.foo/bar',
            'foo',
            'foo/bar.tmp',
            'test.php',
            'test.py',
            'toto',
            'toto/.git',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());

        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->ignoreDotFiles(true));
        $this->assertIterator($this->toAbsolute([
            'foo',
            'foo/bar.tmp',
            'test.php',
            'test.py',
            'toto',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());


    }

    public function testIgnoreDotFilesCanBeDisabledAfterFirstIteration()
    {
        $finder = FileFinder::create();
        $finder->in(self::$tmpDir);

        $this->assertIterator($this->toAbsolute([
            'foo',
            'foo/bar.tmp',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
            'test.php',
            'test.py',
            'toto',
            'foo bar',
        ]), $finder->getIterator());

        $finder->ignoreDotFiles(false)->excludeDirs(".git");
        $this->assertIterator($this->toAbsolute([
            '.gitignore',
            'foo',
            'foo/bar.tmp',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
            'test.php',
            'test.py',
            'toto',
            '.bar',
            '.foo',
            '.foo/.bar',
            '.foo/bar',
            'foo bar',
        ]), $finder->getIterator());


    }

    protected function assertOrderedIterator($expected, \Traversable $iterator)
    {
        $values = array_map(function (\SplFileInfo $fileinfo) { return str_replace('/', \DIRECTORY_SEPARATOR, $fileinfo->getPathname()); }, iterator_to_array($iterator));
        $expected = array_map(function ($path) { return str_replace('/', \DIRECTORY_SEPARATOR, $path); }, $expected);

        $this->assertEquals($expected, array_values($values));
    }

    public function testOrderByName()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->orderByName());
        $this->assertOrderedIterator($this->toAbsolute([
            'foo',
            'foo bar',
            'foo/bar.tmp',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
            'test.php',
            'test.py',
            'toto',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testOrderByType()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->orderByType());
        $this->assertOrderedIterator($this->toAbsolute([
            'foo',
            'qux',
            'toto',
            'foo bar',
            'foo/bar.tmp',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
            'test.php',
            'test.py',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testOrderByAccessedTime()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->orderByAccessedTime());

        $testList = $this->toAbsolute([
            'foo/bar.tmp',
            'test.php',
            'toto',
            'test.py',
            'foo',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]);
        $testList = Enumerable::fromArray($testList)
            ->select(fn($i) => new \SplFileInfo($i))
            ->orderBy(fn(\SplFileInfo $i) => $i->getRealPath() ?: $i->getPathname())
            ->orderBy(fn(\SplFileInfo $i) => $i->getATime())
            ->select(fn(\SplFileInfo $i) => $i->getPathname())
            ->toArray();

        $this->assertOrderedIterator($testList, $finder->in(self::$tmpDir)->getIterator());
    }


    public function testOrderByChangedTime()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->orderByName());
        $this->assertSame($finder, $finder->orderByChangedTime());

        $testList = $this->toAbsolute([
            'toto',
            'test.py',
            'test.php',
            'foo/bar.tmp',
            'foo',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]);
        $testList = Enumerable::fromArray($testList)
            ->select(fn($i) => new \SplFileInfo($i))
            ->sort(fn(\SplFileInfo $a, \SplFileInfo $b) => strcmp($a->getRealPath() ?: $a->getPathname(),$b->getRealPath() ?: $b->getPathname()))
            ->orderBy(fn(\SplFileInfo $i) => $i->getCTime())
            ->select(fn(\SplFileInfo $i) => $i->getPathname())
            ->toArray();

        $this->assertOrderedIterator($testList, $finder->in(self::$tmpDir)->getIterator());


    }



    public function testOrderByModifiedTime()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->orderByName());
        $this->assertSame($finder, $finder->orderByModifiedTime());

        $testList = $this->toAbsolute([
            'foo/bar.tmp',
            'test.php',
            'toto',
            'test.py',
            'foo',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]);
        $testList = Enumerable::fromArray($testList)
            ->select(fn($i) => new \SplFileInfo($i))
            ->sort(fn(\SplFileInfo $a, \SplFileInfo $b) => strcmp($a->getRealPath() ?: $a->getPathname(),$b->getRealPath() ?: $b->getPathname()))
            ->orderBy(fn(\SplFileInfo $i) => $i->getMTime())
            ->select(fn(\SplFileInfo $i) => $i->getPathname())
            ->toArray();


        $this->assertOrderedIterator($testList, $finder->in(self::$tmpDir)->getIterator());

    }



    protected function assertOrderedIteratorInForeach(array $expected, \Traversable $iterator)
    {
        $values = [];
        foreach ($iterator as $file) {
            $this->assertInstanceOf(FileInfo::class, $file);
            $values[] = $file->getPathname();
        }

        $this->assertEquals($expected, array_values($values));
    }

    public function testReverse()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->orderByName());
        $this->assertSame($finder, $finder->reverse());
        $this->assertOrderedIteratorInForeach($this->toAbsolute([
            'toto',
            'test.py',
            'test.php',
            'qux_2_0.php',
            'qux_12_0.php',
            'qux_10_2.php',
            'qux_1002_0.php',
            'qux_1000_1.php',
            'qux_0_1.php',
            'qux/baz_1_2.py',
            'qux/baz_100_1.py',
            'qux',
            'foo/bar.tmp',
            'foo bar',
            'foo',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testSort()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->sort(function (\SplFileInfo $a, \SplFileInfo $b) { return strcmp($a->getRealPath(), $b->getRealPath()); }));
        $this->assertOrderedIterator($this->toAbsolute([
            'foo',
            'foo bar',
            'foo/bar.tmp',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
            'test.php',
            'test.py',
            'toto',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testOrderBy(){
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles());
        $this->assertSame($finder, $finder->orderByName());
        $this->assertSame($finder, $finder->orderBy(fn(\SplFileInfo $i) => $i->getSize()));

        $testList = $this->toAbsolute([
            'foo/bar.tmp',
            'test.php',
            'test.py',
            'foo bar',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]);
        $testList = Enumerable::fromArray($testList)
            ->select(fn($i) => new \SplFileInfo($i))
            ->sort(fn(\SplFileInfo $a, \SplFileInfo $b) => strcmp($a->getRealPath() ?: $a->getPathname(),$b->getRealPath() ?: $b->getPathname()))
            ->orderBy(fn(\SplFileInfo $i) => $i->getSize())
            ->select(fn(\SplFileInfo $i) => $i->getPathname())
            ->toArray();

        $this->assertOrderedIterator($testList, $finder->in(self::$tmpDir)->getIterator());

    }

    public function testOrderBySize(){
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->onlyFiles());
        $this->assertSame($finder, $finder->orderByName());
        $this->assertSame($finder, $finder->orderBySize());

        $testList = $this->toAbsolute([
            'foo/bar.tmp',
            'test.php',
            'test.py',
            'foo bar',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]);
        $testList = Enumerable::fromArray($testList)
            ->select(fn($i) => new \SplFileInfo($i))
            ->sort(fn(\SplFileInfo $a, \SplFileInfo $b) => strcmp($a->getRealPath() ?: $a->getPathname(),$b->getRealPath() ?: $b->getPathname()))
            ->orderBy(fn(\SplFileInfo $i) => $i->getSize())
            ->select(fn(\SplFileInfo $i) => $i->getPathname())
            ->toArray();

        $this->assertOrderedIterator($testList, $finder->in(self::$tmpDir)->getIterator());

    }

    public function testSortAcrossDirectories()
    {
        $finder = FileFinder::create()
            ->in(
                self::$tmpDir,
                self::$tmpDir.'/qux',
                self::$tmpDir.'/foo',
            )
            ->maxDepth(0)
            ->onlyFiles()
            ->where(static function (\SplFileInfo $file): bool {
                return '' !== $file->getExtension();
            })
            //->orderByName()
            ->sort(static function (\SplFileInfo $a, \SplFileInfo $b): int {
                return strcmp($a->getExtension(), $b->getExtension()) ?: strcmp($a->getFilename(), $b->getFilename());
            });

        $this->assertOrderedIterator($this->toAbsolute([
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
            'test.php',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'test.py',
            'foo/bar.tmp',
        ]), $finder->getIterator());
    }

    public function testWhere()
    {
        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->where(function (\SplFileInfo $f) { return false !== strpos($f->getFilename(), 'test'); }));
        $this->assertIterator($this->toAbsolute(['test.php', 'test.py']), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testFollowLinks()
    {
        if ('\\' == \DIRECTORY_SEPARATOR) {
            $this->markTestSkipped('symlinks are not supported on Windows');
        }

        $finder = FileFinder::create();
        $this->assertSame($finder, $finder->followLinks());
        $this->assertIterator($this->toAbsolute([
            'foo',
            'foo/bar.tmp',
            'test.php',
            'test.py',
            'toto',
            'foo bar',
            'qux',
            'qux/baz_100_1.py',
            'qux/baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ]), $finder->in(self::$tmpDir)->getIterator());
    }

    public function testIn()
    {
        $finder = FileFinder::create();
        $iterator = $finder->onlyFiles()
            ->nameFilter('*.php')
            ->maxDepth(0)
            ->in(self::$tmpDir, __DIR__)
            ->getIterator();

        $expected = [
            self::$tmpDir.\DIRECTORY_SEPARATOR.'test.php',
            __DIR__.\DIRECTORY_SEPARATOR.'FileFinderTest.php',
            __DIR__.\DIRECTORY_SEPARATOR.'RegexHelperTest.php',
            self::$tmpDir.\DIRECTORY_SEPARATOR.'qux_0_1.php',
            self::$tmpDir.\DIRECTORY_SEPARATOR.'qux_1000_1.php',
            self::$tmpDir.\DIRECTORY_SEPARATOR.'qux_1002_0.php',
            self::$tmpDir.\DIRECTORY_SEPARATOR.'qux_10_2.php',
            self::$tmpDir.\DIRECTORY_SEPARATOR.'qux_12_0.php',
            self::$tmpDir.\DIRECTORY_SEPARATOR.'qux_2_0.php',
        ];

        $this->assertIterator($expected, $iterator);
    }

    public function testInWithNonExistentDirectory()
    {
        $this->expectException(\InvalidArgumentException::class);
        $finder = new FileFinder();
        $finder->in('foobar');
    }

    public function testInWithGlob()
    {
        $finder = FileFinder::create();
        $finder->in(__DIR__.'/Fixtures/*/B/C/', __DIR__.'/Fixtures/*/*/B/C/')->getIterator();

        $this->assertIterator($this->toAbsoluteFixtures(['A/B/C/abc.dat', 'copy/A/B/C/abc.dat.copy']), $finder);
    }

    public function testInWithNonDirectoryGlob()
    {
        $this->expectException(\InvalidArgumentException::class);
        $finder = new FileFinder();
        $finder->in(__DIR__.'/Fixtures/A/a*');
    }

    public function testInWithGlobBrace()
    {
        if (!\defined('GLOB_BRACE')) {
            $this->markTestSkipped('Glob brace is not supported on this system.');
        }

        $finder = FileFinder::create();
        $finder->in(__DIR__.'/Fixtures/{A,copy/A}/B/C')->getIterator();
        $this->assertIterator($this->toAbsoluteFixtures(['A/B/C/abc.dat', 'copy/A/B/C/abc.dat.copy']), $finder);
    }

    public function testGetIterator()
    {
        $finder = FileFinder::create();
        $dirs = [];
        foreach ($finder->onlyDirs()->in(self::$tmpDir) as $dir) {
            $dirs[] = (string) $dir;
        }

        $expected = $this->toAbsolute(['foo', 'qux', 'toto']);

        sort($dirs);
        sort($expected);

        $this->assertEquals($expected, $dirs, 'implements the \IteratorAggregate interface');

        $finder = FileFinder::create();
        $this->assertEquals(3, iterator_count($finder->onlyDirs()->in(self::$tmpDir)), 'implements the \IteratorAggregate interface');

        $finder = FileFinder::create();
        $a = iterator_to_array($finder->onlyDirs()->in(self::$tmpDir));
        $a = array_values(array_map('strval', $a));
        sort($a);
        $this->assertEquals($expected, $a, 'implements the \IteratorAggregate interface');
    }

    public function testRelativePath()
    {
        $finder = FileFinder::create()->in(self::$tmpDir);

        $paths = [];

        foreach ($finder as $file) {
            $paths[] = $file->getRelativePath();
        }

        $ref = ['', '', '', '', '', '', '', '', '', '', '', 'foo', 'qux', 'qux', ''];

        sort($ref);
        sort($paths);

        $this->assertEquals($ref, $paths);
    }

    public function testRelativePathname()
    {
        $finder = FileFinder::create()->in(self::$tmpDir)->orderByName();

        $paths = [];

        foreach ($finder as $file) {
            $paths[] = $file->getRelativePathname();
        }

        $ref = [
            'test.php',
            'toto',
            'test.py',
            'foo',
            'foo'.\DIRECTORY_SEPARATOR.'bar.tmp',
            'foo bar',
            'qux',
            'qux'.\DIRECTORY_SEPARATOR.'baz_100_1.py',
            'qux'.\DIRECTORY_SEPARATOR.'baz_1_2.py',
            'qux_0_1.php',
            'qux_1000_1.php',
            'qux_1002_0.php',
            'qux_10_2.php',
            'qux_12_0.php',
            'qux_2_0.php',
        ];

        sort($paths);
        sort($ref);

        $this->assertEquals($ref, $paths);
    }

    public function testGetFilenameWithoutExtension()
    {
        $finder = FileFinder::create()->in(self::$tmpDir)->orderByName();

        $fileNames = [];

        foreach ($finder as $file) {
            $fileNames[] = $file->getFilenameWithoutExtension();
        }

        $ref = [
            'test',
            'toto',
            'test',
            'foo',
            'bar',
            'foo bar',
            'qux',
            'baz_100_1',
            'baz_1_2',
            'qux_0_1',
            'qux_1000_1',
            'qux_1002_0',
            'qux_10_2',
            'qux_12_0',
            'qux_2_0',
        ];

        sort($fileNames);
        sort($ref);

        $this->assertEquals($ref, $fileNames);
    }

    public function testCountDirectories()
    {
        $directory = FileFinder::create()->onlyDirs()->in(self::$tmpDir);
        $i = 0;

        foreach ($directory as $dir) {
            ++$i;
        }

        $this->assertCount($i, $directory);
    }

    public function testCountFiles()
    {
        $files = FileFinder::create()->onlyFiles()->in(__DIR__.\DIRECTORY_SEPARATOR.'Fixtures');
        $i = 0;

        foreach ($files as $file) {
            ++$i;
        }

        $this->assertCount($i, $files);
    }

    public function testCountWithoutIn()
    {
        $finder = FileFinder::create()->onlyFiles();
        $this->assertCount(0, $finder);
    }

    public function testAny()
    {
        $finder = FileFinder::create();
        $finder->in(__DIR__);
        $this->assertTrue($finder->any());

        $finder = FileFinder::create();
        $finder->in(__DIR__)->nameFilter('DoesNotExist');
        $this->assertFalse($finder->any());
    }

    public function getContainsTestData()
    {
        return [
            ['', '', []],
            ['foo', 'bar', []],
            ['', 'foobar', ['dolor.txt', 'ipsum.txt', 'lorem.txt']],
            ['lorem ipsum dolor sit amet', 'foobar', ['lorem.txt']],
            ['sit', 'bar', ['dolor.txt', 'ipsum.txt', 'lorem.txt']],
            ['dolor sit amet', '@^L@m', ['dolor.txt', 'ipsum.txt']],
            ['lorem', 'foobar', ['lorem.txt']],
            ['', 'lorem', ['dolor.txt', 'ipsum.txt']],
            ['ipsum dolor sit amet', '/^IPSUM/m', ['lorem.txt']],
            [['lorem', 'dolor'], [], ['lorem.txt', 'ipsum.txt', 'dolor.txt']],
            ['', ['lorem', 'ipsum'], ['dolor.txt']],
            ['/^lorem ipsum dolor sit amet/um', 'foobar', ['lorem.txt']],
        ];
    }

    /**
     * @dataProvider getContainsTestData
     */
    public function testContains($matchPatterns, $noMatchPatterns, $expected)
    {
        $finder = FileFinder::create();
        $finder->in(__DIR__.\DIRECTORY_SEPARATOR.'Fixtures')
            ->nameFilter('*.txt')->orderByName()
            ->containsFilter(...(array)$matchPatterns)
            ->notContainsFilter(...(array)$noMatchPatterns);

        $this->assertIterator($this->toAbsoluteFixtures($expected), $finder);
    }

    public function testContainsOnDirectory()
    {
        $finder = FileFinder::create();
        $finder->in(__DIR__)
            ->onlyDirs()
            ->nameFilter('Fixtures')
            ->containsFilter('abc');
        $this->assertIterator([], $finder);
    }

    public function testNotContainsOnDirectory()
    {
        $finder = FileFinder::create();
        $finder->in(__DIR__)
            ->onlyDirs()
            ->nameFilter('Fixtures')
            ->notContainsFilter('abc');
        $this->assertIterator([], $finder);
    }

    /**
     * Searching in multiple locations involves AppendIterator which does an unnecessary rewind which leaves FilterIterator
     * with inner FilesystemIterator in an invalid state.
     *
     * @see https://bugs.php.net/68557
     */
    public function testMultipleLocations()
    {
        $locations = [
            self::$tmpDir.'/',
            self::$tmpDir.'/toto/',
        ];

        // it is expected that there are test.py test.php in the tmpDir
        $finder = FileFinder::create();
        $finder->in(...$locations)
            // the default flag IGNORE_DOT_FILES fixes the problem indirectly
            // so we set it to false for better isolation
            ->ignoreDotFiles(false)
            ->maxDepth(0)->nameFilter('test.php');

        $this->assertCount(1, $finder);
    }


    /**
     * Same as IteratorTestCase::assertIterator with foreach usage.
     */
    protected function assertIteratorInForeach(array $expected, \Traversable $iterator)
    {
        $values = [];
        foreach ($iterator as $file) {
            $this->assertInstanceOf(FileInfo::class, $file);
            $values[] = $file->getPathname();
        }

        sort($values);
        sort($expected);

        $this->assertEquals($expected, array_values($values));
    }

    /**
     * Searching in multiple locations with sub directories involves
     * AppendIterator which does an unnecessary rewind which leaves
     * FilterIterator with inner FilesystemIterator in an invalid state.
     *
     * @see https://bugs.php.net/68557
     */
    public function testMultipleLocationsWithSubDirectories()
    {
        $locations = [
            __DIR__.'/Fixtures/one',
            self::$tmpDir.\DIRECTORY_SEPARATOR.'toto',
        ];

        $finder = FileFinder::create();
        $finder->in(...$locations)->maxDepth(9)->nameFilter('*.neon');

        $expected = [
            __DIR__.'/Fixtures/one'.\DIRECTORY_SEPARATOR.'b'.\DIRECTORY_SEPARATOR.'c.neon',
            __DIR__.'/Fixtures/one'.\DIRECTORY_SEPARATOR.'b'.\DIRECTORY_SEPARATOR.'d.neon',
        ];

        $this->assertIterator($expected, $finder);
        $this->assertIteratorInForeach($expected, $finder);
    }

    /**
     * Iterator keys must be the file pathname.
     */
    public function testIteratorKeys()
    {
        $finder = FileFinder::create()->in(self::$tmpDir);
        foreach ($finder as $key => $file) {
            $this->assertEquals($file->getPathname(), $key);
        }
    }

    public function testRegexSpecialCharsLocationWithPathRestrictionContainingStartFlag()
    {
        $finder = FileFinder::create();
        $finder->in(__DIR__.\DIRECTORY_SEPARATOR.'Fixtures'.\DIRECTORY_SEPARATOR.'r+e.gex[c]a(r)s')
            ->pathFilter('/^dir/');

        $expected = ['r+e.gex[c]a(r)s'.\DIRECTORY_SEPARATOR.'dir', 'r+e.gex[c]a(r)s'.\DIRECTORY_SEPARATOR.'dir'.\DIRECTORY_SEPARATOR.'bar.dat'];
        $this->assertIterator($this->toAbsoluteFixtures($expected), $finder);
    }

    /**
     * @dataProvider getTestPathData
     */
    public function testPath($matchPatterns, $noMatchPatterns, array $expected)
    {

        $finder = FileFinder::create();
        $finder->in(__DIR__.\DIRECTORY_SEPARATOR.'Fixtures')
            ->pathFilter(...(array)$matchPatterns)
            ->notPathFilter(...(array)$noMatchPatterns);

        $this->assertIterator($this->toAbsoluteFixtures($expected), $finder);
    }

    public function getTestPathData()
    {
        return [
            ['', '', []],
            ['/^A\/B\/C/', '/C$/',
                ['A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C'.\DIRECTORY_SEPARATOR.'abc.dat'],
            ],
            ['/^A\/B/', 'foobar',
                [
                    'A'.\DIRECTORY_SEPARATOR.'B',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'ab.dat',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C'.\DIRECTORY_SEPARATOR.'abc.dat',
                ],
            ],
            ['A/B/C', 'foobar',
                [
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C'.\DIRECTORY_SEPARATOR.'abc.dat',
                    'copy'.\DIRECTORY_SEPARATOR.'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C',
                    'copy'.\DIRECTORY_SEPARATOR.'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C'.\DIRECTORY_SEPARATOR.'abc.dat.copy',
                ],
            ],
            ['A/B', 'foobar',
                [
                    //dirs
                    'A'.\DIRECTORY_SEPARATOR.'B',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C',
                    'copy'.\DIRECTORY_SEPARATOR.'A'.\DIRECTORY_SEPARATOR.'B',
                    'copy'.\DIRECTORY_SEPARATOR.'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C',
                    //files
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'ab.dat',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C'.\DIRECTORY_SEPARATOR.'abc.dat',
                    'copy'.\DIRECTORY_SEPARATOR.'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'ab.dat.copy',
                    'copy'.\DIRECTORY_SEPARATOR.'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C'.\DIRECTORY_SEPARATOR.'abc.dat.copy',
                ],
            ],
            ['/^with space\//', 'foobar',
                [
                    'with space'.\DIRECTORY_SEPARATOR.'foo.txt',
                ],
            ],
            [
                '/^A/',
                ['a.dat', 'abc.dat'],
                [
                    'A',
                    'A'.\DIRECTORY_SEPARATOR.'B',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'ab.dat',
                ],
            ],
            [
                ['/^A/', 'one'],
                'foobar',
                [
                    'A',
                    'A'.\DIRECTORY_SEPARATOR.'B',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C',
                    'A'.\DIRECTORY_SEPARATOR.'a.dat',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'ab.dat',
                    'A'.\DIRECTORY_SEPARATOR.'B'.\DIRECTORY_SEPARATOR.'C'.\DIRECTORY_SEPARATOR.'abc.dat',
                    'one',
                    'one'.\DIRECTORY_SEPARATOR.'a',
                    'one'.\DIRECTORY_SEPARATOR.'b',
                    'one'.\DIRECTORY_SEPARATOR.'b'.\DIRECTORY_SEPARATOR.'c.neon',
                    'one'.\DIRECTORY_SEPARATOR.'b'.\DIRECTORY_SEPARATOR.'d.neon',
                ],
            ],
        ];
    }

    public function testAccessDeniedException()
    {
        if ('\\' === \DIRECTORY_SEPARATOR) {
            $this->markTestSkipped('chmod is not supported on Windows');
        }

        $finder = FileFinder::create();
        $finder->onlyFiles()->in(self::$tmpDir);

        // make 'foo' directory non-readable
        $testDir = self::$tmpDir.\DIRECTORY_SEPARATOR.'foo';
        chmod($testDir, 0333);

        if (false === $couldRead = is_readable($testDir)) {
            try {
                $this->assertIterator($this->toAbsolute(['foo bar', 'test.php', 'test.py']), $finder->getIterator());
                $this->fail('Finder should throw an exception when opening a non-readable directory.');
            } catch (\Exception $e) {
                $expectedExceptionClass = \UnexpectedValueException::class;
                if ($e instanceof \PHPUnit\Framework\ExpectationFailedException) {
                    $this->fail(sprintf("Expected exception:\n%s\nGot:\n%s\nWith comparison failure:\n%s", $expectedExceptionClass, 'PHPUnit\Framework\ExpectationFailedException', $e->getComparisonFailure()->getExpectedAsString()));
                }

                $this->assertInstanceOf($expectedExceptionClass, $e);
            }
        }

        // restore original permissions
        chmod($testDir, 0777);
        clearstatcache(true, $testDir);

        if ($couldRead) {
            $this->markTestSkipped('could read test files while test requires unreadable');
        }
    }

    public function testIgnoredAccessDeniedException()
    {
        if ('\\' === \DIRECTORY_SEPARATOR) {
            $this->markTestSkipped('chmod is not supported on Windows');
        }

        $finder = FileFinder::create();
        $finder->onlyFiles()->ignoreUnreadableDirs()->in(self::$tmpDir);

        // make 'foo' directory non-readable
        $testDir = self::$tmpDir.\DIRECTORY_SEPARATOR.'foo';
        chmod($testDir, 0333);

        if (false === ($couldRead = is_readable($testDir))) {
            $this->assertIterator($this->toAbsolute([
                    'foo bar',
                    'test.php',
                    'test.py',
                    'qux/baz_100_1.py',
                    'qux/baz_1_2.py',
                    'qux_0_1.php',
                    'qux_1000_1.php',
                    'qux_1002_0.php',
                    'qux_10_2.php',
                    'qux_12_0.php',
                    'qux_2_0.php',
                ]
            ), $finder->getIterator());
        }

        // restore original permissions
        chmod($testDir, 0777);
        clearstatcache(true, $testDir);

        if ($couldRead) {
            $this->markTestSkipped('could read test files while test requires unreadable');
        }
    }


}