<?php


namespace Gek\Collections\Tests\Typed;


use ArrayIterator;
use ArrayObject;
use Gek\Collections\Enumerable;
use Gek\Collections\Iterators\ArrayListIterator;
use Gek\Collections\Typed\FloatList;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Iterator;
use PHPUnit\Framework\TestCase;
use TypeError;

/**
 * Class FloatListTest
 * @package Gek\Collections\Tests\Typed
 */
class FloatListTest extends TestCase
{
    public function testConstruct(): void
    {

        // array
        $arr = array(10.1, 15.2, 16.3, 17.4);
        $lst = new FloatList(...$arr);
        $this->assertEquals(
            count($arr),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray()
        );

        //Enumerable
        $enmr = Enumerable::fromArray($arr);
        $lst = new FloatList(...$enmr);
        $this->assertEquals(
            $enmr->count(),
            $lst->count()
        );
        $this->assertEquals(
            $enmr->toArray(),
            $lst->toArray()
        );

        // IteratorAggregate
        $arrObj = new ArrayObject($arr);
        $lst = new FloatList(...$arrObj);
        $this->assertEquals(
            $arrObj->count(),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray()
        );

        // Iterator
        $itr = new ArrayIterator($arr);
        $lst = new FloatList(...$itr);
        $this->assertEquals(
            $itr->count(),
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($itr),
            $lst->toArray()
        );

        // generator
        $generator = function () {
            for ($i = 0; $i < 5; $i++) {
                yield $i + 0.1;
            }
        };
        $lst = new FloatList(...$generator());
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($generator()),
            $lst->toArray()
        );
    }

    public function testGetAt():void {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7);
        $this->assertEquals(
            0.0,
            $lst->getAt(0)
        );
        $this->assertEquals(
            4.4,
            $lst->getAt(4)
        );
        $this->assertEquals(
            7.7,
            $lst->getAt(7)
        );
        $this->assertIsFloat($lst->getAt(3));
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->getAt(15);
    }

    #region IEnumerable

    public function testGetIterator(): void
    {
        $lst = new FloatList();
        $this->assertInstanceOf(
            Iterator::class,
            $lst->getIterator()
        );
        $this->assertInstanceOf(
            ArrayListIterator::class,
            $lst->getIterator()
        );
    }

    public function testCount(): void
    {
        $lst = new FloatList(...range(0, 10));
        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            11,
            count($lst)
        );
    }

    #endregion IEnumerable

    #region ICollection

    public function testAdd(): void
    {
        $lst = new FloatList();
        $this->assertEquals(
            0,
            $lst->count()
        );
        $lst->add(0.1);
        $this->assertEquals(
            1,
            $lst->count()
        );

        $this->assertEquals(
            0.1,
            $lst->getAt(0)
        );

        $lst->add(1.5);
        $this->assertEquals(
            2,
            $lst->count()
        );

        $this->assertEquals(
            1.5,
            $lst->getAt(1)
        );
        $this->assertEquals(
            [
                0 => 0.1,
                1 => 1.5
            ],
            $lst->toArray()
        );

        $this->expectException(TypeError::class);
        $lst->add('sdas');


    }

    public function testAddRange(): void
    {
        $lst = new FloatList(...range(0.1, 5.6));

        $this->assertEquals(
            6,
            $lst->count()
        );

        $lst->addRange(...range(0, 5));
        $this->assertEquals(
            12,
            $lst->count()
        );

        $this->assertEquals(
            [
                0 => 0.1,
                1 => 1.1,
                2 => 2.1,
                3 => 3.1,
                4 => 4.1,
                5 => 5.1,
                6 => 0.0,
                7 => 1.0,
                8 => 2.0,
                9 => 3.0,
                10 => 4.0,
                11 => 5.0
            ],
            $lst->toArray()
        );

        $this->expectException(TypeError::class);
        $lst->addRange('dfdsf','xzdgfsgf');

    }

    public function testClear(): void
    {
        $lst = new FloatList(...range(0, 10));

        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            true,
            $lst->any()
        );

        $lst->clear();

        $this->assertEquals(
            0,
            $lst->count()
        );
        $this->assertEquals(
            false,
            $lst->any()
        );

    }

    public function testContains(): void
    {
        $lst = new FloatList();

        $lst->add(54.5);
        $lst->add(78.5);
        $lst->add(5.5);

        $this->assertEquals(
            true,
            $lst->contains(54.5)
        );

        $this->assertEquals(
            true,
            $lst->contains(78.5)
        );

        $this->assertEquals(
            true,
            $lst->contains(5.5)
        );

        $this->assertEquals(
            false,
            $lst->contains(1.0)
        );

        $this->expectException(TypeError::class);
        $lst->contains('fdsfsd');

    }

    public function testRemove(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5);
        $this->assertEquals(
            6,
            $lst->count()
        );
        $lst->remove(2.2);
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            3.3,
            $lst->getAt(2)
        );

        $lst->remove(0);
        $this->assertEquals(
            4,
            $lst->count()
        );
        $this->assertEquals(
            [
                0 => 1.1,
                1 => 3.3,
                2 => 4.4,
                3 => 5.5
            ],
            $lst->toArray()
        );

        $this->expectException(TypeError::class);
        $lst->remove('dfdsfsdf');
    }

    #endregion ICollection

    #region IList

    public function testIndexOf(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7,8.8,9.9,10.10);
        $this->assertEquals(
            5,
            $lst->indexOf(5.5)
        );
        $this->assertEquals(
            -1,
            $lst->indexOf(35.4)
        );
        $lst->remove(4.4);
        $this->assertEquals(
            4,
            $lst->indexOf(5.5)
        );
        $lst->remove(5.5);
        $this->assertEquals(
            -1,
            $lst->indexOf(5.5)
        );
        $this->expectException(TypeError::class);
        $lst->indexOf('sgfdfsg');

    }

    public function testInsert(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7,8.8,9.9,10.10);
        $lst->insert(5, 25.5);
        $this->assertEquals(
            25.5,
            $lst->getAt(5)
        );
        $lst->insert(0, 20.1);
        $this->assertEquals(
            20.1,
            $lst->getAt(0)
        );

        $lst->insert($lst->count(), 29.9);
        $this->assertEquals(
            29.9,
            $lst->getAt(13)
        );
        $this->assertEquals(
            [
                0 => 20.1,
                1 => 0.0,
                2 => 1.1,
                3 => 2.2,
                4 => 3.3,
                5 => 4.4,
                6 => 25.5,
                7 => 5.5,
                8 => 6.6,
                9 => 7.7,
                10 => 8.8,
                11 => 9.9,
                12 => 10.10,
                13 => 29.9
            ],
            $lst->toArray()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->insert(52, 4564.5);
    }

    public function testInsertTypeError(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5);
        $this->expectException(TypeError::class);
        $lst->insert(3, 'dfgdfgdf');
    }

    public function testInsertRange(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7,8.8,9.9,10.10);

        $lst->insertRange(5, 20.1, 20.2);

        $this->assertEquals(
            20.1,
            $lst->getAt(5)
        );
        $this->assertEquals(
            20.2,
            $lst->getAt(6)
        );

        $lst->insertRange(0, 10.1, 10.2);
        $this->assertEquals(
            10.1,
            $lst->getAt(0)
        );
        $this->assertEquals(
            10.2,
            $lst->getAt(1)
        );

        $lst->insertRange($lst->count(), 90.1, 90.2);
        $this->assertEquals(
            90.1,
            $lst->getAt(15)
        );
        $this->assertEquals(
            90.2,
            $lst->getAt(16)
        );

        $this->assertEquals(
            [
                0 => 10.1,
                1 => 10.2,
                2 => 0.0,
                3 => 1.1,
                4 => 2.2,
                5 => 3.3,
                6 => 4.4,
                7 => 20.1,
                8 => 20.2,
                9 => 5.5,
                10 => 6.6,
                11 => 7.7,
                12 => 8.8,
                13 => 9.9,
                14 => 10.10,
                15 => 90.1,
                16 => 90.2
            ],
            $lst->toArray()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->insertRange(245, 6.5, 3.1);
    }

    public function testInsertRangeTypeError(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7,8.8,9.9,10.10);
        $this->expectException(TypeError::class);
        $lst->insertRange(2, 6.5, 3.1, 'dfgdf', 'dfdhfg');
    }

    public function testRemoveAt(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7,8.8,9.9,10.10);

        $lst->removeAt(5);

        $this->assertEquals(
            6.6,
            $lst->getAt(5)
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->removeAt($lst->count());

    }

    public function testRemoveRange(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7,8.8,9.9,10.10);
        $lst->removeRange(5, 2);
        $this->assertEquals(
            [
                0 => 0.0,
                1 => 1.1,
                2 => 2.2,
                3 => 3.3,
                4 => 4.4,
                5 => 7.7,
                6 => 8.8,
                7 => 9.9,
                8 => 10.10
            ],
            $lst->toArray()
        );
        $lst->removeRange(5, 4);
        $this->assertEquals(
            [
                0 => 0.0,
                1 => 1.1,
                2 => 2.2,
                3 => 3.3,
                4 => 4.4,
            ],
            $lst->toArray()
        );
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7,8.8,9.9,10.10);
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->removeRange(15, 1);

    }

    public function testRemoveAll(): void
    {
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7,8.8,9.9,10.10,11.11,12.12,13.13,14.14,15.15);



        $lst->removeAll(function ($item) {
            if($item > 0){
                return (($item % 2) != 0);
            }
            return false;
        });

        $this->assertEquals(
            [
                0 => 0.0,
                1 => 2.2,
                2 => 4.4,
                3 => 6.6,
                4 => 8.8,
                5 => 10.10,
                6 => 12.12,
                7 => 14.14,
            ],
            $lst->toArray()
        );


    }



    #endregion IList

    #region Serializable

    public function testSerializable(){
        $lst = new FloatList(0.0,1.1,2.2,3.3,4.4,5.5,6.6,7.7);

        $ser = serialize($lst);
        $this->assertNotEmpty($ser);
        $this->assertIsString($ser);
        /** @var  FloatList $unSer */
        $unSer = unserialize($ser);
        $this->assertTrue($unSer instanceof FloatList);
        $this->assertEquals($lst , $unSer);
        $this->assertFalse($lst === $unSer);
        $this->assertEquals(
            $lst->toArray(true),
            $unSer->toArray(true)
        );

    }

    #endregion Serializable

}
