<?php


namespace Gek\Collections\Iterators;


use Closure;
use Iterator;

/**
 * Class ZipIterator
 * @package Gek\Collections\Iterators
 */
class ZipIterator implements Iterator
{
    #region fields

    protected Iterator $iterator;

    protected Iterator $secondIterator;

    protected Closure $zipFn;

    #endregion fields

    #region ctor
    /**
     * ZipIterator constructor.
     * @param Iterator $iterator
     * @param Iterator $secondIterator
     * @param callable $zipFn
     */
    public function __construct(Iterator $iterator, Iterator $secondIterator, callable $zipFn)
    {
        $this->iterator = $iterator;
        $this->secondIterator = $secondIterator;
        $this->zipFn = ($zipFn instanceof Closure) ? $zipFn : Closure::fromCallable($zipFn);;
    }

    #endregion ctor

    #region Iterator

    /**
     * Return the current element
     * @link https://php.net/manual/en/iterator.current.php
     * @return mixed Can return any type.
     * @since 5.0.0
     */
    public function current()
    {
        return $this->zipFn->call($this,$this->iterator->current(), $this->secondIterator->current());
    }

    /**
     * Move forward to next element
     * @link https://php.net/manual/en/iterator.next.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function next()
    {
        $this->iterator->next();
        $this->secondIterator->next();
    }

    /**
     * Return the key of the current element
     * @link https://php.net/manual/en/iterator.key.php
     * @return mixed scalar on success, or null on failure.
     * @since 5.0.0
     */
    public function key()
    {
        return $this->iterator->key();
    }

    /**
     * Checks if current position is valid
     * @link https://php.net/manual/en/iterator.valid.php
     * @return bool The return value will be casted to boolean and then evaluated.
     * Returns true on success or false on failure.
     * @since 5.0.0
     */
    public function valid()
    {
        return $this->iterator->valid() && $this->secondIterator->valid();
    }

    /**
     * Rewind the Iterator to the first element
     * @link https://php.net/manual/en/iterator.rewind.php
     * @return void Any returned value is ignored.
     * @since 5.0.0
     */
    public function rewind()
    {
        $this->iterator->rewind();
        $this->secondIterator->rewind();
    }

    #endregion Iterator
}
