<?php


namespace Gek\Collections\Tests\Typed;


use ArrayIterator;
use ArrayObject;
use Gek\Collections\Enumerable;
use Gek\Collections\Iterators\ArrayListIterator;
use Gek\Collections\Typed\StringList;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Str;
use Iterator;
use PHPUnit\Framework\TestCase;
use TypeError;

/**
 * Class StringListTest
 * @package Gek\Collections\Tests\Typed
 */
class StringListTest extends TestCase
{

    public function testConstruct(): void
    {

        // array
        $arr = array("aaa", "bbb", "ccc", "ddd");
        $lst = new StringList(...$arr);
        $this->assertEquals(
            count($arr),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray()
        );

        //Enumerable
        $enmr = Enumerable::fromArray($arr);
        $lst = new StringList(...$enmr);
        $this->assertEquals(
            $enmr->count(),
            $lst->count()
        );
        $this->assertEquals(
            $enmr->toArray(),
            $lst->toArray()
        );

        // IteratorAggregate
        $arrObj = new ArrayObject($arr);
        $lst = new StringList(...$arrObj);
        $this->assertEquals(
            $arrObj->count(),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray()
        );

        // Iterator
        $itr = new ArrayIterator($arr);
        $lst = new StringList(...$itr);
        $this->assertEquals(
            $itr->count(),
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($itr),
            $lst->toArray()
        );

        // generator
        $generator = function () {
            for ($i = 0; $i < 5; $i++) {
                yield "a" . $i;
            }
        };
        $lst = new StringList(...$generator());
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($generator()),
            $lst->toArray()
        );
    }

    public function testGetAt():void {
        $lst = new StringList('a','b','c','d','e','f','g','h');
        $this->assertEquals(
            'a',
            $lst->getAt(0)
        );
        $this->assertEquals(
            'e',
            $lst->getAt(4)
        );
        $this->assertEquals(
            'h',
            $lst->getAt(7)
        );
        $this->assertIsString($lst->getAt(3));
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->getAt(15);
    }

    #region IEnumerable

    public function testGetIterator(): void
    {
        $lst = new StringList();
        $this->assertInstanceOf(
            Iterator::class,
            $lst->getIterator()
        );
        $this->assertInstanceOf(
            ArrayListIterator::class,
            $lst->getIterator()
        );
    }

    public function testCount(): void
    {
        $lst = new StringList('aaa','bbb','ccc','ddd');
        $this->assertEquals(
            4,
            $lst->count()
        );
        $this->assertEquals(
            4,
            count($lst)
        );
    }

    #endregion IEnumerable

    #region ICollection

    public function testAdd(): void
    {
        $lst = new StringList();
        $this->assertEquals(
            0,
            $lst->count()
        );
        $lst->add('aaa');
        $this->assertEquals(
            1,
            $lst->count()
        );

        $this->assertEquals(
            'aaa',
            $lst->getAt(0)
        );

        $lst->add('bbb');
        $this->assertEquals(
            2,
            $lst->count()
        );

        $this->assertEquals(
            'bbb',
            $lst->getAt(1)
        );
        $this->assertEquals(
            [
                0 => 'aaa',
                1 => 'bbb'
            ],
            $lst->toArray()
        );

        $this->expectException(TypeError::class);
        $lst->add((object)['a' => 'a']);
    }

    public function testAddRange(): void
    {
        $lst = new StringList('aa','bb','cc','dd','ee','ff');

        $this->assertEquals(
            6,
            $lst->count()
        );

        $lst->addRange('aa','bb','cc','dd','ee','ff');
        $this->assertEquals(
            12,
            $lst->count()
        );

        $this->assertEquals(
            [
                0 => 'aa',
                1 => 'bb',
                2 => 'cc',
                3 => 'dd',
                4 => 'ee',
                5 => 'ff',
                6 => 'aa',
                7 => 'bb',
                8 => 'cc',
                9 => 'dd',
                10 => 'ee',
                11 => 'ff'
            ],
            $lst->toArray()
        );
        $this->expectException(TypeError::class);
        $lst->addRange((object)['a' => 'a'], array(2,5,8));

    }

    public function testClear(): void
    {
        $lst = new StringList('aa','bb','cc','dd','ee','ff');

        $this->assertEquals(
            6,
            $lst->count()
        );
        $this->assertEquals(
            true,
            $lst->any()
        );

        $lst->clear();

        $this->assertEquals(
            0,
            $lst->count()
        );
        $this->assertEquals(
            false,
            $lst->any()
        );

    }

    public function testContains(): void
    {
        $lst = new StringList();

        $lst->add('aa');
        $lst->add('ee');
        $lst->add('ff');

        $this->assertEquals(
            true,
            $lst->contains('aa')
        );

        $this->assertEquals(
            true,
            $lst->contains('ee')
        );

        $this->assertEquals(
            true,
            $lst->contains('ff')
        );

        $this->assertEquals(
            false,
            $lst->contains('bb')
        );
        $this->expectException(TypeError::class);
        $lst->contains((object)['a' => 'a']);
    }

    public function testRemove(): void
    {
        $lst = new StringList('aa','bb','cc','dd','ee','ff');
        $this->assertEquals(
            6,
            $lst->count()
        );
        $lst->remove('cc');
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            'dd',
            $lst->getAt(2)
        );

        $lst->remove('aa');
        $this->assertEquals(
            4,
            $lst->count()
        );
        $this->assertEquals(
            [
                0 => 'bb',
                1 => 'dd',
                2 => 'ee',
                3 => 'ff'
            ],
            $lst->toArray()
        );
        $this->expectException(TypeError::class);
        $lst->remove((object)['a' => 'a']);
    }

    #endregion ICollection

    #region IList

    public function testIndexOf(): void
    {
        $lst = new StringList('aa','bb','cc','dd','ee','ff','gg','hh');
        $this->assertEquals(
            5,
            $lst->indexOf('ff')
        );
        $this->assertEquals(
            -1,
            $lst->indexOf('zz')
        );
        $lst->remove('ee');
        $this->assertEquals(
            4,
            $lst->indexOf('ff')
        );
        $lst->remove('ff');
        $this->assertEquals(
            -1,
            $lst->indexOf('ff')
        );
        $this->expectException(TypeError::class);
        $lst->indexOf((object)['a' => 'a']);
    }

    public function testInsert(): void
    {
        $lst = new StringList('aa','bb','cc','dd','ee','ff','gg','hh');
        $lst->insert(5, 'k');
        $this->assertEquals(
            'k',
            $lst->getAt(5)
        );
        $lst->insert(0, 'a');
        $this->assertEquals(
            'a',
            $lst->getAt(0)
        );

        $lst->insert($lst->count(), 'z');
        $this->assertEquals(
            'z',
            $lst->getAt(10)
        );
        $this->assertEquals(
            [
                0 => 'a',
                1 => 'aa',
                2 => 'bb',
                3 => 'cc',
                4 => 'dd',
                5 => 'ee',
                6 => 'k',
                7 => 'ff',
                8 => 'gg',
                9 => 'hh',
                10 => 'z',
            ],
            $lst->toArray()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->insert(52, 'invalid index');
    }

    public function testInsertTypeError(): void
    {
        $lst = new StringList('aa','bb','cc','dd','ee','ff','gg','hh');

        $this->expectException(TypeError::class);
        $lst->insert(2, (object)['a' => 'a']);
    }

    public function testInsertRange(): void
    {
        $lst = new StringList('aa','bb','cc','dd','ee','ff','gg','hh');

        $lst->insertRange(5,'k','l');

        $this->assertEquals(
            'k',
            $lst->getAt(5)
        );
        $this->assertEquals(
            'l',
            $lst->getAt(6)
        );

        $lst->insertRange(0, 'a','b');
        $this->assertEquals(
            'a',
            $lst->getAt(0)
        );
        $this->assertEquals(
            'b',
            $lst->getAt(1)
        );

        $lst->insertRange($lst->count(), 'y','z');
        $this->assertEquals(
            'y',
            $lst->getAt(12)
        );
        $this->assertEquals(
            'z',
            $lst->getAt(13)
        );

        $this->assertEquals(
            [
                0 => 'a',
                1 => 'b',
                2 => 'aa',
                3 => 'bb',
                4 => 'cc',
                5 => 'dd',
                6 => 'ee',
                7 => 'k',
                8 => 'l',
                9 => 'ff',
                10 => 'gg',
                11 => 'hh',
                12 => 'y',
                13 => 'z',
            ],
            $lst->toArray()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->insertRange(245, 'dfs','dfsdf');
    }

    public function testInsertRangeTypeError(): void
    {
        $lst = new StringList('aa','bb','cc','dd','ee','ff','gg','hh');
        $this->expectException(TypeError::class);
        $lst->insertRange(3, (object)['a'=>'a'],[5,6]);
    }

    public function testRemoveAt(): void
    {
        $lst = new StringList('a','b','c','d','e','f','g','h','k');

        $lst->removeAt(5);

        $this->assertEquals(
            'g',
            $lst->getAt(5)
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->removeAt($lst->count());

    }

    public function testRemoveRange(): void
    {
        $lst = new StringList('a','b','c','d','e','f','g','h','k');
        $lst->removeRange(5, 2);
        $this->assertEquals(
            [
                0 => 'a',
                1 => 'b',
                2 => 'c',
                3 => 'd',
                4 => 'e',
                5 => 'h',
                6 => 'k',
            ],
            $lst->toArray()
        );
        $lst->removeRange(3, 4);
        $this->assertEquals(
            [
                0 => 'a',
                1 => 'b',
                2 => 'c',
            ],
            $lst->toArray()
        );
        $lst = new StringList('a','b','c','d','e','f','g','h','k');
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->removeRange(3, 50);

    }

    public function testRemoveAll(): void
    {
        $lst = new StringList();

        $lst->add('Kamil');
        $lst->add('Hasan');
        $lst->add('Selim');
        $lst->add('Mahmut');
        $lst->add('Sibel');
        $lst->add('Murat');
        $lst->add('Harun');
        $lst->add('Samet');
        $lst->add('Levent');
        $lst->add('Serdar');
        $lst->add('Mesut');

        $lst->removeAll(function ($item) {
            return Str::startsWith($item, 'S') || Str::startsWith($item, 'M');
        });

        $this->assertEquals(
            [
                0 => 'Kamil',
                1 => 'Hasan',
                2 => 'Harun',
                3 => 'Levent',
            ],
            $lst->toArray()
        );

    }

    #endregion IList

    #region Serializable

    public function testSerializable(){
        $lst = new StringList('a','b','c','d','e','f','g','h');

        $ser = serialize($lst);
        $this->assertNotEmpty($ser);
        $this->assertIsString($ser);
        /** @var  StringList $unSer */
        $unSer = unserialize($ser);
        $this->assertTrue($unSer instanceof StringList);
        $this->assertEquals($lst , $unSer);
        $this->assertFalse($lst === $unSer);
        $this->assertEquals(
            $lst->toArray(true),
            $unSer->toArray(true)
        );

    }

    #endregion Serializable

}
