<?php


namespace Gek\Collections\Tests\Typed;


use ArrayIterator;
use ArrayObject;
use Gek\Collections\ArrayList;
use Gek\Collections\Enumerable;
use Gek\Collections\IEnumerable;
use Gek\Collections\Iterators\DictionaryIterator;
use Gek\Collections\Tests\Fixtures\MockRefObj;
use Gek\Collections\Typed\IntDictionary;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Str;
use Iterator;
use PHPUnit\Framework\TestCase;
use TypeError;

/**
 * Class IntDictionaryTest
 * @package Gek\Collections\Tests\Typed
 */
class IntDictionaryTest extends TestCase
{
    public function testConstruct(): void
    {

        // array
        $arr = array('a' => 10, 'b' => 15, 'c' => 16, 'd' => 17);
        $lst = new IntDictionary($arr);
        $this->assertEquals(
            count($arr),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray(true)
        );

        //Enumerable
        $enmr = Enumerable::fromArray($arr);
        $lst = new IntDictionary($enmr);
        $this->assertEquals(
            $enmr->count(),
            $lst->count()
        );
        $this->assertEquals(
            $enmr->toArray(true),
            $lst->toArray(true)
        );

        // IteratorAggregate
        $arrObj = new ArrayObject($arr);
        $lst = new IntDictionary($arrObj);
        $this->assertEquals(
            $arrObj->count(),
            $lst->count()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray(true)
        );

        // Iterator
        $itr = new ArrayIterator($arr);
        $lst = new IntDictionary($itr);
        $this->assertEquals(
            $itr->count(),
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($itr),
            $lst->toArray(true)
        );

        // generator
        $generator = function () {
            for ($i = 0; $i < 5; $i++) {
                yield $i . '-a' => $i;
            }
        };
        $lst = new IntDictionary($generator());
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            iterator_to_array($generator()),
            $lst->toArray(true)
        );
    }

    public function testGetByKey(): void
    {
        $lst = new IntDictionary([
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
            'e' => 4,
            'f' => 5,
            'g' => 6,
            'h' => 7,
        ]);
        $this->assertEquals(
            0,
            $lst->getByKey('a')
        );
        $this->assertEquals(
            4,
            $lst->getByKey('e')
        );
        $this->assertEquals(
            7,
            $lst->getByKey('h')
        );
        $this->assertIsInt($lst->getByKey('d'));
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->getByKey('z');
    }

    #region IEnumerable

    public function testGetIterator(): void
    {
        $lst = new IntDictionary();
        $this->assertInstanceOf(
            Iterator::class,
            $lst->getIterator()
        );
        $this->assertInstanceOf(
            DictionaryIterator::class,
            $lst->getIterator()
        );
    }

    public function testCount(): void
    {
        $lst = new IntDictionary(range(0, 10));
        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            11,
            count($lst)
        );
    }

    #endregion IEnumerable

    #region ICollection

    public function testAdd(): void
    {
        $lst = new IntDictionary();
        $this->assertEquals(
            0,
            $lst->count()
        );
        $lst->add(0);
        $this->assertEquals(
            1,
            $lst->count()
        );

        $this->assertEquals(
            0,
            $lst->getByKey('0')
        );

        $lst->add(1);
        $this->assertEquals(
            2,
            $lst->count()
        );

        $this->assertEquals(
            1,
            $lst->getByKey('1')
        );
        $this->assertEquals(
            [
                '0' => 0,
                '1' => 1
            ],
            $lst->toArray(true)
        );
        $this->expectException(TypeError::class);
        $lst->add('sdas');

    }

    public function testAddRange(): void
    {
        $lst = new IntDictionary(range(0, 5));

        $this->assertEquals(
            6,
            $lst->count()
        );

        $lst->addRange(['a' => 0, 'b' => 1, 'c' => 2]);
        $this->assertEquals(
            9,
            $lst->count()
        );

        $this->assertEquals(
            [
                '0' => 0,
                '1' => 1,
                '2' => 2,
                '3' => 3,
                '4' => 4,
                '5' => 5,
                'a' => 0,
                'b' => 1,
                'c' => 2,
            ],
            $lst->toArray(true)
        );
        $this->expectException(TypeError::class);
        $lst->addRange(['z' => 'dfdsf', 'y' => 'xzdgfsgf']);
    }

    public function testClear(): void
    {
        $lst = new IntDictionary(range(0, 10));

        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            true,
            $lst->any()
        );

        $lst->clear();

        $this->assertEquals(
            0,
            $lst->count()
        );
        $this->assertEquals(
            false,
            $lst->any()
        );

    }

    public function testContains(): void
    {
        $lst = new IntDictionary();

        $lst->add(54);
        $lst->add(78);
        $lst->add(5);

        $this->assertEquals(
            true,
            $lst->contains(54)
        );

        $this->assertEquals(
            true,
            $lst->contains(78)
        );

        $this->assertEquals(
            true,
            $lst->contains(5)
        );

        $this->assertEquals(
            false,
            $lst->contains(1)
        );
        $this->expectException(TypeError::class);
        $lst->contains('fdsfsd');
    }

    public function testRemove(): void
    {
        $lst = new IntDictionary([
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
            'e' => 4,
            'f' => 5,
        ]);
        $this->assertEquals(
            6,
            $lst->count()
        );
        $lst->remove(2);
        $this->assertEquals(
            5,
            $lst->count()
        );


        $lst->remove(0);
        $this->assertEquals(
            4,
            $lst->count()
        );
        $this->assertEquals(
            [
                'b' => 1,
                'd' => 3,
                'e' => 4,
                'f' => 5
            ],
            $lst->toArray(true)
        );
        $this->expectException(TypeError::class);
        $lst->remove('dfdsfsdf');
    }

    #endregion ICollection

    #region IDictionary

    public function testAddKeyValue(): void
    {
        $lst = new IntDictionary();
        $lst->addKeyValue('a', 0);
        $this->assertEquals(
            1,
            $lst->count()
        );
        $this->assertEquals(
            0,
            $lst->getByKey('a')
        );
        $lst->addKeyValue('b', 1);
        $this->assertEquals(
            2,
            $lst->count()
        );
        $this->assertEquals(
            1,
            $lst->getByKey('b')
        );

        $this->assertEquals(
            [
                'a' => 0,
                'b' => 1,
            ],
            $lst->toArray(true)
        );

        $this->expectException(TypeError::class);
        $lst->addKeyValue('c', 'adsfdsf');

    }

    public function testTryAddKeyValue(): void
    {
        $lst = new IntDictionary();
        $this->assertTrue(
            $lst->tryAddKeyValue('a', 0)
        );

        $this->assertEquals(
            1,
            $lst->count()
        );
        $this->assertEquals(
            0,
            $lst->getByKey('a')
        );
        $this->assertTrue(
            $lst->tryAddKeyValue('b', 1)
        );
        $this->assertEquals(
            2,
            $lst->count()
        );
        $this->assertEquals(
            1,
            $lst->getByKey('b')
        );

        $this->assertFalse(
            $lst->tryAddKeyValue('a', 5)
        );

        $this->assertEquals(
            [
                'a' => 0,
                'b' => 1,
            ],
            $lst->toArray(true)
        );

        $this->expectException(TypeError::class);
        $lst->addKeyValue('c', 'adsfdsf');

    }

    public function testContainsKey(): void
    {
        $lst = new IntDictionary();
        $lst->addKeyValue('a', 0);
        $lst->addKeyValue('b', 1);
        $lst->addKeyValue('c', 2);
        $lst->addKeyValue('d', 3);

        $this->assertTrue(
            $lst->containsKey('c')
        );
        $this->assertTrue(
            $lst->containsKey('b')
        );
        $this->assertTrue(
            $lst->containsKey('a')
        );
        $this->assertTrue(
            $lst->containsKey('d')
        );
        $this->assertFalse(
            $lst->containsKey('ab')
        );

        $this->expectException(TypeError::class);
        $lst->containsKey((object)['a' => 45645]);
    }

    public function testRemoveByKey(): void
    {
        $lst = new IntDictionary(
            [
                'a' => 0,
                'b' => 1,
                'c' => 2,
                'd' => 3,
                'e' => 4,
                'f' => 5,
            ]
        );
        $this->assertTrue(
            $lst->removeByKey('d')
        );
        $this->assertTrue(
            $lst->removeByKey('f')
        );
        $this->assertTrue(
            $lst->removeByKey('a')
        );
        $this->assertFalse(
            $lst->removeByKey('a')
        );
        $this->assertFalse(
            $lst->removeByKey('zz')
        );

        $this->assertEquals(
            [
                'b' => 1,
                'c' => 2,
                'e' => 4,
            ],
            $lst->toArray(true)
        );

        $this->expectException(TypeError::class);
        $lst->removeByKey((object)['a' => 45645]);

    }

    public function testGetKeys(): void
    {
        $lst = new IntDictionary(
            [
                'a' => 0,
                'b' => 1,
                'c' => 2,
                'd' => 3,
                'e' => 4,
                'f' => 5,
            ]
        );
        $keys = $lst->getKeys();

        $this->assertInstanceOf(
            IEnumerable::class,
            $keys
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $keys
        );
        $this->assertEquals(
            ['a', 'b', 'c', 'd', 'e', 'f'],
            $keys->toArray()
        );
    }

    public function testGetValues(): void
    {
        $lst = new IntDictionary(
            [
                'a' => 0,
                'b' => 1,
                'c' => 2,
                'd' => 3,
                'e' => 4,
                'f' => 5,
            ]
        );
        $values = $lst->getValues();

        $this->assertInstanceOf(
            IEnumerable::class,
            $values
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $values
        );
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5],
            $values->toArray()
        );
    }

    #endregion IDictionary

    #region Enumerable

    public function testWhere(): void
    {
        $lst = new IntDictionary([
            'a1' => 10,
            'b1' => 25,
            'a2' => 18,
            'b2' => 26,
            'a3' => 21,
            'b3' => 30,
            'a4' => 20,
            'b4' => 48,
            'c' => 5
        ]);
        $t1 = $lst->where(function ($item) {
            return $item > 20;
        })->toArray(true);
        $this->assertEquals(
            [
                'b1' => 25,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'b4' => 48
            ],
            $t1
        );
        $t2 = $lst->where(function ($item) {
            return $item < 20;
        });
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $lst->addKeyValue('a5', 8);
        $this->assertEquals(
            [
                'a1' => 10,
                'a2' => 18,
                'c' => 5,
                'a5' => 8,
            ],
            $t2->toArray(true)
        );

        $t3 = $lst->where(function (/** @noinspection PhpUnusedParameterInspection */ int $item, string $key) {
            return Str::startsWith($key, 'b');
        });
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $lst->addKeyValue('b8', 12);
        $this->assertEquals(
            [
                'b1' => 25,
                'b2' => 26,
                'b3' => 30,
                'b4' => 48,
                'b8' => 12,
            ],
            $t3->toArray(true)
        );

    }

    public function testLimit(): void
    {
        $lst = new IntDictionary(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ]
        );
        $t1 = $lst->limit(2, 2)->toArray(true);
        $this->assertEquals(
            [
                'a2' => 18,
                'b2' => 26,
            ],
            $t1
        );
        $t2 = $lst->limit(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $lst->addKeyValue('a5', 8);
        $this->assertEquals(
            [

                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5,
                'a5' => 8
            ],
            $t2->toArray(true)
        );

        $t3 = $lst->limit(50, 10)->toArray();
        $this->assertTrue(empty($t3));
    }

    public function testSkip(): void
    {
        $lst = new IntDictionary(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ]
        );
        $t1 = $lst->skip(0)->toArray(true);
        $this->assertEquals(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ],
            $t1
        );
        $t2 = $lst->skip(5)->toArray(true);
        $this->assertEquals(
            [
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ],
            $t2
        );

        $t3 = $lst->skip(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $lst->addKeyValue('a5', 11);
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5,
                'a5' => 11,
            ],
            $t3->toArray(true)
        );

    }

    public function testTake(): void
    {
        $lst = new IntDictionary(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ]
        );
        $t1 = $lst->take(0)->toArray(true);
        $this->assertEquals(
            [],
            $t1
        );
        $t2 = $lst->take(5)->toArray(true);
        $this->assertEquals(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
            ],
            $t2
        );

        $t3 = $lst->take(10);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $lst->addKeyValue('a5', 11);
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5,
                'a5' => 11,
            ],
            $t3->toArray(true)
        );

    }

    public function testSelect(): void
    {
        $lst = new IntDictionary([
            'a1' => 10,
            'b1' => 25,
            'a2' => 18,
            'b2' => 26,
            'a3' => 21,
            'b3' => 30,
            'a4' => 20,
            'b4' => 48,
            'c' => 5
        ]);
        $t1 = $lst->select(function (int $item, string $key) {
            return $key . '|' . $item;
        })->toArray(true);
        $this->assertEquals(
            [
                'a1' => 'a1|10',
                'b1' => 'b1|25',
                'a2' => 'a2|18',
                'b2' => 'b2|26',
                'a3' => 'a3|21',
                'b3' => 'b3|30',
                'a4' => 'a4|20',
                'b4' => 'b4|48',
                'c' => 'c|5'
            ],
            $t1
        );

        $lst = new ArrayList([
            new MockRefObj(0, 5),
            new MockRefObj(8, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->select(function (MockRefObj $item) {
            return $item->propB;
        })->toTypedClass(IntDictionary::class);

        $this->assertInstanceOf(
            IntDictionary::class,
            $t2
        );

        $this->assertEquals(
            [
                '0' => 5,
                '1' => 5,
                '2' => 5,
                '3' => 5,
                '4' => 5,
                '5' => 5,
            ],
            $t2->toArray(true)
        );

    }

    public function testEnumerableCount(): void
    {
        $lst = new IntDictionary(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ]
        );
        $this->assertEquals(
            9,
            $lst->asEnumerable()->count()
        );
        $this->assertEquals(
            3,
            $lst->count(function ($item) {
                return $item < 20;
            })
        );
        $this->assertEquals(
            2,
            $lst->count(function ($item, $key) {
                return $item < 20 && Str::startsWith($key, 'a');
            })
        );
    }

    public function testSum(): void
    {
        $lst = new IntDictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->sum();
        $this->assertEquals(
            55,
            $t1
        );

        $lst = new IntDictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t2 = $lst->sum(function (int $item) {
            return $item + 1;
        });
        $this->assertEquals(
            66,
            $t2
        );
    }

    public function testAny(): void
    {
        $lst = new IntDictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        $this->assertTrue(
            $lst->any()
        );
        $this->assertTrue(
            $lst->any(function ($item) {
                return $item > 2;
            })
        );
        $this->assertTrue(
            $lst->any(function ($item) {
                return $item === 0;
            })
        );
        $this->assertFalse(
            $lst->any(function ($item) {
                return $item > 15;
            })
        );
        $lst->clear();
        $this->assertFalse(
            $lst->any()
        );
    }

    public function testAll(): void
    {
        $lst = new IntDictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        $this->assertTrue(
            $lst->all(function ($item) {
                return $item < 20;
            })
        );
        $this->assertTrue(
            $lst->all(function ($item) {
                return is_int($item);
            })
        );
        $this->assertFalse(
            $lst->all(function ($item) {
                return $item > 5;
            })
        );

        $lst->clear();
        $this->assertTrue(
            $lst->all(function ($item) {
                return $item > 5;
            })
        );
    }

    public function testAverage(): void
    {
        $lst = new IntDictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->average();
        $this->assertEquals(
            5.0,
            $t1
        );
    }

    public function testMin(): void
    {
        $lst = new IntDictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->min();
        $this->assertEquals(
            0,
            $t1
        );
        $t2 = $lst->min(function (int $item) {
            if ($item > 5) {
                return $item;
            }
            return PHP_INT_MAX;
        });
        $this->assertEquals(
            6,
            $t2
        );
    }

    public function testMax(): void
    {
        $lst = new IntDictionary([0, 1, 2, 3, 4, 5, 25, 6, 7, 8, 9, 10]);
        $t1 = $lst->max();
        $this->assertEquals(
            25,
            $t1
        );
        $t2 = $lst->max(function (int $item) {
            if ($item < 8) {
                return $item;
            }
            return PHP_INT_MIN;
        });
        $this->assertEquals(
            7,
            $t2
        );
    }

    public function testReverse(): void
    {
        $lst = new IntDictionary(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ]
        );
        $t1 = $lst->reverse();
        $this->assertEquals(
            [
                'c' => 5,
                'b4' => 48,
                'a4' => 20,
                'b3' => 30,
                'a3' => 21,
                'b2' => 26,
                'a2' => 18,
                'b1' => 25,
                'a1' => 10,
            ],
            $t1->toArray(true)
        );
    }

    public function testAsEnumerable(): void
    {
        $lst = new IntDictionary(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ]
        );
        $t1 = $lst->where(function (int $item) {
            return $item > 20;
        });

        $t2 = $lst->where(function (int $item) {
            return $item > 20;
        })->asEnumerable();

        $lst->addKeyValue('d', 60);

        $this->assertEquals(
            [
                'b1' => 25,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'b4' => 48,
                'd' => 60
            ],
            $t1->toArray(true)
        );
        $this->assertEquals(
            [
                'b1' => 25,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'b4' => 48,
            ],
            $t2->toArray(true)
        );


    }

    public function testSort(): void
    {
        $lst = new IntDictionary(
            [
                'c' => 3,
                'f' => 8,
                'a' => 4,
                'd' => 1,
                'g' => 7,
                'b' => 0,
                'e' => 9,
                'h' => 5,
                'k' => 2,
                'l' => 6,
            ]
        );
        $t1 = $lst->sort();
        $this->assertEquals(
            [
                'b' => 0,
                'd' => 1,
                'k' => 2,
                'c' => 3,
                'a' => 4,
                'h' => 5,
                'l' => 6,
                'g' => 7,
                'f' => 8,
                'e' => 9,
            ],
            $t1->toArray(true)
        );
        $t2 = $lst->sort(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [
                'e' => 9,
                'f' => 8,
                'g' => 7,
                'l' => 6,
                'h' => 5,
                'a' => 4,
                'c' => 3,
                'k' => 2,
                'd' => 1,
                'b' => 0,
            ],
            $t2->toArray(true)
        );
    }

    public function testSortDesc(): void
    {
        $lst = new IntDictionary(
            [
                'c' => 3,
                'f' => 8,
                'a' => 4,
                'd' => 1,
                'g' => 7,
                'b' => 0,
                'e' => 9,
                'h' => 5,
                'k' => 2,
                'l' => 6,
            ]
        );
        $t1 = $lst->sortDesc();
        $this->assertEquals(
            [
                'e' => 9,
                'f' => 8,
                'g' => 7,
                'l' => 6,
                'h' => 5,
                'a' => 4,
                'c' => 3,
                'k' => 2,
                'd' => 1,
                'b' => 0,
            ],
            $t1->toArray(true)
        );
        $t2 = $lst->sortDesc(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [
                'b' => 0,
                'd' => 1,
                'k' => 2,
                'c' => 3,
                'a' => 4,
                'h' => 5,
                'l' => 6,
                'g' => 7,
                'f' => 8,
                'e' => 9,
            ],
            $t2->toArray(true)
        );
    }

    public function testOrderBy(): void
    {
        $lst = new IntDictionary(
            [
                'c' => 3,
                'f' => 8,
                'a' => 4,
                'd' => 1,
                'g' => 7,
                'b' => 0,
                'e' => 9,
                'h' => 5,
                'k' => 2,
                'l' => 6,
            ]
        );
        $t1 = $lst->orderBy();
        $this->assertEquals(
            [
                'b' => 0,
                'd' => 1,
                'k' => 2,
                'c' => 3,
                'a' => 4,
                'h' => 5,
                'l' => 6,
                'g' => 7,
                'f' => 8,
                'e' => 9,
            ],
            $t1->toArray(true)
        );
    }

    public function testOrderByDesc(): void
    {
        $lst =  new IntDictionary(
            [
                'c' => 3,
                'f' => 8,
                'a' => 4,
                'd' => 1,
                'g' => 7,
                'b' => 0,
                'e' => 9,
                'h' => 5,
                'k' => 2,
                'l' => 6,
            ]
        );
        $t1 = $lst->orderByDesc();
        $this->assertEquals(
            [
                'e' => 9,
                'f' => 8,
                'g' => 7,
                'l' => 6,
                'h' => 5,
                'a' => 4,
                'c' => 3,
                'k' => 2,
                'd' => 1,
                'b' => 0,
            ],
            $t1->toArray(true)
        );
    }

    public function testConcat(): void
    {
        $lst = new IntDictionary([0, 1, 2, 3, 4, 5]);
        $t1 = $lst->concat([6, 7, 8, 9, 10]);
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1->toArray()
        );
    }

    public function testDistinct(): void
    {
        $lst = new IntDictionary([0, 1, 0, 1, 1, 0]);
        $t1 = $lst->distinct();
        $this->assertEquals(
            [0, 1],
            $t1->toArray()
        );
    }

    public function testExcept(): void
    {
        $lst = new IntDictionary([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->except([10, 9, 5, 6, 7, 8]);
        $this->assertEquals(
            [0, 1, 2, 3, 4],
            $t1->toArray()
        );
    }

    public function testFirstOrNUll(): void
    {
        $lst =  new IntDictionary(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ]
        );
        $this->assertEquals(
            10,
            $lst->firstOrNull()
        );
        $this->assertEquals(
            26,
            $lst->firstOrNull(function ($item) {
                return $item > 25;
            })
        );
        $this->assertEquals(
            25,
            $lst->firstOrNull(function (/** @noinspection PhpUnusedParameterInspection */ $item, $key) {
                return Str::startsWith($key,'b');
            })
        );
        $this->assertEquals(
            null,
            $lst->firstOrNull(function ($item) {
                return $item > 1500;
            })
        );
    }

    public function testAggregate(): void
    {
        $lst = new IntDictionary([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            3628800,
            $lst->aggregate(function (&$total, $item) {
                $total *= $item;
            })
        );
        $this->assertEquals(
            181440,
            $lst->aggregate(
                function (&$total, $item) {
                    $total *= $item;
                },
                null,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
        $this->assertEquals(
            3.25,
            $lst->aggregate(
                function (&$total, $item) {
                    $total += $item;
                },
                10,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
    }

    public function testUnion(): void
    {
        $lst = new IntDictionary([-3, 1, 2, 3, 3, 4, 4, 5]);
        $t1 = $lst->union([3, 8, 4, 8]);

        $this->assertEquals(
            [-3, 1, 2, 3, 4, 5, 8],
            $t1->toArray()
        );
        $t2 = $lst->union([3, 8, 4, 8], function ($a, $b) {
            return $a === $b;
        });
        $this->assertEquals(
            [-3, 1, 2, 3, 4, 5, 8],
            $t2->toArray()
        );
    }

    public function testIntersect(): void
    {
        $lst = new IntDictionary([1, 3, 4, 5]);
        $t1 = $lst->intersect([3, 8, 4]);

        $this->assertEquals(
            [3, 4],
            $t1->toArray()
        );
        $t2 = $lst->intersect([]);

        $this->assertEquals(
            [],
            $t2->toArray()
        );
    }

    public function testZip(): void
    {
        $lst = new IntDictionary([1, 2, 3, 4, 5]);
        $t1 = $lst->zip(new ArrayList(['a', 'b', 'c', 'd', 'e']), function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b', '3-c', '4-d', '5-e'],
            $t1->toArray()
        );

        $t2 = $lst->zip(new ArrayList(['a', 'b', 'c']), function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b', '3-c'],
            $t2->toArray()
        );

        $lst2 = new IntDictionary([1, 2]);
        $t3 = $lst2->zip(new ArrayList(['a', 'b', 'c', 'd', 'e']), function ($f, $s) {
            return $f . '-' . $s;
        });

        $this->assertEquals(
            ['1-a', '2-b'],
            $t3->toArray()
        );
    }

    public function testLastOrNull(): void
    {
        $lst =  new IntDictionary(
            [
                'a1' => 10,
                'b1' => 25,
                'a2' => 18,
                'b2' => 26,
                'a3' => 21,
                'b3' => 30,
                'a4' => 20,
                'b4' => 48,
                'c' => 5
            ]
        );
        $this->assertEquals(
            5,
            $lst->lastOrNull()
        );
        $this->assertEquals(
            48,
            $lst->lastOrNull(function ($item) {
                return $item > 25;
            })
        );
        $this->assertEquals(
            20,
            $lst->lastOrNull(function (/** @noinspection PhpUnusedParameterInspection */ $item, $key) {
                return Str::startsWith($key,'a');
            })
        );
        $this->assertEquals(
            null,
            $lst->lastOrNull(function ($item) {
                return $item > 1500;
            })
        );
    }

    #endregion Enumerable

    #region Serializable

    public function testSerializable(){
        $lst = new IntDictionary([
            'a' => 0,
            'b' => 1,
            'c' => 2,
            'd' => 3,
            'e' => 4,
            'f' => 5,
            'g' => 6,
            'h' => 7,
        ]);
        $ser = serialize($lst);
        $this->assertNotEmpty($ser);
        $this->assertIsString($ser);
        /** @var  IntDictionary $unSer */
        $unSer = unserialize($ser);
        $this->assertTrue($unSer instanceof IntDictionary);
        $this->assertEquals($lst , $unSer);
        $this->assertFalse($lst === $unSer);
        $this->assertEquals(
            $lst->toArray(true),
            $unSer->toArray(true)
        );

    }

    #endregion Serializable

}
