<?php


namespace Gek\Collections\Tests;


use ArrayIterator;
use ArrayObject;
use Gek\Collections\Enumerable;
use Gek\Collections\FixedArrayList;
use Gek\Collections\IEnumerable;
use Gek\Collections\Iterators\FixedArrayListIterator;
use Gek\Collections\Tests\Fixtures\MockRefObj;
use Gek\Infrastructure\Exceptions\ArgumentOutOfRangeException;
use Gek\Infrastructure\Str;
use Iterator;
use PHPUnit\Framework\TestCase;

/**
 * Class FixedArrayListTest
 * @package Gek\Collections\Tests
 */
class FixedArrayListTest extends TestCase
{

    public function testConstruct(): void
    {
        // kapasite
        $lst = new FixedArrayList(10);
        $this->assertEquals(
            10,
            $lst->getCapacity()
        );
        // array
        $arr = array(10, 15, 16, 17);
        $lst = new FixedArrayList($arr);
        $this->assertEquals(
            count($arr),
            $lst->getCapacity()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray()
        );

        //Enumerable
        $enmr = Enumerable::fromArray($arr);
        $lst = new FixedArrayList($enmr);
        $this->assertEquals(
            $enmr->count(),
            $lst->getCapacity()
        );
        $this->assertEquals(
            $enmr->toArray(),
            $lst->toArray()
        );

        // IteratorAggregate
        $arrObj = new ArrayObject($arr);
        $lst = new FixedArrayList($arrObj);
        $this->assertEquals(
            $arrObj->count(),
            $lst->getCapacity()
        );
        $this->assertEquals(
            $arr,
            $lst->toArray()
        );

        // Iterator
        $itr = new ArrayIterator($arr);
        $lst = new FixedArrayList($itr);
        $this->assertEquals(
            $itr->count(),
            $lst->getCapacity()
        );
        $this->assertEquals(
            iterator_to_array($itr),
            $lst->toArray()
        );

        // generator
        $generator = function () {
            for ($i = 0; $i < 5; $i++) {
                yield $i;
            }
        };
        $lst = new FixedArrayList($generator());
        $this->assertEquals(
            5,
            $lst->getCapacity()
        );
        $this->assertEquals(
            iterator_to_array($generator()),
            $lst->toArray()
        );

        // object
        $obj = (object)[
            'propA' => 'a',
            'propB' => 'b',
        ];
        $lst = new FixedArrayList($obj);
        $this->assertEquals(
            2,
            $lst->getCapacity()
        );
        $this->assertEquals(
            ['a', 'b'],
            $lst->toArray()
        );

        // class
        $cls = new MockRefObj(3, 5);
        $lst = new FixedArrayList($cls);
        $this->assertEquals(
            2,
            $lst->getCapacity()
        );
        $this->assertEquals(
            [3, 5],
            $lst->toArray()
        );


    }

    #region ArrayAccess

    public function testOffsetExists(): void
    {
        $lst = new FixedArrayList(range(0, 9));
        $this->assertTrue(
            $lst->offsetExists(5)
        );
        // php isset
        $this->assertTrue(
            isset($lst[5])
        );
        $this->assertFalse(
            $lst->offsetExists(20)
        );
        // php isset
        $this->assertFalse(
            isset($lst[20])
        );
    }

    public function testOffsetGet(): void
    {
        $lst = new FixedArrayList(range(0, 9));
        $this->assertEquals(
            5,
            $lst->offsetGet(5)
        );
        // php []
        $this->assertEquals(
            5,
            $lst[5]
        );
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst[20];
    }

    public function testOffsetSet(): void
    {
        $lst = new FixedArrayList(range(0, 9));
        $lst->offsetSet(5, 'beş');
        $this->assertEquals(
            'beş',
            $lst[5]
        );
        // php [] set
        $lst[6] = 'altı';
        $this->assertEquals(
            'altı',
            $lst[6]
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst[65] = 'invalid index (65) set';

    }

    public function testOffsetUnset(): void
    {
        $lst = new FixedArrayList(range(0, 9));

        $lst->offsetUnset(5);
        $this->assertEquals(
            9,
            $lst->count()
        );

        //php unset
        unset($lst[5]);
        $this->assertEquals(
            8,
            $lst->count()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        unset($lst[65]);

    }

    #endregion ArrayAccess

    public function testGetCapacity(): void
    {
        $lst = new FixedArrayList();
        $this->assertEquals(
            0,
            $lst->getCapacity()
        );
        $lst->add('item1');
        $this->assertEquals(
            4,
            $lst->getCapacity()
        );
        $lst->addRange([2, 3, 4, 5]);
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            8,
            $lst->getCapacity()
        );
    }

    #region IEnumerable

    public function testGetIterator(): void
    {
        $lst = new FixedArrayList();
        $this->assertInstanceOf(
            Iterator::class,
            $lst->getIterator()
        );
        $this->assertInstanceOf(
            FixedArrayListIterator::class,
            $lst->getIterator()
        );
    }

    public function testCount(): void
    {
        $lst = new FixedArrayList(range(0, 10));
        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            11,
            count($lst)
        );
    }

    #endregion IEnumerable

    #region ICollection

    public function testAdd(): void
    {
        $lst = new FixedArrayList();
        $this->assertEquals(
            0,
            $lst->count()
        );
        $lst->add('test');
        $this->assertEquals(
            1,
            $lst->count()
        );

        $this->assertEquals(
            'test',
            $lst[0]
        );

        $lst->add('test2');
        $this->assertEquals(
            2,
            $lst->count()
        );

        $this->assertEquals(
            'test2',
            $lst[1]
        );
        $this->assertEquals(
            [
                0 => 'test',
                1 => 'test2'
            ],
            $lst->toArray()
        );


    }

    public function testAddRange(): void
    {
        $lst = new FixedArrayList(range(0, 5));

        $this->assertEquals(
            6,
            $lst->count()
        );

        $lst->addRange(range(0, 5));
        $this->assertEquals(
            12,
            $lst->count()
        );

        $this->assertEquals(
            [
                0 => 0,
                1 => 1,
                2 => 2,
                3 => 3,
                4 => 4,
                5 => 5,
                6 => 0,
                7 => 1,
                8 => 2,
                9 => 3,
                10 => 4,
                11 => 5
            ],
            $lst->toArray()
        );

    }

    public function testClear(): void
    {
        $lst = new FixedArrayList(range(0, 10));

        $this->assertEquals(
            11,
            $lst->count()
        );
        $this->assertEquals(
            true,
            $lst->any()
        );

        $lst->clear();

        $this->assertEquals(
            0,
            $lst->count()
        );
        $this->assertEquals(
            false,
            $lst->any()
        );

    }

    public function testContains(): void
    {
        $lst = new FixedArrayList(10);

        $stdObj = (object)[
            'prop' => 'a'
        ];
        $lst->add($stdObj);

        $arr = array(
            'a', 'b', 'c', 'd'
        );
        $lst->add($arr);

        $vlInt = 5;
        $lst->add($vlInt);

        $vlFloat = 15.4564;
        $lst->add($vlFloat);

        $refObj = new MockRefObj(3, 5);
        $lst->add($refObj);

        $vlStr = 'str-value';
        $lst->add($vlStr);

        $this->assertEquals(
            true,
            $lst->contains($stdObj)
        );
        $stdObj2 = (object)[
            'prop' => 'a',
        ];
        $this->assertEquals(
            true,
            $lst->contains($stdObj2)
        );
        $stdObj->prop = 'b';
        $this->assertEquals(
            false,
            $lst->contains($stdObj2)
        );
        $arr2 = array(
            'a', 'b', 'c', 'd'
        );
        // var_dump($lst->toArray());
        $this->assertEquals(
            true,
            $lst->contains($arr)
        );
        $this->assertEquals(
            true,
            $lst->contains($arr2)
        );
        $arr[] = 'e';
        $this->assertEquals(
            false,
            $lst->contains($arr)
        );

        $this->assertEquals(
            true,
            $lst->contains($refObj)
        );
        $refObj->propA = 10;
        $this->assertEquals(
            true,
            $lst->contains($refObj)
        );
        $refObj2 = $refObj;
        $this->assertEquals(
            true,
            $lst->contains($refObj2)
        );

        $refObj2->propB = 25;
        $this->assertEquals(
            true,
            $lst->contains($refObj2)
        );

        $refObj2 = clone $refObj;
        $refObj2->propB = 40;
        $this->assertEquals(
            false,
            $lst->contains($refObj2)
        );
        $this->assertEquals(
            true,
            $lst->contains(15.4564)
        );
        $this->assertEquals(
            true,
            $lst->contains($vlInt)
        );

        $this->assertEquals(
            true,
            $lst->contains('str-value')
        );


    }

    public function testRemove(): void
    {
        $lst = new FixedArrayList(range(0, 5));
        $this->assertEquals(
            6,
            $lst->count()
        );
        $lst->remove(2);
        $this->assertEquals(
            5,
            $lst->count()
        );
        $this->assertEquals(
            3,
            $lst[2]
        );

        $lst->remove(0);
        $this->assertEquals(
            4,
            $lst->count()
        );
        $this->assertEquals(
            [
                0 => 1,
                1 => 3,
                2 => 4,
                3 => 5
            ],
            $lst->toArray()
        );
    }

    #endregion ICollection

    #region IList

    public function testIndexOf(): void
    {
        $lst = new FixedArrayList(range(0, 10));
        $this->assertEquals(
            5,
            $lst->indexOf(5)
        );
        $this->assertEquals(
            -1,
            $lst->indexOf(35)
        );
        $lst->remove(4);
        $this->assertEquals(
            4,
            $lst->indexOf(5)
        );
        $lst->remove(5);
        $this->assertEquals(
            -1,
            $lst->indexOf(5)
        );
    }

    public function testInsert(): void
    {
        $lst = new FixedArrayList(range(0, 10));
        $lst->insert(5, 'test');
        $this->assertEquals(
            'test',
            $lst[5]
        );
        $lst->insert(0, 'first');
        $this->assertEquals(
            'first',
            $lst[0]
        );

        $lst->insert($lst->count(), 'last');
        $this->assertEquals(
            'last',
            $lst[13]
        );
        $this->assertEquals(
            [
                0 => 'first',
                1 => 0,
                2 => 1,
                3 => 2,
                4 => 3,
                5 => 4,
                6 => 'test',
                7 => 5,
                8 => 6,
                9 => 7,
                10 => 8,
                11 => 9,
                12 => 10,
                13 => 'last'
            ],
            $lst->toArray()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->insert(52, 45645);
    }

    public function testInsertRange(): void
    {
        $lst = new FixedArrayList(range(0, 10));

        $lst->insertRange(5, ['test', 'test2']);

        $this->assertEquals(
            'test',
            $lst[5]
        );
        $this->assertEquals(
            'test2',
            $lst[6]
        );

        $lst->insertRange(0, ['first', 'first2']);
        $this->assertEquals(
            'first',
            $lst[0]
        );
        $this->assertEquals(
            'first2',
            $lst[1]
        );

        $lst->insertRange($lst->count(), ['last', 'last2']);
        $this->assertEquals(
            'last',
            $lst[15]
        );
        $this->assertEquals(
            'last2',
            $lst[16]
        );

        $this->assertEquals(
            [
                0 => 'first',
                1 => 'first2',
                2 => 0,
                3 => 1,
                4 => 2,
                5 => 3,
                6 => 4,
                7 => 'test',
                8 => 'test2',
                9 => 5,
                10 => 6,
                11 => 7,
                12 => 8,
                13 => 9,
                14 => 10,
                15 => 'last',
                16 => 'last2'
            ],
            $lst->toArray()
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->insertRange(245, ['invalid', 'index']);
    }

    public function testRemoveAt(): void
    {
        $lst = new FixedArrayList(range(0, 10));

        $lst->removeAt(5);

        $this->assertEquals(
            6,
            $lst[5]
        );

        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->removeAt($lst->count());

    }

    public function testRemoveRange(): void
    {
        $lst = new FixedArrayList(range(0, 10));
        $lst->removeRange(5, 2);
        $this->assertEquals(
            [
                0 => 0,
                1 => 1,
                2 => 2,
                3 => 3,
                4 => 4,
                5 => 7,
                6 => 8,
                7 => 9,
                8 => 10
            ],
            $lst->toArray()
        );
        $lst->removeRange(5, 4);
        $this->assertEquals(
            [
                0 => 0,
                1 => 1,
                2 => 2,
                3 => 3,
                4 => 4,
            ],
            $lst->toArray()
        );
        $lst = new FixedArrayList(range(0, 10));
        $this->expectException(ArgumentOutOfRangeException::class);
        $lst->removeRange(15, 1);

    }

    public function testRemoveAll(): void
    {
        $lst = new FixedArrayList(10);

        $lst->add('Kamil');
        $lst->add('Hasan');
        $lst->add('Selim');
        $lst->add('Mahmut');
        $lst->add('Sibel');
        $lst->add('Murat');
        $lst->add('Harun');
        $lst->add('Samet');
        $lst->add('Levent');
        $lst->add('Serdar');
        $lst->add('Mesut');

        $lst->removeAll(function ($item) {
            return Str::startsWith($item, 'S') || Str::startsWith($item, 'M');
        });

        $this->assertEquals(
            [
                0 => 'Kamil',
                1 => 'Hasan',
                2 => 'Harun',
                3 => 'Levent',
            ],
            $lst->toArray()
        );


    }



    #endregion IList

    #region Enumerable

    public function testWhere():void {
        $lst = new FixedArrayList([
            10,25,18,26,21,30,20,48,5
        ]);
        $t1 = $lst->where(function ($item){
            return $item > 20;
        })->toArray();
        $this->assertEquals(
            [25,26,21,30,48],
            $t1
        );
        $t2 = $lst->where(function ($item){
            return $item < 20;
        });
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $lst->add(8);
        $this->assertEquals(
            [10,18,5,8],
            $t2->toArray()
        );

    }

    public function testLimit():void {
        $lst = new FixedArrayList([
            10,25,18,26,21,30,20,48,5
        ]);
        $t1 = $lst->limit(2,2)->toArray();
        $this->assertEquals(
            [18,26],
            $t1
        );
        $t2 = $lst->limit(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t2
        );
        $this->assertInstanceOf(
            Enumerable::class,
            $t2
        );
        $lst->add(8);
        $this->assertEquals(
            [30,20,48,5,8],
            $t2->toArray()
        );

        $t3 = $lst->limit(50,10)->toArray();
        $this->assertTrue(empty($t3));
    }

    public function testSkip():void {
        $lst = new FixedArrayList([0,1,2,3,4,5,6,7,8,9,10]);
        $t1 = $lst->skip(0)->toArray();
        $this->assertEquals(
            [0,1,2,3,4,5,6,7,8,9,10],
            $t1
        );
        $t2 = $lst->skip(5)->toArray();
        $this->assertEquals(
            [5,6,7,8,9,10],
            $t2
        );

        $t3 = $lst->skip(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $lst->add(11);
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [5,6,7,8,9,10,11],
            $t3->toArray()
        );

    }

    public function testTake():void {
        $lst = new FixedArrayList([0,1,2,3,4,5,6,7,8,9,10]);
        $t1 = $lst->skip(0)->toArray();
        $this->assertEquals(
            [0,1,2,3,4,5,6,7,8,9,10],
            $t1
        );
        $t2 = $lst->skip(5)->toArray();
        $this->assertEquals(
            [5,6,7,8,9,10],
            $t2
        );

        $t3 = $lst->skip(5);
        $this->assertInstanceOf(
            IEnumerable::class,
            $t3
        );
        $lst->add(11);
        $this->assertInstanceOf(
            Enumerable::class,
            $t3
        );
        $this->assertEquals(
            [5,6,7,8,9,10,11],
            $t3->toArray()
        );

    }

    public function testSelect(): void
    {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->select(function ($item) {
            return $item * 2;
        })->toArray();
        $this->assertEquals(
            [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20],
            $t1
        );

        $lst = new FixedArrayList([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->select(function (MockRefObj $item) {
            return $item->propA;
        })->toArray();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5],
            $t2
        );
        $t3 = $lst->select(function (MockRefObj $item) {
            return $item->propB;
        })->toArray();
        $this->assertEquals(
            [5, 5, 5, 5, 5, 5],
            $t3
        );
    }

    public function testEnumerableCount():void {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            11,
            $lst->asEnumerable()->count()
        );
        $this->assertEquals(
            5,
            $lst->count(function ($item){
                return $item > 5;
            })
        );
    }

    public function testSum():void {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->sum();
        $this->assertEquals(
            55,
            $t1
        );

        $lst = new FixedArrayList([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->sum(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            15,
            $t2
        );
    }

    public function testAny():void {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        $this->assertTrue(
            $lst->any()
        );
        $this->assertTrue(
            $lst->any(function ($item){return $item > 2;})
        );
        $this->assertTrue(
            $lst->any(function ($item){return $item === 0;})
        );
        $this->assertFalse(
            $lst->any(function ($item){return $item > 15;})
        );
        $lst->clear();
        $this->assertFalse(
            $lst->any()
        );
    }

    public function testAll():void {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        $this->assertTrue(
            $lst->all(function ($item){return $item < 20;})
        );
        $this->assertTrue(
            $lst->all(function ($item){return is_int($item);})
        );
        $this->assertFalse(
            $lst->all(function ($item){return $item > 5;})
        );

        $lst->clear();
        $this->assertTrue(
            $lst->all(function ($item){return $item > 5;})
        );
    }

    public function testAverage():void {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->average();
        $this->assertEquals(
            5.0,
            $t1
        );

        $lst = new FixedArrayList([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->average(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            2.5,
            $t2
        );
    }

    public function testMin():void {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $t1 = $lst->min();
        $this->assertEquals(
            0,
            $t1
        );

        $lst = new FixedArrayList([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->min(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            0,
            $t2
        );
        $lst->removeAt(0);
        $lst->removeAt(0);
        $t3 = $lst->min(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            2,
            $t3
        );
    }

    public function testMax(): void
    {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5, 25, 6, 7, 8, 9, 10]);
        $t1 = $lst->max();
        $this->assertEquals(
            25,
            $t1
        );

        $lst = new FixedArrayList([
            new MockRefObj(0, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 5),
            new MockRefObj(25, 5),
            new MockRefObj(3, 5),
            new MockRefObj(4, 5),
            new MockRefObj(5, 5),
        ]);
        $t2 = $lst->max(function (MockRefObj $item) {
            return $item->propA;
        });
        $this->assertEquals(
            25,
            $t2
        );
    }

    public function testReverse(): void
    {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5]);
        $t1 = $lst->reverse();
        $this->assertEquals(
            [5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
    }

    public function testAsEnumarable(): void
    {
        $lst = new FixedArrayList([0, 2, 3, 4, 5]);
        $t1 = $lst->where(function (int $item) {
            return $item > 3;
        });

        $t2 = $lst->where(function (int $item) {
            return $item > 3;
        })->asEnumerable();

        $lst->add(8);

        $this->assertEquals(
            [4, 5, 8],
            $t1->toArray()
        );
        $this->assertEquals(
            [4, 5],
            $t2->toArray()
        );


    }

    public function testSort(): void
    {
        $lst = new FixedArrayList([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->sort();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t1->toArray()
        );
        $t2 = $lst->sort(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t2->toArray()
        );
    }

    public function testSortDesc(): void
    {
        $lst = new FixedArrayList([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->sortDesc();
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
        $t2 = $lst->sortDesc(function ($a, $b) {
            if ($a == $b)
                return 0;
            return $a > $b ? -1 : 1;
        });
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t2->toArray()
        );
    }

    public function testOrderBy(): void
    {
        $lst = new FixedArrayList([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->orderBy();
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
            $t1->toArray()
        );
        $lst = new FixedArrayList([
            new MockRefObj(0, 3),
            new MockRefObj(1, 8),
            new MockRefObj(2, 1),
            new MockRefObj(3, 5),
            new MockRefObj(4, 6),
            new MockRefObj(5, 2),
        ]);
        $t2 = $lst->orderBy(function (MockRefObj $item) {
            return $item->propB;
        });
        $this->assertEquals(
            [
                new MockRefObj(2, 1),
                new MockRefObj(5, 2),
                new MockRefObj(0, 3),
                new MockRefObj(3, 5),
                new MockRefObj(4, 6),
                new MockRefObj(1, 8),
            ],
            $t2->toArray()
        );

        $t3 = $lst->orderBy(function (MockRefObj $item) {
            return $item->propB + $item->propA;
        });
        $this->assertEquals(
            [
                new MockRefObj(0, 3), // 3
                new MockRefObj(2, 1), // 3
                new MockRefObj(5, 2), // 7
                new MockRefObj(3, 5), // 8
                new MockRefObj(1, 8), // 9
                new MockRefObj(4, 6), // 10

            ],
            $t3->toArray()
        );

    }

    public function testOrderByDesc(): void
    {
        $lst = new FixedArrayList([3, 8, 4, 1, 7, 0, 9, 5, 2, 6]);
        $t1 = $lst->orderByDesc();
        $this->assertEquals(
            [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
            $t1->toArray()
        );
        $lst = new FixedArrayList([
            new MockRefObj(0, 3),
            new MockRefObj(1, 8),
            new MockRefObj(2, 1),
            new MockRefObj(3, 5),
            new MockRefObj(4, 6),
            new MockRefObj(5, 2),
        ]);
        $t2 = $lst->orderByDesc(function (MockRefObj $item) {
            return $item->propB;
        });
        $this->assertEquals(
            [
                new MockRefObj(1, 8),
                new MockRefObj(4, 6),
                new MockRefObj(3, 5),
                new MockRefObj(0, 3),
                new MockRefObj(5, 2),
                new MockRefObj(2, 1),
            ],
            $t2->toArray()
        );

        $t3 = $lst->orderByDesc(function (MockRefObj $item) {
            return $item->propB + $item->propA;
        });
        $this->assertEquals(
            [
                new MockRefObj(4, 6), // 10
                new MockRefObj(1, 8), // 9
                new MockRefObj(3, 5), // 8
                new MockRefObj(5, 2), // 7
                new MockRefObj(2, 1), // 3
                new MockRefObj(0, 3), // 3
            ],
            $t3->toArray()
        );
    }

    public function testConcat(): void
    {
        $lst = new FixedArrayList([0, 1, 2, 3, 4, 5]);
        $t1 = $lst->concat([6, 7, 8, 9, 10]);
        $this->assertEquals(
            [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
            $t1->toArray()
        );
    }

    public function testDistinct(): void
    {
        $lst = new FixedArrayList([0, 1, 0, 1, 1, 0]);
        $t1 = $lst->distinct();
        $this->assertEquals(
            [0, 1],
            $t1->toArray()
        );

        $lst = new FixedArrayList([
            new MockRefObj(1, 5),
            new MockRefObj(1, 5),
            new MockRefObj(2, 4),
            new MockRefObj(3, 6),
            new MockRefObj(2, 4),

        ]);
        $t2 = $lst->distinct();
        $this->assertEquals(
            [
                new MockRefObj(1, 5),
                new MockRefObj(2, 4),
                new MockRefObj(3, 6),
            ],
            $t2->toArray()
        );

        $a = new MockRefObj(1, 5);
        $b = new MockRefObj(1, 5);
        $c = new MockRefObj(1, 5);

        $lst = new FixedArrayList([
            new MockRefObj(1, 5),
            $c,
            new MockRefObj(1, 5),
            $a,
            $b,
        ]);
        $t3 = $lst->distinct();
        $this->assertEquals(
            [
                new MockRefObj(1, 5),
            ],
            $t3->toArray()
        );

        $lst = new FixedArrayList([
            $a,
            $c,
            $b,
            $a,
            $b,
            $c,
        ]);
        $t4 = $lst->distinct(function ($a, $b){
            return $a === $b;
        });
        $this->assertEquals(
            [
                $a,
                $c,
                $b,
            ],
            $t4->toArray()
        );


    }

    public function testExcept(): void
    {
        $lst  = new FixedArrayList([0,1,2,3,4,5,6,7,8,9,10]);
        $t1 = $lst->except([10,9,5,6,7,8]);
        $this->assertEquals(
            [0,1,2,3,4],
            $t1->toArray()
        );
    }

    public function testFirstOrNUll():void {
        $lst = new FixedArrayList([0,1,2,3,4,5,6,7,8,9,10]);
        $this->assertEquals(
            0,
            $lst->firstOrNull()
        );
        $this->assertEquals(
            6,
            $lst->firstOrNull(function ($item){return $item > 5;})
        );
        $this->assertEquals(
            null,
            $lst->firstOrNull(function ($item){return $item > 15;})
        );
    }

    public function testAggregate(): void
    {
        $lst = new FixedArrayList([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
        $this->assertEquals(
            3628800,
            $lst->aggregate(function (&$total, $item) {
                $total *= $item;
            })
        );
        $this->assertEquals(
            181440,
            $lst->aggregate(
                function (&$total, $item) {
                    $total *= $item;
                },
                null,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
        $this->assertEquals(
            3.25,
            $lst->aggregate(
                function (&$total, $item) {
                    $total += $item;
                },
                10,
                function ($total, Enumerable $th) {
                    return ($total / $th->count()) / 2;
                })
        );
    }

    public function testUnion():void {
        $lst = new FixedArrayList([-3,'1',1,2,3,3,4,4,5]);
        $t1 = $lst->union([3,8,4,8]);

        $this->assertEquals(
            [-3,1,2,3,4,5,8],
            $t1->toArray()
        );
        $t2 = $lst->union([3,8,4,8],function ($a, $b){return $a === $b;});
        $this->assertEquals(
            [-3,'1',1,2,3,4,5,8],
            $t2->toArray()
        );
    }

    public function testIntersect():void {
        $lst = new FixedArrayList([1,3,4,5]);
        $t1 = $lst->intersect([3,8,4]);

        $this->assertEquals(
            [3,4],
            $t1->toArray()
        );
        $t2 = $lst->intersect([]);

        $this->assertEquals(
            [],
            $t2->toArray()
        );

        $a = new MockRefObj(1,5);
        $b = new MockRefObj(1,5);
        $c = new MockRefObj(1,5);
        $d = new MockRefObj(1,5);
        $e = new MockRefObj(1,5);
        $f = new MockRefObj(1,5);

        $lst2 = new FixedArrayList([$a,$b,$c,$d]);
        $t3 = $lst2->intersect([$f,$d,$e,$b], function ($first, $second){return $first === $second;});

        $this->assertEquals(
            [$b,$d],
            $t3->toArray()
        );

    }

    public function testZip():void {
        $lst = new FixedArrayList([1,2,3,4,5]);
        $t1 = $lst->zip( new FixedArrayList(['a','b','c','d','e']), function ($f, $s){return $f . '-' . $s;});

        $this->assertEquals(
            ['1-a','2-b','3-c','4-d','5-e'],
            $t1->toArray()
        );

        $t2 = $lst->zip(new FixedArrayList(['a','b','c']), function ($f, $s){return $f . '-' . $s;});

        $this->assertEquals(
            ['1-a','2-b','3-c'],
            $t2->toArray()
        );

        $lst2 = new FixedArrayList([1,2]);
        $t3 = $lst2->zip(new FixedArrayList(['a','b','c','d','e']), function ($f, $s){return $f . '-' . $s;});

        $this->assertEquals(
            ['1-a','2-b'],
            $t3->toArray()
        );
    }

    public function testLastOrNull():void {
        $enmr = new FixedArrayList([0,1,2,3,4,5,6,7,8,9,10]);
        $t1 = $enmr->lastOrNull();
        $this->assertEquals(
            10,
            $t1
        );
        $t2 = $enmr->lastOrNull(function ($item){return $item < 5;});
        $this->assertEquals(
            4,
            $t2
        );
    }

    #endregion Enumerable

    #region serializable

    public function testSerializable(){
        $lst = new FixedArrayList(range(0, 9));
        $ser = serialize($lst);
        $this->assertNotEmpty($ser);
        $this->assertIsString($ser);
        /** @var  FixedArrayList $unSer */
        $unSer = unserialize($ser);
        $this->assertTrue($unSer instanceof FixedArrayList);
        $this->assertEquals($lst , $unSer);
        $this->assertFalse($lst === $unSer);
        $this->assertEquals(
            $lst->toArray(),
            $unSer->toArray()
        );
    }

    #endregion serializable


}
